"""
Contains Image panes including renderers for PNG, SVG, GIF and JPG
file types.
"""
from __future__ import absolute_import, division, unicode_literals

import base64

from io import BytesIO
from six import string_types

import param

from .markup import escape, DivPaneBase
from ..util import isfile, isurl


class ImageBase(DivPaneBase):
    """
    Encodes an image as base64 and wraps it in a Bokeh Div model.
    This is an abstract base class that needs the image type
    to be specified and specific code for determining the image shape.

    The imgtype determines the filetype, extension, and MIME type for
    this image. Each image type (png,jpg,gif) has a base class that
    supports anything with a `_repr_X_` method (where X is `png`,
    `gif`, etc.), a local file with the given file extension, or a
    HTTP(S) url with the given extension.  Subclasses of each type can
    provide their own way of obtaining or generating a PNG.
    """

    alt_text = param.String(default=None, doc="""
        alt text to add to the image tag. The alt text is shown when a
        user cannot load or display the image.""")

    link_url = param.String(default=None, doc="""
        A link URL to make the image clickable and link to some other
        website.""")

    embed = param.Boolean(default=True, doc="""
        Whether to embed the image as base64.""")

    imgtype = 'None'

    _rerender_params = ['alt_text', 'link_url', 'embed', 'object', 'style']

    _target_transforms = {'object': """'<img src="' + value + '"></img>'"""}

    __abstract = True

    @classmethod
    def applies(cls, obj):
        imgtype = cls.imgtype
        if hasattr(obj, '_repr_{}_'.format(imgtype)):
            return True
        if isinstance(obj, string_types):
            if isfile(obj) and obj.endswith('.'+imgtype):
                return True
            if isurl(obj, [cls.imgtype]):
                return True
        if hasattr(obj, 'read'):  # Check for file like object
            return True
        return False

    def _type_error(self, object):
        if isinstance(object, string_types):
            raise ValueError("%s pane cannot parse string that is not a filename "
                             "or URL." % type(self).__name__)
        super(ImageBase, self)._type_error(object)

    def _img(self):
        if hasattr(self.object, '_repr_{}_'.format(self.imgtype)):
            return getattr(self.object, '_repr_' + self.imgtype + '_')()
        if isinstance(self.object, string_types):
            if isfile(self.object):
                with open(self.object, 'rb') as f:
                    return f.read()
        if hasattr(self.object, 'read'):
            return self.object.read()
        if isurl(self.object, [self.imgtype]):
            import requests
            r = requests.request(url=self.object, method='GET')
            return r.content

    def _imgshape(self, data):
        """Calculate and return image width,height"""
        raise NotImplementedError

    def _get_properties(self):
        p = super(ImageBase, self)._get_properties()
        if self.object is None:
            return dict(p, text='<img></img>')
        data = self._img()
        if not isinstance(data, bytes):
            data = base64.b64decode(data)
        width, height = self._imgshape(data)
        if self.width is not None:
            if self.height is None:
                height = int((self.width/width)*height)
            else:
                height = self.height
            width = self.width
        elif self.height is not None:
            width = int((self.height/height)*width)
            height = self.height
        if not self.embed:
            src = self.object
        else:
            b64 = base64.b64encode(data).decode("utf-8")
            src = "data:image/"+self.imgtype+";base64,{b64}".format(b64=b64)

        smode = self.sizing_mode
        if smode in ['fixed', None]:
            w, h = '%spx' % width, '%spx' % height
        elif smode == 'stretch_both':
            w, h = '100%', '100%'
        elif smode == 'stretch_width':
            w, h = '%spx' % width, '100%'
        elif smode == 'stretch_height':
            w, h = '100%', '%spx' % height
        elif smode == 'scale_height':
            w, h = 'auto', '100%'
        else:
            w, h = '100%', 'auto'

        html = '<img src="{src}" width="{width}" height="{height}" alt="{alt}"></img>'.format(
            src=src, width=w, height=h, alt=self.alt_text or '')

        if self.link_url:
            html = '<a href="{url}" target="_blank">{html}</a>'.format(
                url=self.link_url, html=html)

        return dict(p, width=width, height=height, text=escape(html))


class PNG(ImageBase):

    imgtype = 'png'

    @classmethod
    def _imgshape(cls, data):
        import struct
        w, h = struct.unpack('>LL', data[16:24])
        return int(w), int(h)


class GIF(ImageBase):

    imgtype = 'gif'

    @classmethod
    def _imgshape(cls, data):
        import struct
        w, h = struct.unpack("<HH", data[6:10])
        return int(w), int(h)


class JPG(ImageBase):

    imgtype = 'jpg'

    @classmethod
    def _imgshape(cls, data):
        import struct
        b = BytesIO(data)
        b.read(2)
        c = b.read(1)
        while (c and ord(c) != 0xDA):
            while (ord(c) != 0xFF): c = b.read(1)
            while (ord(c) == 0xFF): c = b.read(1)
            if (ord(c) >= 0xC0 and ord(c) <= 0xC3):
                b.read(3)
                h, w = struct.unpack(">HH", b.read(4))
                break
            else:
                b.read(int(struct.unpack(">H", b.read(2))[0])-2)
            c = b.read(1)
        return int(w), int(h)


class SVG(ImageBase):

    encode = param.Boolean(default=False, doc="""
        Whether to enable base64 encoding of the SVG, base64 encoded
        SVGs do not support links.""")

    imgtype = 'svg'

    _rerender_params = ImageBase._rerender_params + ['encode']

    @classmethod
    def applies(cls, obj):
        return (super(SVG, cls).applies(obj) or
                (isinstance(obj, string_types) and obj.lstrip().startswith('<svg')))

    def _type_error(self, object):
        if isinstance(object, string_types):
            raise ValueError("%s pane cannot parse string that is not a filename, "
                             "URL or a SVG XML contents." % type(self).__name__)
        super(SVG, self)._type_error(object)

    def _img(self):
        if (isinstance(self.object, string_types) and
            self.object.lstrip().startswith('<svg')):
            return self.object
        return super(SVG, self)._img()

    def _imgshape(self, data):
        return (self.width, self.height)

    def _get_properties(self):
        p = super(ImageBase, self)._get_properties()
        if self.object is None:
            return dict(p, text='<img></img>')
        data = self._img()
        width, height = self._imgshape(data)
        if not isinstance(data, bytes):
            data = data.encode('utf-8')

        if self.encode:
            b64 = base64.b64encode(data).decode("utf-8")
            src = "data:image/svg+xml;base64,{b64}".format(b64=b64)
            html = "<img src='{src}' width={width} height={height}></img>".format(
                src=src, width=width, height=height
            )
        else:
            html = data.decode("utf-8")
        return dict(p, width=width, height=height, text=escape(html))
