from abc import abstractmethod
from decimal import Decimal
from typing import Any, Optional, Union

from loguru import logger
from pandas import DataFrame
from tompy.datetime import date_add

from otlpy.base.inventory import InventoryMap
from otlpy.base.market import PQ, LimitOrderBook
from otlpy.base.net import AsyncHttpClient
from otlpy.base.order import (
    ORDER_TYPE,
    Buy,
    Cancel,
    Order,
    OrderAPI2,
    Replace,
    Sell,
)
from otlpy.base.timer import Timer
from otlpy.kis.common import Common


def str_order_type(order_type: ORDER_TYPE, /) -> str:
    if order_type == ORDER_TYPE.LIMIT:
        s = "01"
    elif order_type == ORDER_TYPE.MARKET:
        s = "02"
    else:
        assert False
    return s


class DomesticFutureOption(OrderAPI2):
    def __init__(self, common: Common, client: AsyncHttpClient, /) -> None:
        self.__common = common
        self.__client = client

    @property
    def _common(self, /) -> Common:
        return self.__common

    @property
    def _client(self, /) -> AsyncHttpClient:
        return self.__client

    async def token(self, /) -> None:
        await self._common.token(self._client)

    async def _new_order(
        self,
        order: Union[Buy, Sell],
        /,
        *,
        omsg: str,
    ) -> None:
        common = self._common
        client = self._client
        otype = str_order_type(order.otype)
        tr_id = "TTTO1101U"
        url_path = "/uapi/domestic-futureoption/v1/trading/order"
        data = {
            "ORD_PRCS_DVSN_CD": "02",
            "CANO": common.account_cano_domestic_futureoption,
            "ACNT_PRDT_CD": common.account_prdt_domestic_futureoption,
            "SLL_BUY_DVSN_CD": omsg,
            "SHTN_PDNO": order.ticker,
            "ORD_QTY": f"{order.qty:f}",
            "UNIT_PRICE": f"{order.price:f}",
            "NMPR_TYPE_CD": otype,
            "KRX_NMPR_CNDT_CD": "0",
            "ORD_DVSN_CD": otype,
        }
        headers = {
            **common.headers4(),
            "tr_id": tr_id,
            "hashkey": await common.hash(client, data),
        }
        err, rheaders, rdata = await client.post(url_path, headers, data)
        if err:
            return
        if rdata["rt_cd"] != "0":
            logger.error(
                f"\n, {url_path}"
                f"\n, {headers}"
                f"\n, {data}"
                f"\n, {rheaders}"
                f"\n, {rdata}"
            )
            return
        order.acknowledgment(rdata, rdata["output"]["ODNO"], order.qty)

    async def _cancel_or_replace_order(
        self,
        order: Union[Cancel, Replace],
        /,
        *,
        omsg: str,
        ack_qty: Decimal,
    ) -> None:
        common = self._common
        client = self._client
        rdata1 = order.origin.rdata["output"]
        otype = str_order_type(order.otype)
        tr_id = "TTTO1103U"
        url_path = "/uapi/domestic-futureoption/v1/trading/order-rvsecncl"
        data = {
            "ORD_PRCS_DVSN_CD": "02",
            "CANO": common.account_cano_domestic_futureoption,
            "ACNT_PRDT_CD": common.account_prdt_domestic_futureoption,
            "RVSE_CNCL_DVSN_CD": omsg,
            "ORGN_ODNO": rdata1["ODNO"],
            "ORD_QTY": "",
            "UNIT_PRICE": f"{order.price:f}",
            "NMPR_TYPE_CD": otype,
            "KRX_NMPR_CNDT_CD": "0",
            "RMN_QTY_YN": "Y",
            "ORD_DVSN_CD": otype,
        }
        headers = {
            **common.headers4(),
            "tr_id": tr_id,
            "hashkey": await common.hash(client, data),
        }
        err, rheaders, rdata = await client.post(url_path, headers, data)
        if err:
            return
        if rdata["rt_cd"] != "0":
            logger.error(
                f"\n, {url_path}"
                f"\n, {headers}"
                f"\n, {data}"
                f"\n, {rheaders}"
                f"\n, {rdata}"
            )
            return
        order.acknowledgment(rdata, rdata["output"]["ODNO"], ack_qty)

    async def _all_orders(
        self,
        yyyymmdd: str,
        /,
        *,
        tr_cont: str,
        ctx_area_fk200: str,
        ctx_area_nk200: str,
        ticker: str = "",
    ) -> tuple[bool, Any]:
        common = self._common
        tr_id = "TTTO5201R"
        url_path = "/uapi/domestic-futureoption/v1/trading/inquire-ccnl"
        data = {
            "CANO": common.account_cano_domestic_futureoption,
            "ACNT_PRDT_CD": common.account_prdt_domestic_futureoption,
            "STRT_ORD_DT": yyyymmdd,
            "END_ORD_DT": yyyymmdd,
            "SLL_BUY_DVSN_CD": "00",
            "CCLD_NCCS_DVSN": "00",
            "SORT_SQN": "DS",
            "STRT_ODNO": "",
            "PDNO": ticker,
            "MKET_ID_CD": "",
            "CTX_AREA_FK200": ctx_area_fk200,
            "CTX_AREA_NK200": ctx_area_nk200,
        }
        headers = {
            **common.headers4(),
            "tr_id": tr_id,
            "tr_cont": tr_cont,
        }
        err, rheaders, rdata = await self._client.get(url_path, headers, data)
        if err:
            return False, None
        if rdata["rt_cd"] != "0":
            logger.error(
                f"\n, {url_path}"
                f"\n, {headers}"
                f"\n, {data}"
                f"\n, {rheaders}"
                f"\n, {rdata}"
            )
            return False, None
        if rheaders["tr_cont"] == "F" or rheaders["tr_cont"] == "M":
            return True, rdata
        if rheaders["tr_cont"] == "D" or rheaders["tr_cont"] == "E":
            return False, rdata
        assert False

    async def _limitorderbook(self, ticker: str, /) -> Any:
        meta = self.meta()
        common = self._common
        tr_id = "FHMIF10010000"
        url_path = (
            "/uapi/domestic-futureoption/v1/quotations/inquire-asking-price"
        )
        data = {
            "FID_COND_MRKT_DIV_CODE": meta["fid_cond_mrkt_div_code"],
            "FID_INPUT_ISCD": ticker,
        }
        headers = {
            **common.headers4(),
            "tr_id": tr_id,
        }
        err, rheaders, rdata = await self._client.get(url_path, headers, data)
        if err:
            return None
        if rdata["rt_cd"] != "0":
            logger.error(
                f"\n, {url_path}"
                f"\n, {headers}"
                f"\n, {data}"
                f"\n, {rheaders}"
                f"\n, {rdata}"
            )
            return None
        return rdata

    async def _candlestick(
        self,
        ticker: str,
        /,
        yyyymmdd1: str,
        yyyymmdd2: str,
        *,
        interval: str = "D",
    ) -> Any:
        meta = self.meta()
        common = self._common
        tr_id = "FHKIF03020100"
        url_path = "/uapi/domestic-futureoption/v1/quotations/inquire-daily-fuopchartprice"
        data = {
            "FID_COND_MRKT_DIV_CODE": meta["fid_cond_mrkt_div_code"],
            "FID_INPUT_ISCD": ticker,
            "FID_INPUT_DATE_1": yyyymmdd1,
            "FID_INPUT_DATE_2": yyyymmdd2,
            "FID_PERIOD_DIV_CODE": interval,
        }
        headers = {
            **common.headers4(),
            "tr_id": tr_id,
        }
        err, rheaders, rdata = await self._client.get(url_path, headers, data)
        if err:
            return None
        if rdata["rt_cd"] != "0":
            logger.error(
                f"\n, {url_path}"
                f"\n, {headers}"
                f"\n, {data}"
                f"\n, {rheaders}"
                f"\n, {rdata}"
            )
            return None
        return rdata

    async def buy(
        self,
        order_type: ORDER_TYPE,
        ticker: str,
        /,
        qty: Decimal,
        price: Decimal,
        *,
        comment: str = "",
    ) -> Buy:
        order = Buy(order_type, ticker, qty, price, comment=comment)
        await self._new_order(order, omsg="02")
        return order

    async def sell(
        self,
        order_type: ORDER_TYPE,
        ticker: str,
        /,
        qty: Decimal,
        price: Decimal,
        *,
        comment: str = "",
    ) -> Sell:
        order = Sell(order_type, ticker, qty, price, comment=comment)
        await self._new_order(order, omsg="01")
        return order

    async def cancel(
        self,
        origin: Order,
        /,
        *,
        comment: str = "",
    ) -> Cancel:
        order = Cancel(origin, ORDER_TYPE.LIMIT, comment=comment)
        await self._cancel_or_replace_order(
            order, omsg="02", ack_qty=Decimal()
        )
        return order

    async def replace(
        self,
        origin: Order,
        order_type: ORDER_TYPE,
        /,
        price: Decimal,
        *,
        comment: str = "",
    ) -> Replace:
        order = Replace(origin, order_type, price, comment=comment)
        await self._cancel_or_replace_order(
            order, omsg="01", ack_qty=order.qty
        )
        return order

    async def all_orders(
        self,
        inv_map: InventoryMap,
        first_order: Optional[Order],
        lob: Optional[LimitOrderBook],
        t: Timer,
        /,
    ) -> None:
        if first_order is None or lob is None:
            return

        mid = (lob.bid[0].price + lob.ask[0].price) / 2
        yyyymmdd = t.now_date().strftime("%Y%m%d")
        tr_cont = ""
        ctx_area_fk200 = ""
        ctx_area_nk200 = ""
        while True:
            is_cont, rdata = await self._all_orders(
                yyyymmdd,
                tr_cont=tr_cont,
                ctx_area_fk200=ctx_area_fk200,
                ctx_area_nk200=ctx_area_nk200,
                ticker=first_order.ticker,
            )
            tr_cont = ""
            ctx_area_fk200 = ""
            ctx_area_nk200 = ""
            if rdata is None:
                return

            raw_orders = rdata["output1"]
            for raw in raw_orders:
                uid = raw["odno"]
                total_filled = Decimal(raw["tot_ccld_qty"])
                total_filled_price = Decimal(raw["avg_idx"])
                total_opened = Decimal(raw["qty"])
                inv_map.filled_total(
                    uid,
                    total_filled,
                    total_filled_price,
                    total_opened,
                    mid=mid,
                )
                if uid == first_order.uid:
                    return

            if not is_cont:
                return
            tr_cont = "N"
            ctx_area_fk200 = rdata["ctx_area_fk200"]
            ctx_area_nk200 = rdata["ctx_area_nk200"]

    async def limitorderbook(self, ticker: str, /) -> Optional[LimitOrderBook]:
        raw = await self._limitorderbook(ticker)
        if raw is None:
            return None
        out1 = raw["output2"]
        bidp1 = out1.get("futs_bidp1")
        askp1 = out1.get("futs_askp1")
        if bidp1 and askp1:
            return LimitOrderBook(
                out1["aspr_acpt_hour"],
                ticker,
                [
                    PQ(Decimal(bidp1), Decimal(out1.get("bidp_rsqn1", "1"))),
                ],
                [
                    PQ(Decimal(askp1), Decimal(out1.get("askp_rsqn1", "1"))),
                ],
            )
        return None

    async def candlestick(
        self, ticker: str, /, t: Timer, *, interval: str = "D"
    ) -> Optional[DataFrame]:
        today = t.now_date()
        raw = await self._candlestick(
            ticker,
            date_add(today, -140).strftime("%Y%m%d"),
            today.strftime("%Y%m%d"),
            interval=interval,
        )
        if not raw:
            return None
        out2 = raw["output2"]
        return DataFrame(
            data={
                "Open": [float(x["futs_oprc"]) for x in out2],
                "High": [float(x["futs_hgpr"]) for x in out2],
                "Low": [float(x["futs_lwpr"]) for x in out2],
                "Close": [float(x["futs_prpr"]) for x in out2],
            },
            index=[int(x["stck_bsop_date"]) for x in out2],
        ).sort_index()

    @abstractmethod
    def meta(self, /) -> dict[str, str]:
        raise NotImplementedError


class IndexFuture(DomesticFutureOption):
    def meta(self, /) -> dict[str, str]:
        return {
            "fid_cond_mrkt_div_code": "F",
        }


class IndexOption(DomesticFutureOption):
    def meta(self, /) -> dict[str, str]:
        return {
            "fid_cond_mrkt_div_code": "O",
        }


class StockFuture(DomesticFutureOption):
    def meta(self, /) -> dict[str, str]:
        return {
            "fid_cond_mrkt_div_code": "JF",
        }


class StockOption(DomesticFutureOption):
    def meta(self, /) -> dict[str, str]:
        return {
            "fid_cond_mrkt_div_code": "JO",
        }
