from asyncio import sleep as _sleep
from typing import Any, Optional

from httpx import AsyncClient, Headers, Response, codes
from loguru import logger
from websockets.legacy.client import Connect, WebSocketClientProtocol


class AsyncWebSocketClient:
    def __init__(
        self,
        uri: str,
        /,
        *,
        ping_interval: Optional[float] = None,
        delay_send: float = 0,
        delay_recv: float = 0,
        debug: bool = False,
    ) -> None:
        self.__con = Connect(uri, ping_interval=ping_interval)
        self.__ws: Optional[WebSocketClientProtocol] = None
        self.__delay_send = delay_send
        self.__delay_recv = delay_recv
        self.__debug = debug

    @property
    def _con(self, /) -> Connect:
        return self.__con

    @property
    def _ws(self, /) -> WebSocketClientProtocol:
        assert self.__ws is not None
        return self.__ws

    def _set_ws(self, ws: WebSocketClientProtocol, /) -> None:
        self.__ws = ws

    @property
    def delay_send(self, /) -> float:
        return self.__delay_send

    @property
    def delay_recv(self, /) -> float:
        return self.__delay_recv

    @property
    def debug(self, /) -> bool:
        return self.__debug

    async def __aenter__(self, /) -> Any:
        self._set_ws(await self._con.__aenter__())
        return self

    async def __aexit__(self, *args: Any, **kwargs: Any) -> None:
        await self._con.__aexit__(*args, **kwargs)

    async def _sleep_send(self, /) -> None:
        if self.delay_send > 0:
            await _sleep(self.delay_send)

    async def _sleep_recv(self, /) -> None:
        if self.delay_recv > 0:
            await _sleep(self.delay_recv)

    async def send(self, msg: Any, /) -> None:
        try:
            await self._sleep_send()
            await self._ws.send(msg)
            if self.debug:
                logger.debug(f"\n, SEND, {msg}")
        except Exception as e:
            logger.error(f"\n, {e}")
            return

    async def recv(self, /) -> Any:
        try:
            await self._sleep_recv()
            msg = await self._ws.recv()
            if self.debug:
                logger.debug(f"\n, RECV, {str(msg)}")
            return msg
        except Exception as e:
            logger.error(f"\n, {e}")
            return None


class AsyncHttpClient:
    def __init__(
        self,
        base_url: str,
        /,
        *,
        timeout: Optional[float] = None,
        delay: float = 0,
        debug: bool = False,
    ) -> None:
        self.__client = AsyncClient(base_url=base_url, timeout=timeout)
        self.__delay = delay
        self.__debug = debug

    @property
    def _client(self, /) -> AsyncClient:
        return self.__client

    @property
    def delay(self, /) -> float:
        return self.__delay

    @property
    def debug(self, /) -> bool:
        return self.__debug

    async def __aenter__(self, /) -> Any:
        await self._client.__aenter__()
        return self

    async def __aexit__(self, *args: Any, **kwargs: Any) -> None:
        await self._client.__aexit__(*args, **kwargs)

    async def _sleep(self, /) -> None:
        if self.delay > 0:
            await _sleep(self.delay)

    def _response_processing(
        self,
        method: str,
        r: Response,
        url_path: str,
        headers: dict[str, Any],
        data: dict[str, Any],
        /,
    ) -> tuple[bool, Headers, Any]:
        scode = r.status_code
        rheaders = r.headers
        rdata = r.json()
        err = scode != codes.OK
        if err:
            logger.error(
                f"\n, {r}, {method}, {url_path}"
                f"\n, {headers}"
                f"\n, {data}"
                f"\n, {rheaders}"
                f"\n, {rdata}"
            )
            return err, rheaders, None
        if self.debug:
            logger.debug(
                f"\n, {r}, {method}, {url_path}"
                f"\n, {headers}"
                f"\n, {data}"
                f"\n, {rheaders}"
                f"\n, {rdata}"
            )
        return err, rheaders, rdata

    async def get(
        self,
        url_path: str,
        headers: dict[str, Any],
        data: dict[str, Any],
        /,
    ) -> tuple[bool, Headers, Any]:
        try:
            await self._sleep()
            r = await self._client.get(url_path, headers=headers, params=data)
            return self._response_processing("GET", r, url_path, headers, data)
        except Exception as e:
            logger.error(f"\n, {e}")
            return True, Headers(), None

    async def post(
        self,
        url_path: str,
        headers: dict[str, Any],
        data: dict[str, Any],
        /,
    ) -> tuple[bool, Headers, Any]:
        try:
            await self._sleep()
            r = await self._client.post(url_path, headers=headers, json=data)
            return self._response_processing(
                "POST", r, url_path, headers, data
            )
        except Exception as e:
            logger.error(f"\n, {e}")
            return True, Headers(), None

    async def post_params(
        self,
        url_path: str,
        headers: dict[str, Any],
        data: dict[str, Any],
        /,
    ) -> tuple[bool, Headers, Any]:
        try:
            await self._sleep()
            r = await self._client.post(url_path, headers=headers, params=data)
            return self._response_processing(
                "POST", r, url_path, headers, data
            )
        except Exception as e:
            logger.error(f"\n, {e}")
            return True, Headers(), None

    async def delete(
        self,
        url_path: str,
        headers: dict[str, Any],
        data: dict[str, Any],
        /,
    ) -> tuple[bool, Headers, Any]:
        try:
            await self._sleep()
            r = await self._client.delete(
                url_path, headers=headers, params=data
            )
            return self._response_processing(
                "DELETE", r, url_path, headers, data
            )
        except Exception as e:
            logger.error(f"\n, {e}")
            return True, Headers(), None
