from abc import ABC, abstractmethod
from datetime import datetime, timedelta
from decimal import Decimal
from enum import Enum, auto
from typing import Any, Optional

from loguru import logger
from tompy.datetime import now as _now


class ORDER_SIDE(Enum):
    BUY = auto()
    SELL = auto()


class ORDER_TYPE(Enum):
    LIMIT = auto()
    MARKET = auto()


class Order:
    def __init__(
        self,
        oside: ORDER_SIDE,
        otype: ORDER_TYPE,
        ticker: str,
        /,
        qty: Decimal,
        price: Decimal,
        *,
        comment: str = "",
    ) -> None:
        self.__oside = oside
        self.__otype = otype
        self.__ticker = ticker
        self.__qty = qty
        self.__price = price
        self.__created_at = _now()
        self.__rdata: dict[str, Any] = {}
        self.__uid = ""
        self.__filled = Decimal()
        self.__filled_price = Decimal()
        self.__opened = Decimal()
        self.__comment = comment

    @property
    def oside(self, /) -> ORDER_SIDE:
        return self.__oside

    @property
    def otype(self, /) -> ORDER_TYPE:
        return self.__otype

    @property
    def ticker(self, /) -> str:
        return self.__ticker

    @property
    def qty(self, /) -> Decimal:
        return self.__qty

    @property
    def price(self, /) -> Decimal:
        return self.__price

    @property
    def created_at(self, /) -> datetime:
        return self.__created_at

    def created_timedelta(self, /) -> timedelta:
        return _now() - self.created_at

    @property
    def rdata(self, /) -> dict[str, Any]:
        return self.__rdata

    def _set_rdata(self, rdata: dict[str, Any], /) -> None:
        self.__rdata = rdata

    @property
    def uid(self, /) -> str:
        return self.__uid

    def _set_uid(self, uid: str, /) -> None:
        self.__uid = uid

    @property
    def filled(self, /) -> Decimal:
        return self.__filled

    def _set_filled(self, filled: Decimal, /) -> None:
        self.__filled = filled

    @property
    def filled_price(self, /) -> Decimal:
        return self.__filled_price

    def _set_filled_price(self, filled_price: Decimal, /) -> None:
        self.__filled_price = filled_price

    @property
    def opened(self, /) -> Decimal:
        return self.__opened

    def _set_opened(self, opened: Decimal, /) -> None:
        self.__opened = opened

    @property
    def comment(self, /) -> str:
        return self.__comment

    def acknowledgment(
        self,
        rdata: dict[str, Any],
        uid: str,
        opened: Decimal,
        /,
    ) -> None:
        self._set_rdata(rdata)
        self._set_uid(uid)
        self._set_opened(opened)

    def filled_event(
        self,
        filled: Decimal,
        filled_price: Decimal,
        /,
        *,
        opened: Optional[Decimal] = None,
    ) -> None:
        if opened is None:
            opened = filled
        assert filled <= self.opened and opened <= self.opened
        if filled > 0:
            total_filled = self.filled + filled
            self._set_filled_price(
                (self.filled * self.filled_price + filled * filled_price)
                / total_filled
            )
            self._set_filled(total_filled)
        if opened > 0:
            self._set_opened(self.opened - opened)

    def filled_total(
        self,
        total_filled: Decimal,
        total_filled_price: Decimal,
        total_opened: Decimal,
        /,
    ) -> tuple[Decimal, Decimal, Decimal]:
        assert total_filled >= self.filled and self.opened >= total_opened
        filled = total_filled - self.filled
        if filled > 0:
            filled_price = (
                total_filled * total_filled_price
                - self.filled * self.filled_price
            ) / filled
        else:
            filled_price = Decimal()
        opened = self.opened - total_opened
        self.filled_event(filled, filled_price, opened=opened)
        return filled, filled_price, opened

    def logger_info(self, /, msg: str) -> None:
        logger.info(
            f"{msg}"
            f"\n, ORD, {self.oside.name}, {self.ticker}, {self.uid}"
            f", {self.qty:f}, {self.price:f}"
            f", {self.filled:f}, {self.filled_price:f}"
            f", {self.opened:f}"
            f", {self.comment}"
        )


class Buy(Order):
    def __init__(
        self,
        otype: ORDER_TYPE,
        ticker: str,
        /,
        qty: Decimal,
        price: Decimal,
        *,
        comment: str = "",
    ) -> None:
        super().__init__(
            ORDER_SIDE.BUY,
            otype,
            ticker,
            qty,
            price,
            comment=comment,
        )


class Sell(Order):
    def __init__(
        self,
        otype: ORDER_TYPE,
        ticker: str,
        /,
        qty: Decimal,
        price: Decimal,
        *,
        comment: str = "",
    ) -> None:
        super().__init__(
            ORDER_SIDE.SELL,
            otype,
            ticker,
            qty,
            price,
            comment=comment,
        )


class Cancel(Order):
    def __init__(
        self,
        origin: Order,
        otype: ORDER_TYPE,
        /,
        *,
        comment: str = "",
    ) -> None:
        super().__init__(
            origin.oside,
            otype,
            origin.ticker,
            origin.qty,
            Decimal(),
            comment=comment,
        )
        self.__origin = origin

    @property
    def origin(self, /) -> Order:
        return self.__origin


class Replace(Order):
    def __init__(
        self,
        origin: Order,
        otype: ORDER_TYPE,
        /,
        price: Decimal,
        *,
        comment: str = "",
    ) -> None:
        super().__init__(
            origin.oside,
            otype,
            origin.ticker,
            origin.qty,
            price,
            comment=comment,
        )
        self.__origin = origin

    @property
    def origin(self, /) -> Order:
        return self.__origin


class OrderAPI1(ABC):
    @abstractmethod
    async def buy(
        self,
        order_type: ORDER_TYPE,
        ticker: str,
        /,
        qty: Decimal,
        price: Decimal,
        *,
        comment: str = "",
    ) -> Buy:
        raise NotImplementedError

    @abstractmethod
    async def sell(
        self,
        order_type: ORDER_TYPE,
        ticker: str,
        /,
        qty: Decimal,
        price: Decimal,
        *,
        comment: str = "",
    ) -> Sell:
        raise NotImplementedError

    @abstractmethod
    async def cancel(
        self,
        origin: Order,
        /,
        *,
        comment: str = "",
    ) -> Cancel:
        raise NotImplementedError

    async def buy_market(
        self,
        ticker: str,
        /,
        qty: Decimal,
        *,
        comment: str = "",
    ) -> Buy:
        return await self.buy(
            ORDER_TYPE.MARKET, ticker, qty, Decimal(), comment=comment
        )

    async def buy_limit(
        self,
        ticker: str,
        /,
        qty: Decimal,
        price: Decimal,
        *,
        comment: str = "",
    ) -> Buy:
        return await self.buy(
            ORDER_TYPE.LIMIT, ticker, qty, price, comment=comment
        )

    async def sell_market(
        self,
        ticker: str,
        /,
        qty: Decimal,
        *,
        comment: str = "",
    ) -> Sell:
        return await self.sell(
            ORDER_TYPE.MARKET, ticker, qty, Decimal(), comment=comment
        )

    async def sell_limit(
        self,
        ticker: str,
        /,
        qty: Decimal,
        price: Decimal,
        *,
        comment: str = "",
    ) -> Sell:
        return await self.sell(
            ORDER_TYPE.LIMIT, ticker, qty, price, comment=comment
        )


class OrderAPI2(OrderAPI1):
    @abstractmethod
    async def replace(
        self,
        origin: Order,
        order_type: ORDER_TYPE,
        /,
        price: Decimal,
        *,
        comment: str = "",
    ) -> Replace:
        raise NotImplementedError

    async def replace_market(
        self,
        origin: Order,
        /,
        *,
        comment: str = "",
    ) -> Replace:
        return await self.replace(
            origin, ORDER_TYPE.MARKET, Decimal(), comment=comment
        )

    async def replace_limit(
        self,
        origin: Order,
        /,
        price: Decimal,
        *,
        comment: str = "",
    ) -> Replace:
        return await self.replace(
            origin, ORDER_TYPE.LIMIT, price, comment=comment
        )
