from abc import abstractmethod
from decimal import Decimal
from typing import Any, Optional, Union

from loguru import logger
from pandas import DataFrame

from otlpy.base.inventory import InventoryMap
from otlpy.base.market import PQN, LimitOrderBook
from otlpy.base.net import AsyncHttpClient
from otlpy.base.order import (
    ORDER_TYPE,
    Buy,
    Cancel,
    Order,
    OrderAPI2,
    Replace,
    Sell,
)
from otlpy.base.timer import Timer
from otlpy.kis.common import Common


class OverseasStock(OrderAPI2):
    def __init__(self, common: Common, client: AsyncHttpClient, /) -> None:
        self.__common = common
        self.__client = client

    @property
    def _common(self, /) -> Common:
        return self.__common

    @property
    def _client(self, /) -> AsyncHttpClient:
        return self.__client

    async def token(self, /) -> None:
        await self._common.token(self._client)

    async def _new_order(
        self,
        order: Union[Buy, Sell],
        /,
        *,
        tr_id: str,
        excg_cd: str,
        ord_dvsn: str,
    ) -> None:
        common = self._common
        client = self._client
        url_path = "/uapi/overseas-stock/v1/trading/order"
        data = {
            "CANO": common.account_cano_overseas_stock,
            "ACNT_PRDT_CD": common.account_prdt_overseas_stock,
            "OVRS_EXCG_CD": excg_cd,
            "PDNO": order.ticker,
            "ORD_QTY": f"{order.qty:f}",
            "OVRS_ORD_UNPR": f"{order.price:f}",
            "ORD_SVR_DVSN_CD": "0",
        }
        if ord_dvsn:
            data["ORD_DVSN"] = ord_dvsn
        headers = {
            **common.headers4(),
            "tr_id": tr_id,
            "hashkey": await common.hash(client, data),
        }
        err, rheaders, rdata = await client.post(url_path, headers, data)
        if err:
            return
        if rdata["rt_cd"] != "0":
            logger.error(
                f"\n, {url_path}"
                f"\n, {headers}"
                f"\n, {data}"
                f"\n, {rheaders}"
                f"\n, {rdata}"
            )
            return
        order.acknowledgment(rdata, rdata["output"]["ODNO"], order.qty)

    async def _cancel_or_replace_order(
        self,
        order: Union[Cancel, Replace],
        /,
        *,
        tr_id: str,
        excg_cd: str,
        omsg: str,
        ack_qty: Decimal,
    ) -> None:
        common = self._common
        client = self._client
        rdata1 = order.origin.rdata["output"]
        url_path = "/uapi/overseas-stock/v1/trading/order-rvsecncl"
        data = {
            "CANO": common.account_cano_overseas_stock,
            "ACNT_PRDT_CD": common.account_prdt_overseas_stock,
            "OVRS_EXCG_CD": excg_cd,
            "PDNO": order.ticker,
            "ORGN_ODNO": rdata1["ODNO"],
            "RVSE_CNCL_DVSN_CD": omsg,
            "ORD_QTY": f"{order.qty:f}",
            "OVRS_ORD_UNPR": f"{order.price:f}",
            "ORD_SVR_DVSN_CD": "0",
        }
        headers = {
            **common.headers4(),
            "tr_id": tr_id,
            "hashkey": await common.hash(client, data),
        }
        err, rheaders, rdata = await client.post(url_path, headers, data)
        if err:
            return
        if rdata["rt_cd"] != "0":
            logger.error(
                f"\n, {url_path}"
                f"\n, {headers}"
                f"\n, {data}"
                f"\n, {rheaders}"
                f"\n, {rdata}"
            )
            return
        order.acknowledgment(rdata, rdata["output"]["ODNO"], ack_qty)

    async def _all_orders(
        self,
        yyyymmdd: str,
        /,
        *,
        tr_cont: str,
        ctx_area_fk200: str,
        ctx_area_nk200: str,
        ticker: str = "",
    ) -> tuple[bool, Any]:
        meta = self.meta()
        common = self._common
        url_path = "/uapi/overseas-stock/v1/trading/inquire-ccnl"
        data = {
            "CANO": common.account_cano_overseas_stock,
            "ACNT_PRDT_CD": common.account_prdt_overseas_stock,
            "PDNO": ticker,
            "ORD_STRT_DT": yyyymmdd,
            "ORD_END_DT": yyyymmdd,
            "SLL_BUY_DVSN": "00",
            "CCLD_NCCS_DVSN": "00",
            "OVRS_EXCG_CD": meta["excg_cd"],
            "SORT_SQN": "AS",
            "ORD_DT": "",
            "ORD_GNO_BRNO": "",
            "ODNO": "",
            "CTX_AREA_NK200": ctx_area_nk200,
            "CTX_AREA_FK200": ctx_area_fk200,
        }
        headers = {
            **common.headers4(),
            "tr_id": meta["tr_all_orders"],
            "tr_cont": tr_cont,
        }
        err, rheaders, rdata = await self._client.get(url_path, headers, data)
        if err:
            return False, None
        if rdata["rt_cd"] != "0":
            logger.error(
                f"\n, {url_path}"
                f"\n, {headers}"
                f"\n, {data}"
                f"\n, {rheaders}"
                f"\n, {rdata}"
            )
            return False, None
        if rheaders["tr_cont"] == "F" or rheaders["tr_cont"] == "M":
            return True, rdata
        if rheaders["tr_cont"] == "D" or rheaders["tr_cont"] == "E":
            return False, rdata
        assert False

    async def _price(self, ticker: str, /) -> Any:
        meta = self.meta()
        common = self._common
        tr_id = "HHDFS00000300"
        url_path = "/uapi/overseas-price/v1/quotations/price"
        data = {
            "AUTH": "",
            "EXCD": meta["excd"],
            "SYMB": ticker,
        }
        headers = {
            **common.headers4(),
            "tr_id": tr_id,
        }
        err, rheaders, rdata = await self._client.get(url_path, headers, data)
        if err:
            return None
        if rdata["rt_cd"] != "0":
            logger.error(
                f"\n, {url_path}"
                f"\n, {headers}"
                f"\n, {data}"
                f"\n, {rheaders}"
                f"\n, {rdata}"
            )
            return None
        return rdata

    async def _candlestick(
        self,
        ticker: str,
        /,
        *,
        interval: str = "0",
    ) -> Any:
        meta = self.meta()
        common = self._common
        tr_id = "HHDFS76240000"
        url_path = "/uapi/overseas-price/v1/quotations/dailyprice"
        data = {
            "AUTH": "",
            "EXCD": meta["excd"],
            "SYMB": ticker,
            "GUBN": interval,
            "BYMD": "",
            "MODP": "1",
        }
        headers = {
            **common.headers4(),
            "tr_id": tr_id,
        }
        err, rheaders, rdata = await self._client.get(url_path, headers, data)
        if err:
            return None
        if rdata["rt_cd"] != "0":
            logger.error(
                f"\n, {url_path}"
                f"\n, {headers}"
                f"\n, {data}"
                f"\n, {rheaders}"
                f"\n, {rdata}"
            )
            return None
        return rdata

    async def buy(
        self,
        order_type: ORDER_TYPE,
        ticker: str,
        /,
        qty: Decimal,
        price: Decimal,
        *,
        comment: str = "",
    ) -> Buy:
        meta = self.meta()
        order = Buy(order_type, ticker, qty, price, comment=comment)
        await self._new_order(
            order,
            tr_id=meta["tr_buy"],
            excg_cd=meta["excg_cd"],
            ord_dvsn=meta[order.otype.name],
        )
        return order

    async def sell(
        self,
        order_type: ORDER_TYPE,
        ticker: str,
        /,
        qty: Decimal,
        price: Decimal,
        *,
        comment: str = "",
    ) -> Sell:
        meta = self.meta()
        order = Sell(order_type, ticker, qty, price, comment=comment)
        await self._new_order(
            order,
            tr_id=meta["tr_sell"],
            excg_cd=meta["excg_cd"],
            ord_dvsn=meta[order.otype.name],
        )
        return order

    async def cancel(
        self,
        origin: Order,
        /,
        *,
        comment: str = "",
    ) -> Cancel:
        meta = self.meta()
        order = Cancel(origin, ORDER_TYPE.LIMIT, comment=comment)
        await self._cancel_or_replace_order(
            order,
            tr_id=meta["tr_cancel"],
            excg_cd=meta["excg_cd"],
            omsg=meta["omsg_cancel"],
            ack_qty=Decimal(),
        )
        return order

    async def replace(
        self,
        origin: Order,
        order_type: ORDER_TYPE,
        /,
        price: Decimal,
        *,
        comment: str = "",
    ) -> Replace:
        meta = self.meta()
        order = Replace(origin, order_type, price, comment=comment)
        await self._cancel_or_replace_order(
            order,
            tr_id=meta["tr_replace"],
            excg_cd=meta["excg_cd"],
            omsg=meta["omsg_replace"],
            ack_qty=order.qty,
        )
        return order

    async def all_orders(
        self,
        inv_map: InventoryMap,
        first_order: Optional[Order],
        lob: Optional[LimitOrderBook],
        t: Timer,
        /,
    ) -> None:
        if first_order is None or lob is None:
            return

        mid = (lob.bid[0].price + lob.ask[0].price) / 2
        yyyymmdd = t.now_date().strftime("%Y%m%d")
        tr_cont = ""
        ctx_area_fk200 = ""
        ctx_area_nk200 = ""
        while True:
            is_cont, rdata = await self._all_orders(
                yyyymmdd,
                tr_cont=tr_cont,
                ctx_area_fk200=ctx_area_fk200,
                ctx_area_nk200=ctx_area_nk200,
                ticker=first_order.ticker,
            )
            tr_cont = ""
            ctx_area_fk200 = ""
            ctx_area_nk200 = ""
            if rdata is None:
                return

            raw_orders = rdata["output"]
            for raw in raw_orders:
                uid = raw["odno"]
                total_filled = Decimal(raw["ft_ccld_qty"])
                if total_filled > 0:
                    total_filled_price = (
                        Decimal(raw["ft_ccld_amt3"]) / total_filled
                    )
                else:
                    total_filled_price = Decimal()
                total_opened = Decimal(raw["nccs_qty"])
                inv_map.filled_total(
                    uid,
                    total_filled,
                    total_filled_price,
                    total_opened,
                    mid=mid,
                )
                if uid == first_order.uid:
                    return

            if not is_cont:
                return
            tr_cont = "N"
            ctx_area_fk200 = rdata["ctx_area_fk200"]
            ctx_area_nk200 = rdata["ctx_area_nk200"]

    async def limitorderbook(self, ticker: str, /) -> Optional[LimitOrderBook]:
        raw = await self._price(ticker)
        if raw is None:
            return None
        out1 = raw["output"]
        last = out1.get("last")
        if last:
            return LimitOrderBook(
                out1["base"],
                ticker,
                [
                    PQN(Decimal(last), Decimal("1")),
                ],
                [
                    PQN(Decimal(last), Decimal("1")),
                ],
            )
        return None

    async def candlestick(
        self, ticker: str, /, *, interval: str = "D"
    ) -> Optional[DataFrame]:
        raw = await self._candlestick(
            ticker,
            interval=interval,
        )
        if not raw:
            return None
        out2 = raw["output2"]
        return DataFrame(
            data={
                "Open": [float(x["open"]) for x in out2],
                "High": [float(x["high"]) for x in out2],
                "Low": [float(x["low"]) for x in out2],
                "Close": [float(x["clos"]) for x in out2],
            },
            index=[int(x["xymd"]) for x in out2],
        ).sort_index()

    @abstractmethod
    def meta(self, /) -> dict[str, str]:
        raise NotImplementedError


class US(OverseasStock):
    def _meta(self, /) -> dict[str, str]:
        return {
            "LIMIT": "00",
            "omsg_cancel": "02",
            "omsg_replace": "01",
            "tr_all_orders": "JTTT3001R",
            "tr_buy": "JTTT1002U",
            "tr_cancel": "JTTT1004U",
            "tr_replace": "JTTT1004U",
            "tr_sell": "JTTT1006U",
        }


class NASD(US):
    def meta(self, /) -> dict[str, str]:
        return {
            **self._meta(),
            "excd": "NAS",
            "excg_cd": "NASD",
        }


class NYSE(US):
    def meta(self, /) -> dict[str, str]:
        return {
            **self._meta(),
            "excd": "NYS",
            "excg_cd": "NYSE",
        }


class AMEX(US):
    def meta(self, /) -> dict[str, str]:
        return {
            **self._meta(),
            "excd": "AMS",
            "excg_cd": "AMEX",
        }
