from decimal import Decimal
from typing import Any, Optional, Union

from pandas import DataFrame

from otlpy.base.inventory import InventoryMap
from otlpy.base.market import PQN, LimitOrderBook
from otlpy.base.net import AsyncHttpClient
from otlpy.base.order import ORDER_TYPE, Buy, Cancel, Order, OrderAPI1, Sell
from otlpy.binance.common import Common


class Spot(OrderAPI1):
    def __init__(self, common: Common, client: AsyncHttpClient, /) -> None:
        self.__common = common
        self.__client = client

    @property
    def _common(self, /) -> Common:
        return self.__common

    @property
    def _client(self, /) -> AsyncHttpClient:
        return self.__client

    async def _new_order(self, order: Union[Buy, Sell], /) -> None:
        common = self._common
        url_path = "/api/v3/order"
        data = common.signature(
            {
                "symbol": order.ticker,
                "side": order.oside.name,
                "type": order.otype.name,
                "timeInForce": "GTC",
                "quantity": f"{order.qty:f}",
                "price": f"{order.price:f}",
            }
        )
        headers = common.headers2()
        err, _rheaders, rdata = await self._client.post_params(
            url_path, headers, data
        )
        if err:
            return
        order.acknowledgment(rdata, f'{rdata["orderId"]}', order.qty)

    async def _cancel_order(self, order: Cancel, /) -> None:
        common = self._common
        url_path = "/api/v3/order"
        data = common.signature(
            {
                "symbol": order.ticker,
                "orderId": order.origin.rdata["orderId"],
            }
        )
        headers = common.headers2()
        err, _rheaders, rdata = await self._client.delete(
            url_path, headers, data
        )
        if err:
            return
        order.acknowledgment(rdata, f'C{rdata["orderId"]}', Decimal())

    async def _all_orders(self, ticker: str, orderid: int, /) -> Any:
        common = self._common
        url_path = "/api/v3/allOrders"
        data = common.signature(
            {
                "symbol": ticker,
                "orderId": orderid,
            }
        )
        headers = common.headers2()
        err, _rheaders, rdata = await self._client.get(url_path, headers, data)
        if err:
            return None
        return rdata

    async def _limitorderbook(self, ticker: str, /) -> Any:
        common = self._common
        url_path = "/api/v3/depth"
        data = {
            "symbol": ticker,
        }
        headers = common.headers1()
        err, _rheaders, rdata = await self._client.get(url_path, headers, data)
        if err:
            return None
        return rdata

    async def _candlestick(
        self,
        ticker: str,
        /,
        *,
        interval: str = "1d",
    ) -> Any:
        common = self._common
        url_path = "/api/v3/klines"
        data = {
            "symbol": ticker,
            "interval": interval,
        }
        headers = common.headers1()
        err, _rheaders, rdata = await self._client.get(url_path, headers, data)
        if err:
            return None
        return rdata

    async def buy(
        self,
        order_type: ORDER_TYPE,
        ticker: str,
        /,
        qty: Decimal,
        price: Decimal,
        *,
        comment: str = "",
    ) -> Buy:
        order = Buy(order_type, ticker, qty, price, comment=comment)
        await self._new_order(order)
        return order

    async def sell(
        self,
        order_type: ORDER_TYPE,
        ticker: str,
        /,
        qty: Decimal,
        price: Decimal,
        *,
        comment: str = "",
    ) -> Sell:
        order = Sell(order_type, ticker, qty, price, comment=comment)
        await self._new_order(order)
        return order

    async def cancel(
        self,
        origin: Order,
        /,
        *,
        comment: str = "",
    ) -> Cancel:
        order = Cancel(origin, ORDER_TYPE.LIMIT, comment=comment)
        await self._cancel_order(order)
        return order

    async def all_orders(
        self,
        inv_map: InventoryMap,
        first_order: Optional[Order],
        lob: Optional[LimitOrderBook],
        /,
    ) -> None:
        if first_order is None or lob is None:
            return

        raw_orders = await self._all_orders(
            first_order.ticker, first_order.rdata["orderId"]
        )
        for raw in raw_orders:
            uid = f'{raw["orderId"]}'
            total_filled = Decimal(raw["executedQty"])
            total_filled_price = Decimal(raw["price"])
            status = raw["status"]
            if status in (
                "NEW",
                "PARTIALLY_FILLED",
                "PENDING_CANCEL",
            ):
                total_opened = Decimal(raw["origQty"]) - total_filled
            elif status in (
                "FILLED",
                "CANCELED",
                "REJECTED",
                "EXPIRED",
            ):
                total_opened = Decimal()
            else:
                assert False
            mid = (lob.bid[0].price + lob.ask[0].price) / 2
            inv_map.filled_total(
                uid, total_filled, total_filled_price, total_opened, mid=mid
            )

    async def limitorderbook(self, ticker: str, /) -> Optional[LimitOrderBook]:
        raw = await self._limitorderbook(ticker)
        if not raw:
            return None
        bids = raw.get("bids")
        asks = raw.get("asks")
        if bids and asks:
            return LimitOrderBook(
                f'{raw["lastUpdateId"]}',
                ticker,
                [
                    PQN(Decimal(bids[0][0]), Decimal(bids[0][1])),
                ],
                [
                    PQN(Decimal(asks[0][0]), Decimal(asks[0][1])),
                ],
            )
        return None

    async def candlestick(
        self, ticker: str, /, *, interval: str = "1d"
    ) -> Optional[DataFrame]:
        raw = await self._candlestick(ticker, interval=interval)
        if not raw:
            return None
        return DataFrame(
            data={
                "Open": [float(x[1]) for x in raw],
                "High": [float(x[2]) for x in raw],
                "Low": [float(x[3]) for x in raw],
                "Close": [float(x[4]) for x in raw],
            },
            index=[x[0] for x in raw],
        ).sort_index()
