from decimal import Decimal
from typing import Any, Optional, Union

from loguru import logger
from pandas import DataFrame
from tompy.datetime import date_add

from otlpy.base.inventory import InventoryMap
from otlpy.base.market import PQN, LimitOrderBook
from otlpy.base.net import AsyncHttpClient, AsyncWebSocketClient
from otlpy.base.order import (
    ORDER_TYPE,
    Buy,
    Cancel,
    Order,
    OrderAPI2,
    Replace,
    Sell,
)
from otlpy.base.timer import BaseTimer
from otlpy.kis.common import Common


def str_order_type(order_type: ORDER_TYPE, /) -> str:
    if order_type == ORDER_TYPE.LIMIT:
        s = "00"
    elif order_type == ORDER_TYPE.MARKET:
        s = "01"
    else:
        assert False
    return s


class DomesticStock(OrderAPI2):
    def __init__(self, common: Common, client: AsyncHttpClient, /) -> None:
        self.__common = common
        self.__client = client

    @property
    def _common(self, /) -> Common:
        return self.__common

    @property
    def _client(self, /) -> AsyncHttpClient:
        return self.__client

    async def token(self, /) -> None:
        await self._common.token(self._client)

    async def _new_order(
        self,
        order: Union[Buy, Sell],
        /,
        *,
        tr_id: str,
    ) -> None:
        common = self._common
        client = self._client
        url_path = "/uapi/domestic-stock/v1/trading/order-cash"
        data = {
            "CANO": common.account_cano_domestic_stock,
            "ACNT_PRDT_CD": common.account_prdt_domestic_stock,
            "PDNO": order.ticker,
            "ORD_DVSN": str_order_type(order.otype),
            "ORD_QTY": f"{order.qty:f}",
            "ORD_UNPR": f"{order.price:f}",
        }
        headers = {
            **common.headers4(),
            "tr_id": tr_id,
            "hashkey": await common.hash(client, data),
        }
        err, rheaders, rdata = await client.post(url_path, headers, data)
        if err:
            return
        if rdata["rt_cd"] != "0":
            logger.error(
                f"\n, {url_path}"
                f"\n, {headers}"
                f"\n, {data}"
                f"\n, {rheaders}"
                f"\n, {rdata}"
            )
            return
        order.acknowledgment(rdata, rdata["output"]["ODNO"], order.qty)

    async def _cancel_or_replace_order(
        self,
        order: Union[Cancel, Replace],
        /,
        *,
        omsg: str,
        ack_qty: Decimal,
    ) -> None:
        common = self._common
        client = self._client
        rdata1 = order.origin.rdata["output"]
        tr_id = "TTTC0803U"
        url_path = "/uapi/domestic-stock/v1/trading/order-rvsecncl"
        data = {
            "CANO": common.account_cano_domestic_stock,
            "ACNT_PRDT_CD": common.account_prdt_domestic_stock,
            "KRX_FWDG_ORD_ORGNO": rdata1["KRX_FWDG_ORD_ORGNO"],
            "ORGN_ODNO": rdata1["ODNO"],
            "ORD_DVSN": str_order_type(order.otype),
            "RVSE_CNCL_DVSN_CD": omsg,
            "ORD_QTY": f"{order.qty:f}",
            "ORD_UNPR": f"{order.price:f}",
            "QTY_ALL_ORD_YN": "Y",
        }
        headers = {
            **common.headers4(),
            "tr_id": tr_id,
            "hashkey": await common.hash(client, data),
        }
        err, rheaders, rdata = await client.post(url_path, headers, data)
        if err:
            return
        if rdata["rt_cd"] != "0":
            logger.error(
                f"\n, {url_path}"
                f"\n, {headers}"
                f"\n, {data}"
                f"\n, {rheaders}"
                f"\n, {rdata}"
            )
            return
        order.acknowledgment(rdata, rdata["output"]["ODNO"], ack_qty)

    async def _all_orders(
        self,
        yyyymmdd: str,
        /,
        *,
        tr_cont: str,
        ctx_area_fk100: str,
        ctx_area_nk100: str,
        ticker: str = "",
    ) -> tuple[bool, Any]:
        common = self._common
        tr_id = "TTTC8001R"
        url_path = "/uapi/domestic-stock/v1/trading/inquire-daily-ccld"
        data = {
            "CANO": common.account_cano_domestic_stock,
            "ACNT_PRDT_CD": common.account_prdt_domestic_stock,
            "INQR_STRT_DT": yyyymmdd,
            "INQR_END_DT": yyyymmdd,
            "SLL_BUY_DVSN_CD": "00",
            "INQR_DVSN": "00",
            "PDNO": ticker,
            "CCLD_DVSN": "00",
            "ORD_GNO_BRNO": "",
            "ODNO": "",
            "INQR_DVSN_3": "",
            "INQR_DVSN_1": "",
            "CTX_AREA_FK100": ctx_area_fk100,
            "CTX_AREA_NK100": ctx_area_nk100,
        }
        headers = {
            **common.headers4(),
            "tr_id": tr_id,
            "tr_cont": tr_cont,
        }
        err, rheaders, rdata = await self._client.get(url_path, headers, data)
        if err:
            return False, None
        if rdata["rt_cd"] != "0":
            logger.error(
                f"\n, {url_path}"
                f"\n, {headers}"
                f"\n, {data}"
                f"\n, {rheaders}"
                f"\n, {rdata}"
            )
            return False, None
        if rheaders["tr_cont"] == "F" or rheaders["tr_cont"] == "M":
            return True, rdata
        if rheaders["tr_cont"] == "D" or rheaders["tr_cont"] == "E":
            return False, rdata
        assert False

    async def _limitorderbook(self, ticker: str, /) -> Any:
        common = self._common
        tr_id = "FHKST01010200"
        url_path = (
            "/uapi/domestic-stock/v1/quotations/inquire-asking-price-exp-ccn"
        )
        data = {
            "FID_COND_MRKT_DIV_CODE": "J",
            "FID_INPUT_ISCD": ticker,
        }
        headers = {
            **common.headers4(),
            "tr_id": tr_id,
        }
        err, rheaders, rdata = await self._client.get(url_path, headers, data)
        if err:
            return None
        if rdata["rt_cd"] != "0":
            logger.error(
                f"\n, {url_path}"
                f"\n, {headers}"
                f"\n, {data}"
                f"\n, {rheaders}"
                f"\n, {rdata}"
            )
            return None
        return rdata

    async def _candlestick(
        self,
        ticker: str,
        /,
        yyyymmdd1: str,
        yyyymmdd2: str,
        *,
        interval: str = "D",
    ) -> Any:
        common = self._common
        tr_id = "FHKST03010100"
        url_path = (
            "/uapi/domestic-stock/v1/quotations/inquire-daily-itemchartprice"
        )
        data = {
            "FID_COND_MRKT_DIV_CODE": "J",
            "FID_INPUT_ISCD": ticker,
            "FID_INPUT_DATE_1": yyyymmdd1,
            "FID_INPUT_DATE_2": yyyymmdd2,
            "FID_PERIOD_DIV_CODE": interval,
            "FID_ORG_ADJ_PRC": "0",
        }
        headers = {
            **common.headers4(),
            "tr_id": tr_id,
        }
        err, rheaders, rdata = await self._client.get(url_path, headers, data)
        if err:
            return None
        if rdata["rt_cd"] != "0":
            logger.error(
                f"\n, {url_path}"
                f"\n, {headers}"
                f"\n, {data}"
                f"\n, {rheaders}"
                f"\n, {rdata}"
            )
            return None
        return rdata

    async def buy(
        self,
        order_type: ORDER_TYPE,
        ticker: str,
        /,
        qty: Decimal,
        price: Decimal,
        *,
        comment: str = "",
    ) -> Buy:
        order = Buy(order_type, ticker, qty, price, comment=comment)
        await self._new_order(order, tr_id="TTTC0802U")
        return order

    async def sell(
        self,
        order_type: ORDER_TYPE,
        ticker: str,
        /,
        qty: Decimal,
        price: Decimal,
        *,
        comment: str = "",
    ) -> Sell:
        order = Sell(order_type, ticker, qty, price, comment=comment)
        await self._new_order(order, tr_id="TTTC0801U")
        return order

    async def cancel(
        self,
        origin: Order,
        /,
        *,
        comment: str = "",
    ) -> Cancel:
        order = Cancel(origin, ORDER_TYPE.LIMIT, comment=comment)
        await self._cancel_or_replace_order(
            order, omsg="02", ack_qty=Decimal()
        )
        return order

    async def replace(
        self,
        origin: Order,
        order_type: ORDER_TYPE,
        /,
        price: Decimal,
        *,
        comment: str = "",
    ) -> Replace:
        order = Replace(origin, order_type, price, comment=comment)
        await self._cancel_or_replace_order(
            order, omsg="01", ack_qty=order.qty
        )
        return order

    async def all_orders(
        self,
        inv_map: InventoryMap,
        first_order: Optional[Order],
        lob: Optional[LimitOrderBook],
        t: BaseTimer,
        /,
    ) -> None:
        if first_order is None or lob is None:
            return

        mid = (lob.bid[0].price + lob.ask[0].price) / 2
        yyyymmdd = t.date().strftime("%Y%m%d")
        tr_cont = ""
        ctx_area_fk100 = ""
        ctx_area_nk100 = ""
        while True:
            is_cont, rdata = await self._all_orders(
                yyyymmdd,
                tr_cont=tr_cont,
                ctx_area_fk100=ctx_area_fk100,
                ctx_area_nk100=ctx_area_nk100,
                ticker=first_order.ticker,
            )
            tr_cont = ""
            ctx_area_fk100 = ""
            ctx_area_nk100 = ""
            if rdata is None:
                return

            raw_orders = rdata["output1"]
            for raw in raw_orders:
                uid = raw["odno"]
                total_filled = Decimal(raw["tot_ccld_qty"])
                if total_filled > 0:
                    total_filled_price = (
                        Decimal(raw["tot_ccld_amt"]) / total_filled
                    )
                else:
                    total_filled_price = Decimal()
                total_opened = Decimal(raw["rmn_qty"])
                inv_map.filled_total(
                    uid,
                    total_filled,
                    total_filled_price,
                    total_opened,
                    mid=mid,
                )
                if uid == first_order.uid:
                    return

            if not is_cont:
                return
            tr_cont = "N"
            ctx_area_fk100 = rdata["ctx_area_fk100"]
            ctx_area_nk100 = rdata["ctx_area_nk100"]

    async def limitorderbook(self, ticker: str, /) -> Optional[LimitOrderBook]:
        raw = await self._limitorderbook(ticker)
        if raw is None:
            return None
        out1 = raw["output1"]
        bidp1 = out1.get("bidp1")
        askp1 = out1.get("askp1")
        if bidp1 and askp1:
            return LimitOrderBook(
                out1["aspr_acpt_hour"],
                ticker,
                [
                    PQN(Decimal(bidp1), Decimal(out1.get("bidp_rsqn1", "1"))),
                ],
                [
                    PQN(Decimal(askp1), Decimal(out1.get("askp_rsqn1", "1"))),
                ],
            )
        return None

    async def candlestick(
        self, ticker: str, /, t: BaseTimer, *, interval: str = "D"
    ) -> Optional[DataFrame]:
        today = t.date()
        raw = await self._candlestick(
            ticker,
            date_add(today, -140).strftime("%Y%m%d"),
            today.strftime("%Y%m%d"),
            interval=interval,
        )
        if not raw:
            return None
        out2 = raw["output2"]
        return DataFrame(
            data={
                "Open": [float(x["stck_oprc"]) for x in out2],
                "High": [float(x["stck_hgpr"]) for x in out2],
                "Low": [float(x["stck_lwpr"]) for x in out2],
                "Close": [float(x["stck_clpr"]) for x in out2],
            },
            index=[int(x["stck_bsop_date"]) for x in out2],
        ).sort_index()


class DomesticStockWS:
    def __init__(self, common: Common, ws: AsyncWebSocketClient, /) -> None:
        self.__common = common
        self.__ws = ws

    @property
    def _common(self, /) -> Common:
        return self.__common

    @property
    def _ws(self, /) -> AsyncWebSocketClient:
        return self.__ws

    def _senddata_trade(self, ticker: str, subscribe: bool, /) -> str:
        return self._common.ws_senddata(subscribe, "H0STCNT0", ticker)

    def _senddata_orderbook(self, ticker: str, subscribe: bool, /) -> str:
        return self._common.ws_senddata(subscribe, "H0STASP0", ticker)

    def _senddata_execution(self, subscribe: bool, /) -> str:
        return self._common.ws_senddata(
            subscribe, "H0STCNI0", self._common.account_htsid
        )

    async def send_trade(
        self, ticker: str, /, *, subscribe: bool = True
    ) -> None:
        await self._ws.send(self._senddata_trade(ticker, subscribe))

    async def send_orderbook(
        self, ticker: str, /, *, subscribe: bool = True
    ) -> None:
        await self._ws.send(self._senddata_orderbook(ticker, subscribe))

    async def send_execution(self, /, *, subscribe: bool = True) -> None:
        await self._ws.send(self._senddata_execution(subscribe))

    async def recv(self, /) -> Any:
        return await self._ws.recv()
