from decimal import Decimal
from typing import Any, Union

from otlpy.base.net import AsyncHttpClient
from otlpy.base.order import (
    ORDER_TYPE,
    Buy,
    Cancel,
    Order,
    OrderAPI,
    Replace,
    Sell,
)
from otlpy.binance.common import Common


class Future(OrderAPI):
    def __init__(self, common: Common, client: AsyncHttpClient, /) -> None:
        super().__init__(has_replace=False)
        self.__common = common
        self.__client = client

    @property
    def _common(self, /) -> Common:
        return self.__common

    @property
    def _client(self, /) -> AsyncHttpClient:
        return self.__client

    async def _new_order(self, order: Union[Buy, Sell], /) -> None:
        common = self._common
        url_path = "/fapi/v1/order"
        data = common.signature(
            {
                "symbol": order.ticker,
                "side": order.oside.name,
                "type": order.otype.name,
                "timeInForce": "GTC",
                "quantity": str(order.qty),
                "price": str(order.price),
            }
        )
        headers = common.headers2()
        err, _rheaders, rdata = await self._client.post_params(
            url_path, headers, data
        )
        if err:
            return
        order.acknowledgment(rdata, f'{rdata["orderId"]}', order.qty)

    async def _cancel_order(self, order: Cancel, /) -> None:
        common = self._common
        url_path = "/fapi/v1/order"
        data = common.signature(
            {
                "symbol": order.ticker,
                "orderId": order.origin.rdata["orderId"],
            }
        )
        headers = common.headers2()
        err, _rheaders, rdata = await self._client.delete(
            url_path, headers, data
        )
        if err:
            return
        order.acknowledgment(rdata, f'C{rdata["orderId"]}', Decimal())

    async def buy(
        self,
        order_type: ORDER_TYPE,
        ticker: str,
        /,
        qty: Decimal,
        price: Decimal,
    ) -> Buy:
        order = Buy(order_type, ticker, qty, price)
        await self._new_order(order)
        return order

    async def sell(
        self,
        order_type: ORDER_TYPE,
        ticker: str,
        /,
        qty: Decimal,
        price: Decimal,
    ) -> Sell:
        order = Sell(order_type, ticker, qty, price)
        await self._new_order(order)
        return order

    async def cancel(self, origin: Order, /) -> Cancel:
        order = Cancel(origin, ORDER_TYPE.LIMIT)
        await self._cancel_order(order)
        return order

    async def replace(
        self, origin: Order, order_type: ORDER_TYPE, /, price: Decimal
    ) -> Replace:
        raise NotImplementedError

    async def all_orders(self, ticker: str, orderid: int, /) -> Any:
        common = self._common
        url_path = "/fapi/v1/allOrders"
        data = common.signature(
            {
                "symbol": ticker,
                "orderId": orderid,
            }
        )
        headers = common.headers2()
        err, _rheaders, rdata = await self._client.get(url_path, headers, data)
        if err:
            return None
        return rdata

    async def limitorderbook(self, ticker: str, /) -> Any:
        common = self._common
        url_path = "/fapi/v1/depth"
        data = {
            "symbol": ticker,
        }
        headers = common.headers1()
        err, _rheaders, rdata = await self._client.get(url_path, headers, data)
        if err:
            return None
        return rdata
