from decimal import Decimal
from typing import Optional

from otlpy.base.order import ORDER_SIDE, Order


class Inventory:
    def __init__(
        self,
        name: str,
        ticker: str,
        /,
        ticksize: Decimal,
        qty_unit: Decimal,
        unit: Decimal,
        fee: Decimal,
        fee_rate: Decimal,
        *,
        is_future: bool = False,
    ) -> None:
        assert (
            name
            and ticker
            and ticksize > 0
            and qty_unit > 0
            and unit > 0
            and fee >= 0
            and fee_rate >= 0
        )
        self.__name = name
        self.__ticker = ticker
        self.__ticksize = ticksize
        self.__qty_unit = qty_unit
        self.__unit = unit
        self.__fee = fee
        self.__fee_rate = fee_rate
        self.__is_future = is_future
        self.__cash = Decimal()
        self.__pos = Decimal()
        self.__price = Decimal()
        self.__realized_pnl = Decimal()
        self.__realized_fee = Decimal()
        self.__orders: dict[str, Order] = {}
        self.__opened_buy = Decimal()
        self.__opened_sell = Decimal()

    @property
    def name(self, /) -> str:
        return self.__name

    @property
    def ticker(self, /) -> str:
        return self.__ticker

    @property
    def ticksize(self, /) -> Decimal:
        return self.__ticksize

    @property
    def qty_unit(self, /) -> Decimal:
        return self.__qty_unit

    @property
    def unit(self, /) -> Decimal:
        return self.__unit

    @property
    def fee(self, /) -> Decimal:
        return self.__fee

    @property
    def fee_rate(self, /) -> Decimal:
        return self.__fee_rate

    @property
    def is_future(self, /) -> bool:
        return self.__is_future

    @property
    def cash(self, /) -> Decimal:
        return self.__cash

    def _add_cash(self, cash: Decimal, /) -> None:
        self.__cash += cash

    @property
    def pos(self, /) -> Decimal:
        return self.__pos

    def _add_pos(self, pos: Decimal, /) -> None:
        self.__pos += pos

    @property
    def price(self, /) -> Decimal:
        return self.__price

    def _set_price(self, price: Decimal, /) -> None:
        self.__price = price

    @property
    def realized_pnl(self, /) -> Decimal:
        return self.__realized_pnl

    def _add_realized_pnl(self, realized_pnl: Decimal, /) -> None:
        self.__realized_pnl += realized_pnl

    @property
    def realized_fee(self, /) -> Decimal:
        return self.__realized_fee

    def _add_realized_fee(self, realized_fee: Decimal, /) -> None:
        self.__realized_fee += realized_fee

    def init(
        self,
        /,
        *,
        cash: Decimal,
        pos: Decimal,
        price: Decimal,
        realized_pnl: Decimal,
        realized_fee: Decimal,
    ) -> None:
        self.__cash = cash
        self.__pos = pos
        self.__price = price
        self.__realized_pnl = realized_pnl
        self.__realized_fee = realized_fee
        self.clear_orders()

    def unrealized_pnl(self, price: Decimal, /) -> Decimal:
        return (price - self.price) * self.pos * self.unit

    def total_pnl(self, price: Decimal, /) -> Decimal:
        return (
            self.realized_pnl - self.realized_fee + self.unrealized_pnl(price)
        )

    @property
    def opened_buy(self, /) -> Decimal:
        return self.__opened_buy

    def _add_opened_buy(self, opened_buy: Decimal, /) -> None:
        self.__opened_buy += opened_buy

    @property
    def opened_sell(self, /) -> Decimal:
        return self.__opened_sell

    def _add_opened_sell(self, opened_sell: Decimal, /) -> None:
        self.__opened_sell += opened_sell

    @property
    def _orders(self, /) -> dict[str, Order]:
        return self.__orders

    @property
    def first_order(self, /) -> Optional[Order]:
        values = list(self._orders.values())
        if values:
            return values[0]
        return None

    def clear_orders(self, /) -> None:
        self.__orders = {}
        self.__opened_buy = Decimal()
        self.__opened_sell = Decimal()

    def add_order(self, order: Order, /, *, mid: Decimal) -> None:
        uid = order.uid
        assert uid and order.opened > 0
        assert self.ticker == order.ticker
        assert self._orders.get(uid) is None
        self.__orders[uid] = order
        if order.oside == ORDER_SIDE.BUY:
            self._add_opened_buy(order.opened)
        elif order.oside == ORDER_SIDE.SELL:
            self._add_opened_sell(order.opened)
        else:
            assert False
        self.logger_info(order, "", mid=mid)

    def filled_position(
        self,
        pos: Decimal,
        price: Decimal,
        /,
    ) -> None:
        s_pos = self.pos
        s_price = self.price
        s_unit = self.unit
        if s_pos * pos > 0:
            realized_pnl = Decimal()
            pp = s_price * s_pos + price * pos
            self._add_pos(pos)
            self._set_price(pp / s_pos)
        elif abs(pos) <= abs(s_pos):
            realized_pnl = (s_price - price) * pos * s_unit
            self._add_realized_pnl(realized_pnl)
            self._add_pos(pos)
        else:
            realized_pnl = (price - s_price) * s_pos * s_unit
            self._add_realized_pnl(realized_pnl)
            self._set_price(price)
            self._add_pos(pos)
        realized_fee = (self.fee + price * s_unit * self.fee_rate) * abs(pos)
        self._add_realized_fee(realized_fee)
        if not self.is_future:
            self._add_cash(-price * pos * s_unit - realized_fee)
        else:
            self._add_cash(realized_pnl - realized_fee)

    def filled_total(
        self,
        order: Order,
        total_filled: Decimal,
        total_filled_price: Decimal,
        total_opened: Decimal,
        /,
        *,
        mid: Decimal,
    ) -> None:
        filled, filled_price, opened = order.filled_total(
            total_filled,
            total_filled_price,
            total_opened,
        )
        if order.oside == ORDER_SIDE.BUY:
            self._add_opened_buy(-opened)
            pos = filled
        elif order.oside == ORDER_SIDE.SELL:
            self._add_opened_sell(-opened)
            pos = -filled
        else:
            assert False
        if pos != 0:
            self.filled_position(pos, filled_price)
        if total_opened == 0:
            del self.__orders[order.uid]
        if filled > 0 or opened > 0:
            self.logger_info(
                order,
                f"\nFIL, {order.uid}, {filled:f}, {filled_price:f}, {opened:f}",
                mid=mid,
            )

    def logger_info(self, order: Order, /, msg: str, *, mid: Decimal) -> None:
        n = len(self._orders)
        rpnl1 = self.realized_pnl
        rpnl2 = self.realized_fee
        upnl = self.unrealized_pnl(mid)
        tpnl = rpnl1 - rpnl2 + upnl
        order.logger_info(
            f"{msg}"
            f"\nINV, {self.name}, {n}, {mid:.2f}"
            f", {tpnl:.2f} = {rpnl1:.2f} - {rpnl2:.2f} + {upnl:.2f}"
            f", {self.pos:f}, {self.price:.2f}"
            f", {self.opened_buy:f}, {self.opened_sell:f}"
        )


class OrderInventoryMap:
    def __init__(self, /) -> None:
        self.__ois: dict[str, tuple[Order, Inventory]] = {}

    @property
    def _ois(self, /) -> dict[str, tuple[Order, Inventory]]:
        return self.__ois

    def clear_orders(self, /) -> None:
        for _order, inventory in self._ois.values():
            inventory.clear_orders()
        self.__ois = {}

    @property
    def first_order(self, /) -> Optional[Order]:
        values = list(self._ois.values())
        if values:
            return values[0][0]
        return None

    def add_opened(
        self, order: Order, inventory: Inventory, /, *, mid: Decimal
    ) -> None:
        uid = order.uid
        assert uid and order.opened > 0
        assert order.ticker == inventory.ticker
        assert self._ois.get(uid) is None
        self.__ois[uid] = (order, inventory)
        inventory.add_order(order, mid=mid)

    def filled_total(
        self,
        uid: str,
        total_filled: Decimal,
        total_filled_price: Decimal,
        total_opened: Decimal,
        /,
        *,
        mid: Decimal,
    ) -> None:
        assert uid
        oi = self._ois.get(uid)
        if oi is None:
            return
        order, inventory = oi
        inventory.filled_total(
            order, total_filled, total_filled_price, total_opened, mid=mid
        )
        if total_opened == 0:
            del self.__ois[uid]
