from typing import Any, Union

from loguru import logger

from otlpy.base.account import Buy, Cancel, Order, OrderAPI, Replace, Sell
from otlpy.base.market import ORDER_TYPE
from otlpy.base.net import AsyncHttpClient
from otlpy.binance.common import Common


class Future(OrderAPI):
    def __init__(self, common: Common, client: AsyncHttpClient, /) -> None:
        super().__init__(has_replace=False)
        self.__common = common
        self.__client = client

    @property
    def _common(self, /) -> Common:
        return self.__common

    @property
    def _client(self, /) -> AsyncHttpClient:
        return self.__client

    async def _new_order(self, order: Union[Buy, Sell], /) -> None:
        common = self._common
        url_path = "/fapi/v1/order"
        data = common.signature(
            {
                "symbol": order.ticker,
                "side": order.oside.name,
                "type": order.otype.name,
                "timeInForce": "GTC",
                "quantity": str(order.qty),
                "price": str(order.price),
            }
        )
        headers = common.headers2()
        err, _rheaders, rdata = await self._client.post_params(
            url_path, headers, data
        )
        logger.info(
            f"NEW {order.oside.name} {order.ticker} {order.qty} {order.price} {rdata}"
        )
        if err:
            return
        order.acknowledgment(rdata, f'{rdata["orderId"]}', order.qty)

    async def _cancel_order(self, order: Cancel, /) -> None:
        common = self._common
        url_path = "/fapi/v1/order"
        data = common.signature(
            {
                "symbol": order.ticker,
                "orderId": order.origin.rdata["orderId"],
            }
        )
        headers = common.headers2()
        err, _rheaders, rdata = await self._client.delete(
            url_path, headers, data
        )
        logger.info(f"CANCEL {order.oside.name} {order.ticker} {rdata}")
        if err:
            return
        order.acknowledgment(rdata, f'C{rdata["orderId"]}', 0)

    async def buy(
        self, order_type: ORDER_TYPE, ticker: str, /, qty: float, price: float
    ) -> Buy:
        order = Buy(order_type, ticker, qty, price)
        await self._new_order(order)
        return order

    async def sell(
        self, order_type: ORDER_TYPE, ticker: str, /, qty: float, price: float
    ) -> Sell:
        order = Sell(order_type, ticker, qty, price)
        await self._new_order(order)
        return order

    async def cancel(self, origin: Order, /) -> Cancel:
        order = Cancel(origin, ORDER_TYPE.LIMIT)
        await self._cancel_order(order)
        return order

    async def replace(
        self, origin: Order, order_type: ORDER_TYPE, /, price: float
    ) -> Replace:
        raise NotImplementedError

    async def all_orders(self, ticker: str, starttime: int, /) -> Any:
        common = self._common
        url_path = "/fapi/v1/allOrders"
        data = common.signature(
            {
                "symbol": ticker,
                "startTime": starttime,
            }
        )
        headers = common.headers2()
        err, _rheaders, rdata = await self._client.get(url_path, headers, data)
        if err:
            return None
        return rdata

    async def limitorderbook(self, ticker: str, /) -> Any:
        common = self._common
        url_path = "/fapi/v1/depth"
        data = {
            "symbol": ticker,
        }
        headers = common.headers1()
        err, _rheaders, rdata = await self._client.get(url_path, headers, data)
        if err:
            return None
        return rdata
