from datetime import datetime
from typing import Any, Optional

from numpy import abs as _abs
from tompy.datetime import now as _now

from otlpy.base.market import ORDER_SIDE, ORDER_TYPE, BaseOrder


class Order(BaseOrder):
    def __init__(
        self,
        oside: ORDER_SIDE,
        otype: ORDER_TYPE,
        ticker: str,
        /,
        qty: float,
        price: float,
    ) -> None:
        super().__init__(oside, otype, ticker, qty, price)
        self.__rdata: dict[str, Any] = {}
        self.__uid = ""
        self.__filled: float = 0
        self.__filled_price: float = 0
        self.__opened: float = 0

    @property
    def rdata(self, /) -> dict[str, Any]:
        return self.__rdata

    def _set_rdata(self, rdata: dict[str, Any], /) -> None:
        self.__rdata = rdata

    @property
    def uid(self, /) -> str:
        return self.__uid

    def _set_uid(self, uid: str, /) -> None:
        self.__uid = uid

    @property
    def filled(self, /) -> float:
        return self.__filled

    def _set_filled(self, filled: float, /) -> None:
        self.__filled = filled

    @property
    def filled_price(self, /) -> float:
        return self.__filled_price

    def _set_filled_price(self, filled_price: float, /) -> None:
        self.__filled_price = filled_price

    @property
    def opened(self, /) -> float:
        return self.__opened

    def _set_opened(self, opened: float, /) -> None:
        self.__opened = opened

    def acknowledgment(
        self,
        rdata: dict[str, Any],
        uid: str,
        opened: float,
        /,
    ) -> None:
        self._set_rdata(rdata)
        self._set_uid(uid)
        self._set_opened(opened)

    def filled_event(
        self,
        filled: float,
        filled_price: float,
        /,
        *,
        opened: Optional[float] = None,
    ) -> None:
        if opened is None:
            opened = filled
        assert filled <= self.opened and opened <= self.opened
        if filled > 0:
            total_filled = self.filled + filled
            self._set_filled_price(
                (self.filled * self.filled_price + filled * filled_price)
                / total_filled
            )
            self._set_filled(total_filled)
        if opened > 0:
            self._set_opened(self.opened - opened)

    def filled_total(
        self,
        total_filled: float,
        total_filled_price: float,
        total_opened: float,
        /,
    ) -> tuple[float, float, float]:
        assert total_filled >= self.filled and self.opened >= total_opened
        filled = total_filled - self.filled
        if filled > 0:
            filled_price = (
                total_filled * total_filled_price
                - self.filled * self.filled_price
            ) / filled
        else:
            filled_price = 0
        opened = self.opened - total_opened
        self.filled_event(filled, filled_price, opened=opened)
        return filled, filled_price, opened


class Buy(Order):
    def __init__(
        self,
        otype: ORDER_TYPE,
        ticker: str,
        /,
        qty: float,
        price: float,
    ) -> None:
        super().__init__(
            ORDER_SIDE.BUY,
            otype,
            ticker,
            qty,
            price,
        )


class Sell(Order):
    def __init__(
        self,
        otype: ORDER_TYPE,
        ticker: str,
        /,
        qty: float,
        price: float,
    ) -> None:
        super().__init__(
            ORDER_SIDE.SELL,
            otype,
            ticker,
            qty,
            price,
        )


class Cancel(Order):
    def __init__(
        self,
        origin: Order,
        otype: ORDER_TYPE,
        /,
    ) -> None:
        super().__init__(
            origin.oside,
            otype,
            origin.ticker,
            origin.qty,
            0,
        )
        self.__origin = origin

    @property
    def origin(self, /) -> Order:
        return self.__origin


class Replace(Order):
    def __init__(
        self,
        origin: Order,
        otype: ORDER_TYPE,
        /,
        price: float,
    ) -> None:
        super().__init__(
            origin.oside,
            otype,
            origin.ticker,
            origin.qty,
            price,
        )
        self.__origin = origin

    @property
    def origin(self, /) -> Order:
        return self.__origin


class OrderAPI:
    def __init__(self, /, *, has_replace: bool = False):
        self.__has_replace = has_replace

    @property
    def has_replace(self, /) -> bool:
        return self.__has_replace

    async def buy(
        self, order_type: ORDER_TYPE, ticker: str, /, qty: float, price: float
    ) -> Buy:
        raise NotImplementedError

    async def buy_market(self, ticker: str, /, qty: float) -> Buy:
        return await self.buy(ORDER_TYPE.MARKET, ticker, qty, 0)

    async def buy_limit(self, ticker: str, /, qty: float, price: float) -> Buy:
        return await self.buy(ORDER_TYPE.LIMIT, ticker, qty, price)

    async def sell(
        self, order_type: ORDER_TYPE, ticker: str, /, qty: float, price: float
    ) -> Sell:
        raise NotImplementedError

    async def sell_market(self, ticker: str, /, qty: float) -> Sell:
        return await self.sell(ORDER_TYPE.MARKET, ticker, qty, 0)

    async def sell_limit(
        self, ticker: str, /, qty: float, price: float
    ) -> Sell:
        return await self.sell(ORDER_TYPE.LIMIT, ticker, qty, price)

    async def cancel(self, origin: Order, /) -> Cancel:
        raise NotImplementedError

    async def replace(
        self, origin: Order, order_type: ORDER_TYPE, /, price: float
    ) -> Replace:
        raise NotImplementedError

    async def replace_market(self, origin: Order, /) -> Replace:
        return await self.replace(origin, ORDER_TYPE.MARKET, 0)

    async def replace_limit(self, origin: Order, /, price: float) -> Replace:
        return await self.replace(origin, ORDER_TYPE.LIMIT, price)


class Inventory:
    def __init__(
        self,
        name: str,
        ticker: str,
        /,
        ticksize: float,
        unit: float,
        fee: float,
        fee_rate: float,
    ) -> None:
        assert (
            name
            and ticker
            and ticksize > 0
            and unit > 0
            and fee >= 0
            and fee_rate >= 0
        )
        self.__name = name
        self.__ticker = ticker
        self.__ticksize = ticksize
        self.__unit = unit
        self.__fee = fee
        self.__fee_rate = fee_rate
        self.__orders: dict[str, Order] = {}
        self.__realized_pnl: float = 0
        self.__realized_fee: float = 0
        self.__pos: float = 0
        self.__price: float = 0
        self.__opened_buy: float = 0
        self.__opened_sell: float = 0
        self.__updated_at = _now()

    @property
    def name(self, /) -> str:
        return self.__name

    @property
    def ticker(self, /) -> str:
        return self.__ticker

    @property
    def ticksize(self, /) -> float:
        return self.__ticksize

    @property
    def unit(self, /) -> float:
        return self.__unit

    @property
    def fee(self, /) -> float:
        return self.__fee

    @property
    def fee_rate(self, /) -> float:
        return self.__fee_rate

    @property
    def realized_pnl(self, /) -> float:
        return self.__realized_pnl

    def _add_realized_pnl(self, realized_pnl: float, /) -> None:
        self.__realized_pnl += realized_pnl

    @property
    def realized_fee(self, /) -> float:
        return self.__realized_fee

    def _add_realized_fee(self, realized_fee: float, /) -> None:
        self.__realized_fee += realized_fee

    @property
    def pos(self, /) -> float:
        return self.__pos

    def _add_pos(self, pos: float, /) -> None:
        self.__pos += pos

    @property
    def price(self, /) -> float:
        return self.__price

    def _set_price(self, price: float, /) -> None:
        self.__price = price

    @property
    def opened_buy(self, /) -> float:
        return self.__opened_buy

    def _add_opened_buy(self, opened_buy: float, /) -> None:
        self.__opened_buy += opened_buy

    @property
    def opened_sell(self, /) -> float:
        return self.__opened_sell

    def _add_opened_sell(self, opened_sell: float, /) -> None:
        self.__opened_sell += opened_sell

    @property
    def updated_at(self, /) -> datetime:
        return self.__updated_at

    def _set_updated_at(self, /) -> None:
        self.__updated_at = _now()

    def unrealized_pnl(self, price: float, /) -> float:
        return (price - self.price) * self.pos * self.unit

    def total_pnl(self, price: float, /) -> float:
        return (
            self.realized_pnl - self.realized_fee + self.unrealized_pnl(price)
        )

    @property
    def first_order(self, /) -> Optional[Order]:
        if self.__orders:
            return next(iter(self.__orders.values()))
        return None

    def clear_orders(self, /) -> None:
        self.__orders = {}
        self.__realized_pnl = 0
        self.__realized_fee = 0
        self.__opened_buy = 0
        self.__opened_sell = 0

    def add_order(self, order: Order, /) -> None:
        uid = order.uid
        assert uid and order.opened > 0
        assert self.ticker == order.ticker
        assert self.__orders.get(uid) is None
        self.__orders[uid] = order
        if order.oside == ORDER_SIDE.BUY:
            self._add_opened_buy(order.opened)
        elif order.oside == ORDER_SIDE.SELL:
            self._add_opened_sell(order.opened)
        else:
            assert False

    def filled_position(
        self,
        pos: float,
        price: float,
        /,
    ) -> None:
        if self.pos * pos > 0:
            pp = self.price * self.pos + price * pos
            self._add_pos(pos)
            self._set_price(pp / self.pos)
        elif _abs(pos) <= _abs(self.pos):
            self._add_realized_pnl((self.price - price) * pos * self.unit)
            self._add_pos(pos)
        else:
            self._add_realized_pnl((price - self.price) * self.pos * self.unit)
            self._set_price(price)
            self._add_pos(pos)
        self._add_realized_fee(
            (self.fee + price * self.unit * self.fee_rate) * _abs(pos)
        )
        self._set_updated_at()

    def filled_total(
        self,
        order: Order,
        total_filled: float,
        total_filled_price: float,
        total_opened: float,
        /,
    ) -> None:
        filled, filled_price, opened = order.filled_total(
            total_filled,
            total_filled_price,
            total_opened,
        )
        if order.oside == ORDER_SIDE.BUY:
            self._add_opened_buy(-opened)
            pos = filled
        elif order.oside == ORDER_SIDE.SELL:
            self._add_opened_sell(-opened)
            pos = -filled
        else:
            assert False
        if pos != 0:
            self.filled_position(pos, filled_price)
        if total_opened == 0:
            del self.__orders[order.uid]


class Book:
    def __init__(self, /) -> None:
        self.__ois: dict[str, tuple[Order, Inventory]] = {}

    @property
    def first_order(self, /) -> Optional[Order]:
        if self.__ois:
            order, _inventory = next(iter(self.__ois.values()))
            return order
        return None

    def clear_orders(self, /) -> None:
        self.__ois = {}

    def add(self, order: Order, inventory: Inventory, /) -> None:
        uid = order.uid
        assert uid and order.opened > 0
        assert order.ticker == inventory.ticker
        assert self.__ois.get(uid) is None
        self.__ois[uid] = (order, inventory)
        inventory.add_order(order)

    def filled_total(
        self,
        uid: str,
        total_filled: float,
        total_filled_price: float,
        total_opened: float,
        /,
    ) -> None:
        assert uid
        oi = self.__ois.get(uid)
        if oi is None:
            return
        order, inventory = oi
        inventory.filled_total(
            order, total_filled, total_filled_price, total_opened
        )
        if total_opened == 0:
            del self.__ois[uid]
