from typing import Any
from urllib.parse import urlencode as _urlencode

from tompy.datetime import unix_time

from otlpy.base.crypto import hmac_sha256 as _hmac_sha256
from otlpy.binance.settings import Settings


def _drop_none(data: dict[str, Any]) -> dict[str, Any]:
    out = {}
    for k, v in data.items():
        if v is not None:
            out[k] = v
    return out


def _timestamp() -> int:
    return int(unix_time() * 1000)


class Common:
    def __init__(self, settings: Settings) -> None:
        self.__settings = settings
        self.__url_base = "https://api.binance.com"
        self.__content_type = "application/json; charset=UTF-8"

    @property
    def _settings(self) -> Settings:
        return self.__settings

    @property
    def url_base(self) -> str:
        return self.__url_base

    @property
    def _content_type(self) -> str:
        return self.__content_type

    @property
    def _app_key(self) -> str:
        return self._settings.binance_app_key

    @property
    def _app_secret(self) -> str:
        return self._settings.binance_app_secret

    def headers1(self) -> dict[str, str]:
        return {
            "content-type": self._content_type,
        }

    def headers2(self) -> dict[str, str]:
        return {
            "content-type": self._content_type,
            "X-MBX-APIKEY": self._app_key,
        }

    def signature(self, data: dict[str, Any]) -> dict[str, Any]:
        out = _drop_none(data)
        out["timestamp"] = _timestamp()
        out["signature"] = _hmac_sha256(
            self._app_secret, _urlencode(out, True)
        )
        return out
