# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['numpy_ros']

package_data = \
{'': ['*']}

install_requires = \
['numpy-quaternion>=2021.4.5,<2022.0.0', 'numpy>=1.19.5,<2.0.0']

extras_require = \
{'quaternion-extras': ['numba>=0.53.1,<0.54.0', 'scipy>=0.18.0']}

setup_kwargs = {
    'name': 'numpy-ros',
    'version': '0.1.1',
    'description': 'Seamlessly convert between ROS messages and NumPy arrays.',
    'long_description': '# <img alt="numpy-ros" src="media/logo.png" >\n\nSeamlessly convert between ROS messages and NumPy arrays. Installable via your \nfavorite Python dependency management system (pip, poetry, pipenv, ...) &ndash; \nno matter what ROS version you\'re on.\n\n### Table of Contents\n - [Installation](#installation)\n - [Usage](#usage)\n   - [Supported Types](#supported-types)\n   - [Custom Handlers](#custom-handlers)\n - [License](#license)\n - [Acknowledgements](#acknowledgements)\n\n\n# Installation\n\nnumpy-ros is distributed via PyPI for Python 3.6 and higher. To install, run:\n\n```bash\n# pip\n$ pip install numpy-ros\n\n# poetry\n$ poetry add numpy-ros\n\n# pipenv\n$ pipenv install numpy-ros\n```\n\nROS messages of type `Quaternion` are by default converted into numpy arrays of\ntype `np.quaternion`, which are provided by the \n[numpy-quaternion](https://github.com/moble/quaternion) package. To make use\nof hardware acceleration and certain advanced features of numpy-quaternion,\nconsider installing with the optional scipy and numba dependencies. For more \ninformation, see the numpy-quaternion documentation.\n\n```bash\n$ pip install \'numpy-ros[quaternion-extras]\'\n```\n\nSupport for installation via conda is targeted for future releases.\n\n# Usage\n\nConverting a ROS message into a NumPy representation and back is as simple as\ncalling the `to_numpy` and `to_message` functions:\n\n\n```python\nfrom numpy_ros import to_numpy, to_message\nfrom geometry_msgs.msg import Point\n\n\npoint = Point(3.141, 2.718, 42.1337)\n\n# ROS to NumPy\nas_array = to_numpy(point)\n\n\n# NumPy to ROS\nmessage = to_message(Point, as_array)\n```\n\nNote that `to_numpy` expects the ROS message to convert as its first \npositional argument, whereas `to_message` expects the ROS message _type_ (i.e.\nthe specific subclass of `genpy.Message`). Other than that, either function may\ntake any number of positional and keyword arguments. \n\nFor some message types, keyword arguments can be used to finely control the\nconversion process. For example, `Point` messages (among others) can be\nconverted directly into homogeneous coordinates:\n\n```python\n# Returns array([ 3.141 ,  2.718 , 42.1337])\nas_array = to_numpy(point)\n\n# Returns array([ 3.141 , 2.718 , 42.1337 , 1.0])\nas_array_hom = to_numpy(point, homogeneous=True)\n```\n\nAlso note that `to_numpy` may return multiple values, depending on the message\ntype. Similarly, `to_message` may require multiple positional arguments to\nassemble the message object:\n\n```python\nfrom geometry_msgs.msg import Inertia\n\ninertia = Inertia(...)\n\nmass, center_of_mass, inertia_tensor = to_numpy(inertia)\nmessage = to_message(Inertia, mass, center_of_mass, inertia_tensor)\n\ninertia == message # true\n```\n\n## Supported Types\n\nCurrently, numpy-ros provides conversions of the message types listed below. More extensive documentation will be provided in the near future. \n\n## geometry_msgs\n\n| Message Type                 | `to_numpy` | `from_numpy` | `kwargs`                         |\n|------------------------------|:----------:|:------------:|----------------------------------|\n| `Accel`                      | ✅         | ✅           | `homogeneous` (default: `False`) |\n| `AccelStamped`               | ✅         | N/A          | `homogeneous` (default: `False`) |\n| `AccelWithCovariance`        | ✅         | ✅           | `homogeneous` (default: `False`) |\n| `AccelWithCovarianceStamped` | ✅         | N/A          | `homogeneous` (default: `False`) |\n| `Inertia`                    | ✅         | ✅           | `homogeneous` (default: `False`) |\n| `InertiaStamped`             | ✅         | N/A          | `homogeneous` (default: `False`) |\n| `Point`                      | ✅         | ✅           | `homogeneous` (default: `False`) |\n| `Point32`                    | ✅         | ✅           | `homogeneous` (default: `False`) |\n| `PointStamped`               | ✅         | N/A          | `homogeneous` (default: `False`) |\n| `Polygon`                    | ✅         | ✅           | `homogeneous` (default: `False`) |\n| `PolygonStamped`             | ✅         | N/A          | `homogeneous` (default: `False`) |\n| `Pose`                       | ✅         | ✅           | `homogeneous` (default: `False`) |\n| `PoseArray`                  | ✅         | ✅           | `homogeneous` (default: `False`)<br>`as_array` (default: `False`) |\n| `PoseStamped`                | ✅         | N/A          | `homogeneous` (default: `False`) |\n| `PoseWithCovariance`         | ✅         | ✅           | `homogeneous` (default: `False`) |\n| `PoseWithCovarianceStamped`  | ✅         | N/A          | `homogeneous` (default: `False`) |\n| `Quaternion`                 | ✅         | ✅           |\n| `QuaternionStamped`          | ✅         | N/A          |\n| `Transform`                  | ✅         | ✅           | `homogeneous` (default: `False`) |\n| `TransformStamped`           | ✅         | N/A          | `homogeneous` (default: `False`) |\n| `Twist`                      | ✅         | ✅           | `homogeneous` (default: `False`) |\n| `TwistStamped`               | ✅         | N/A          | `homogeneous` (default: `False`) |\n| `TwistWithCovariance`        | ✅         | ✅           | `homogeneous` (default: `False`) |\n| `TwistWithCovarianceStamped` | ✅         | N/A          | `homogeneous` (default: `False`) |\n| `Vector3`                    | ✅         | ✅           | `homogeneous` (default: `False`) |\n| `Vector3Stamped`             | ✅         | N/A          | `homogeneous` (default: `False`) |\n| `Wrench`                     | ✅         | ✅           | `homogeneous` (default: `False`) |\n| `WrenchStamped`              | ✅         | N/A          | `homogeneous` (default: `False`) |\n\n\nMore message types will be added in future versions.\n\n## Custom Handlers\n\nCustom conversion handlers can be registred by decorating them with\n`converts_to_numpy` or `converts_to_message`, respectively. \n\n```python\nfrom my_ros_package.msg import MyMessageType, MyOtherMessageType \nfrom numpy_ros import converts_to_numpy, converts_to_message\n\n\n@converts_to_numpy(MyMessageType, MyOtherMessageType)\ndef my_messages_to_numpy(message, option1=True, option2=False):\n\n    as_array = ...\n    return as_array\n\n\n@converts_to_message(MyMessageType, MyOtherMessageType)\ndef numpy_to_my_messages(message_type, arg1, arg2, option3=\'foo\'):\n\n    if issubclass(message_type, MyMessageType):\n        ...\n\n    elif issubclass(message, MyOtherMessageType):\n        ...\n\n    else:\n        raise TypeError\n\n\n    as_message = message_type(...)\n    return as_message\n\n\n# This works now!\nmessage = MyMessage(...)\nas_array = to_numpy(message)\n```\n\nAfter registring, `to_numpy` and `to_message` will transparently dispatch to the\nrespective handlers. Note that the same handler can be registered to multiple\nmessage types, as in the example above. Moreover, to-message-handlers are \nrequired to expect the message type as their first positional argument.\n\n\n# License\n\nnumpy-ros is available under the MIT license. For more information, see the\n[LICENSE](LICENSE) file in this repository.\n\n\n# Acknowledgements\n\nThis work started as a side-project during my internship at \n[V-R Robotics](https://v-r-robotics.com/), inspired by an \n[earlier work](https://github.com/eric-wieser/ros_numpy) by Eric Wieser.\n\nnumpy-ros is in no way, shape, or form affiliated with numpy, NumFocus, ROS, or\nthe Open Robotics Foundation.',
    'author': 'Lucas Wolf',
    'author_email': 'iamlucaswolf@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/iamlucaswolf/numpy-ros',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'extras_require': extras_require,
    'python_requires': '>=3.6,<3.10',
}


setup(**setup_kwargs)
