from django import forms
from django.core.exceptions import ValidationError

from nautobot.core.forms import (
    add_blank_choice,
    BootstrapMixin,
    BulkEditNullBooleanSelect,
    BulkRenameForm,
    CommentField,
    ConfirmationForm,
    CSVChoiceField,
    CSVModelChoiceField,
    DynamicModelChoiceField,
    DynamicModelMultipleChoiceField,
    ExpandableNameField,
    form_from_model,
    SlugField,
    SmallTextarea,
    StaticSelect2,
    TagFilterField,
)
from nautobot.core.forms.constants import BOOLEAN_WITH_BLANK_CHOICES
from nautobot.dcim.choices import InterfaceModeChoices
from nautobot.dcim.constants import INTERFACE_MTU_MAX, INTERFACE_MTU_MIN
from nautobot.dcim.forms import InterfaceCommonForm, INTERFACE_MODE_HELP_TEXT
from nautobot.dcim.form_mixins import (
    LocatableModelBulkEditFormMixin,
    LocatableModelCSVFormMixin,
    LocatableModelFilterFormMixin,
    LocatableModelFormMixin,
)
from nautobot.dcim.models import Device, Location, Platform, Rack
from nautobot.extras.forms import (
    CustomFieldModelBulkEditFormMixin,
    CustomFieldModelCSVForm,
    NautobotBulkEditForm,
    NautobotModelForm,
    NautobotFilterForm,
    LocalContextFilterForm,
    LocalContextModelForm,
    LocalContextModelBulkEditForm,
    RoleModelBulkEditFormMixin,
    RoleModelCSVFormMixin,
    RoleModelFilterFormMixin,
    StatusModelBulkEditFormMixin,
    StatusModelCSVFormMixin,
    StatusModelFilterFormMixin,
    TagsBulkEditFormMixin,
)
from nautobot.extras.models import Status
from nautobot.ipam.models import IPAddress, IPAddressToInterface, VLAN
from nautobot.tenancy.forms import TenancyFilterForm, TenancyForm
from nautobot.tenancy.models import Tenant
from .models import Cluster, ClusterGroup, ClusterType, VirtualMachine, VMInterface


#
# Cluster types
#


class ClusterTypeForm(NautobotModelForm):
    slug = SlugField()

    class Meta:
        model = ClusterType
        fields = [
            "name",
            "slug",
            "description",
        ]


class ClusterTypeCSVForm(CustomFieldModelCSVForm):
    class Meta:
        model = ClusterType
        fields = ClusterType.csv_headers


#
# Cluster groups
#


class ClusterGroupForm(NautobotModelForm):
    slug = SlugField()

    class Meta:
        model = ClusterGroup
        fields = [
            "name",
            "slug",
            "description",
        ]


class ClusterGroupCSVForm(CustomFieldModelCSVForm):
    class Meta:
        model = ClusterGroup
        fields = ClusterGroup.csv_headers


#
# Clusters
#


class ClusterForm(LocatableModelFormMixin, NautobotModelForm, TenancyForm):
    cluster_type = DynamicModelChoiceField(queryset=ClusterType.objects.all())
    cluster_group = DynamicModelChoiceField(queryset=ClusterGroup.objects.all(), required=False)
    comments = CommentField()

    class Meta:
        model = Cluster
        fields = (
            "name",
            "cluster_type",
            "cluster_group",
            "tenant",
            "location",
            "comments",
            "tags",
        )


class ClusterCSVForm(LocatableModelCSVFormMixin, CustomFieldModelCSVForm):
    cluster_type = CSVModelChoiceField(
        queryset=ClusterType.objects.all(),
        to_field_name="name",
        help_text="Type of cluster",
    )
    cluster_group = CSVModelChoiceField(
        queryset=ClusterGroup.objects.all(),
        to_field_name="name",
        required=False,
        help_text="Assigned cluster group",
    )
    tenant = CSVModelChoiceField(
        queryset=Tenant.objects.all(),
        to_field_name="name",
        required=False,
        help_text="Assigned tenant",
    )

    class Meta:
        model = Cluster
        fields = Cluster.csv_headers


class ClusterBulkEditForm(
    TagsBulkEditFormMixin,
    LocatableModelBulkEditFormMixin,
    NautobotBulkEditForm,
):
    pk = forms.ModelMultipleChoiceField(queryset=Cluster.objects.all(), widget=forms.MultipleHiddenInput())
    cluster_type = DynamicModelChoiceField(queryset=ClusterType.objects.all(), required=False)
    cluster_group = DynamicModelChoiceField(queryset=ClusterGroup.objects.all(), required=False)
    tenant = DynamicModelChoiceField(queryset=Tenant.objects.all(), required=False)
    comments = CommentField(widget=SmallTextarea, label="Comments")

    class Meta:
        model = Cluster
        nullable_fields = [
            "cluster_group",
            "location",
            "comments",
            "tenant",
        ]


class ClusterFilterForm(NautobotFilterForm, LocatableModelFilterFormMixin, TenancyFilterForm):
    model = Cluster
    field_order = ["q", "cluster_type", "location", "cluster_group", "tenant_group", "tenant"]
    q = forms.CharField(required=False, label="Search")
    cluster_type = DynamicModelMultipleChoiceField(
        queryset=ClusterType.objects.all(), to_field_name="slug", required=False
    )
    cluster_group = DynamicModelMultipleChoiceField(
        queryset=ClusterGroup.objects.all(),
        to_field_name="slug",
        required=False,
        null_option="None",
    )
    tag = TagFilterField(model)


class ClusterAddDevicesForm(BootstrapMixin, forms.Form):
    location = DynamicModelChoiceField(
        queryset=Location.objects.all(),
        required=False,
        query_params={"content_type": "virtualization.cluster"},
    )
    rack = DynamicModelChoiceField(
        queryset=Rack.objects.all(),
        required=False,
        null_option="None",
        query_params={
            "location": "$location",
        },
    )
    devices = DynamicModelMultipleChoiceField(
        queryset=Device.objects.all(),
        query_params={
            "location": "$location",
            "rack": "$rack",
            "cluster": "null",
        },
    )

    class Meta:
        fields = [
            "location",
            "rack",
            "devices",
        ]

    def __init__(self, cluster, *args, **kwargs):

        self.cluster = cluster

        super().__init__(*args, **kwargs)

        self.fields["devices"].choices = []

    def clean(self):
        super().clean()

        # If the Cluster is assigned to a Location, all Devices must exist within that Location
        if self.cluster.location is not None:
            for device in self.cleaned_data.get("devices", []):
                if device.location and self.cluster.location not in device.location.ancestors(include_self=True):
                    raise ValidationError(
                        {
                            "devices": f"{device} belongs to a location ({device.location}) that "
                            f"does not fall within this cluster's location ({self.cluster.location})."
                        }
                    )


class ClusterRemoveDevicesForm(ConfirmationForm):
    pk = forms.ModelMultipleChoiceField(queryset=Device.objects.all(), widget=forms.MultipleHiddenInput())


#
# Virtual Machines
#


class VirtualMachineForm(NautobotModelForm, TenancyForm, LocalContextModelForm):
    cluster_group = DynamicModelChoiceField(
        queryset=ClusterGroup.objects.all(),
        required=False,
        null_option="None",
        initial_params={"clusters": "$cluster"},
    )
    cluster = DynamicModelChoiceField(
        queryset=Cluster.objects.all(), query_params={"cluster_group_id": "$cluster_group"}
    )
    platform = DynamicModelChoiceField(queryset=Platform.objects.all(), required=False)

    class Meta:
        model = VirtualMachine
        fields = [
            "name",
            "status",
            "cluster_group",
            "cluster",
            "role",
            "tenant_group",
            "tenant",
            "platform",
            "primary_ip4",
            "primary_ip6",
            "vcpus",
            "memory",
            "disk",
            "comments",
            "tags",
            "local_config_context_data",
            "local_config_context_schema",
        ]
        help_texts = {
            "local_config_context_data": "Local config context data overwrites all sources contexts in the final rendered "
            "config context",
        }
        widgets = {
            "primary_ip4": StaticSelect2(),
            "primary_ip6": StaticSelect2(),
        }

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)

        if self.instance.present_in_database:

            # Compile list of choices for primary IPv4 and IPv6 addresses
            for family in [4, 6]:
                ip_choices = [(None, "---------")]

                # Gather PKs of all interfaces belonging to this VM
                interface_ids = self.instance.interfaces.values_list("pk", flat=True)

                # Collect interface IPs
                interface_ip_assignments = IPAddressToInterface.objects.filter(
                    vm_interface__in=interface_ids
                ).prefetch_related("ip_address")
                if interface_ip_assignments.exists():
                    ip_list = [
                        (
                            assignment.ip_address.id,
                            f"{assignment.ip_address.address} ({assignment.vm_interface})",
                        )
                        for assignment in interface_ip_assignments
                        if assignment.ip_address.family == family
                    ]
                    ip_choices.append(("Interface IPs", ip_list))

                    # Collect NAT IPs
                    nat_ips = []
                    for ip_assignment in interface_ip_assignments:
                        if not ip_assignment.ip_address.nat_outside_list.exists():
                            continue
                        nat_ips.extend(
                            [
                                (ip.id, f"{ip.address} (NAT)")
                                for ip in ip_assignment.ip_address.nat_outside_list.all()
                                if ip.family == family
                            ]
                        )
                    ip_choices.append(("NAT IPs", nat_ips))
                self.fields[f"primary_ip{family}"].choices = ip_choices

        else:

            # An object that doesn't exist yet can't have any IPs assigned to it
            self.fields["primary_ip4"].choices = []
            self.fields["primary_ip4"].widget.attrs["readonly"] = True
            self.fields["primary_ip6"].choices = []
            self.fields["primary_ip6"].widget.attrs["readonly"] = True


class VirtualMachineCSVForm(StatusModelCSVFormMixin, RoleModelCSVFormMixin, CustomFieldModelCSVForm):
    cluster = CSVModelChoiceField(
        queryset=Cluster.objects.all(),
        to_field_name="name",
        help_text="Assigned cluster",
    )
    tenant = CSVModelChoiceField(
        queryset=Tenant.objects.all(),
        required=False,
        to_field_name="name",
        help_text="Assigned tenant",
    )
    platform = CSVModelChoiceField(
        queryset=Platform.objects.all(),
        required=False,
        to_field_name="name",
        help_text="Assigned platform",
    )

    class Meta:
        model = VirtualMachine
        fields = VirtualMachine.csv_headers


class VirtualMachineBulkEditForm(
    TagsBulkEditFormMixin,
    StatusModelBulkEditFormMixin,
    RoleModelBulkEditFormMixin,
    NautobotBulkEditForm,
    LocalContextModelBulkEditForm,
):
    pk = forms.ModelMultipleChoiceField(queryset=VirtualMachine.objects.all(), widget=forms.MultipleHiddenInput())
    cluster = DynamicModelChoiceField(queryset=Cluster.objects.all(), required=False)
    tenant = DynamicModelChoiceField(queryset=Tenant.objects.all(), required=False)
    platform = DynamicModelChoiceField(queryset=Platform.objects.all(), required=False)
    vcpus = forms.IntegerField(required=False, label="vCPUs")
    memory = forms.IntegerField(required=False, label="Memory (MB)")
    disk = forms.IntegerField(required=False, label="Disk (GB)")
    comments = CommentField(widget=SmallTextarea, label="Comments")

    class Meta:
        nullable_fields = [
            "tenant",
            "platform",
            "vcpus",
            "memory",
            "disk",
            "comments",
        ]


class VirtualMachineFilterForm(
    NautobotFilterForm,
    LocatableModelFilterFormMixin,
    TenancyFilterForm,
    StatusModelFilterFormMixin,
    RoleModelFilterFormMixin,
    LocalContextFilterForm,
):
    model = VirtualMachine
    field_order = [
        "q",
        "cluster_group",
        "cluster_type",
        "cluster_id",
        "status",
        "role",
        "location",
        "tenant_group",
        "tenant",
        "platform",
        "mac_address",
    ]
    q = forms.CharField(required=False, label="Search")
    cluster_group = DynamicModelMultipleChoiceField(
        queryset=ClusterGroup.objects.all(),
        to_field_name="slug",
        required=False,
        null_option="None",
    )
    cluster_type = DynamicModelMultipleChoiceField(
        queryset=ClusterType.objects.all(),
        to_field_name="slug",
        required=False,
        null_option="None",
    )
    cluster_id = DynamicModelMultipleChoiceField(queryset=Cluster.objects.all(), required=False, label="Cluster")
    platform = DynamicModelMultipleChoiceField(
        queryset=Platform.objects.all(),
        to_field_name="slug",
        required=False,
        null_option="None",
    )
    mac_address = forms.CharField(required=False, label="MAC address")
    has_primary_ip = forms.NullBooleanField(
        required=False,
        label="Has a primary IP",
        widget=StaticSelect2(choices=BOOLEAN_WITH_BLANK_CHOICES),
    )
    tag = TagFilterField(model)


#
# VM interfaces
#


class VMInterfaceForm(NautobotModelForm, InterfaceCommonForm):
    parent_interface = DynamicModelChoiceField(
        queryset=VMInterface.objects.all(),
        required=False,
        label="Parent interface",
        help_text="Assigned parent VMinterface",
    )
    bridge = DynamicModelChoiceField(
        queryset=VMInterface.objects.all(),
        required=False,
        label="Bridge interface",
        help_text="Assigned bridge VMinterface",
    )
    untagged_vlan = DynamicModelChoiceField(
        queryset=VLAN.objects.all(),
        required=False,
        label="Untagged VLAN",
        brief_mode=False,
        query_params={
            "location_id": "null",
        },
    )
    tagged_vlans = DynamicModelMultipleChoiceField(
        queryset=VLAN.objects.all(),
        required=False,
        label="Tagged VLANs",
        brief_mode=False,
        query_params={
            "location_id": "null",
        },
    )
    ip_addresses = DynamicModelMultipleChoiceField(
        queryset=IPAddress.objects.all(),
        required=False,
        label="IP Addresses",
        brief_mode=False,
    )

    class Meta:
        model = VMInterface
        fields = [
            "virtual_machine",
            "name",
            "enabled",
            "parent_interface",
            "bridge",
            "mac_address",
            "ip_addresses",
            "mtu",
            "description",
            "mode",
            "tags",
            "untagged_vlan",
            "tagged_vlans",
            "status",
        ]
        widgets = {"virtual_machine": forms.HiddenInput(), "mode": StaticSelect2()}
        labels = {
            "mode": "802.1Q Mode",
        }
        help_texts = {
            "mode": INTERFACE_MODE_HELP_TEXT,
        }

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        vm_id = self.initial.get("virtual_machine") or self.data.get("virtual_machine")

        # Restrict parent interface assignment by VM
        self.fields["parent_interface"].widget.add_query_param("virtual_machine_id", vm_id)
        self.fields["bridge"].widget.add_query_param("virtual_machine_id", vm_id)

        virtual_machine = VirtualMachine.objects.get(
            pk=self.initial.get("virtual_machine") or self.data.get("virtual_machine")
        )

        # Add current location to VLANs query params
        location = virtual_machine.location
        if location:
            self.fields["untagged_vlan"].widget.add_query_param("location_id", location.pk)
            self.fields["tagged_vlans"].widget.add_query_param("location_id", location.pk)


class VMInterfaceCreateForm(BootstrapMixin, InterfaceCommonForm):
    virtual_machine = DynamicModelChoiceField(queryset=VirtualMachine.objects.all())
    name_pattern = ExpandableNameField(label="Name")
    enabled = forms.BooleanField(required=False, initial=True)
    parent_interface = DynamicModelChoiceField(
        queryset=VMInterface.objects.all(),
        required=False,
        query_params={
            "virtual_machine_id": "$virtual_machine",
        },
        help_text="Assigned parent VMinterface",
    )
    bridge = DynamicModelChoiceField(
        queryset=VMInterface.objects.all(),
        required=False,
        query_params={
            "virtual_machine_id": "$virtual_machine",
        },
        help_text="Assigned bridge VMinterface",
    )
    mtu = forms.IntegerField(
        required=False,
        min_value=INTERFACE_MTU_MIN,
        max_value=INTERFACE_MTU_MAX,
        label="MTU",
    )
    mac_address = forms.CharField(required=False, label="MAC Address")
    description = forms.CharField(max_length=100, required=False)
    mode = forms.ChoiceField(
        choices=add_blank_choice(InterfaceModeChoices),
        required=False,
        widget=StaticSelect2(),
    )
    untagged_vlan = DynamicModelChoiceField(
        queryset=VLAN.objects.all(),
        required=False,
        brief_mode=False,
        query_params={
            "location_id": "null",
        },
    )
    tagged_vlans = DynamicModelMultipleChoiceField(
        queryset=VLAN.objects.all(),
        required=False,
        brief_mode=False,
        query_params={
            "location_id": "null",
        },
    )
    status = DynamicModelChoiceField(
        queryset=Status.objects.all(),
        query_params={
            "content_types": VMInterface._meta.label_lower,
        },
    )

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        vm_id = self.initial.get("virtual_machine") or self.data.get("virtual_machine")

        # Restrict parent interface assignment by VM
        self.fields["parent_interface"].widget.add_query_param("virtual_machine_id", vm_id)
        self.fields["bridge"].widget.add_query_param("virtual_machine_id", vm_id)

        virtual_machine = VirtualMachine.objects.get(
            pk=self.initial.get("virtual_machine") or self.data.get("virtual_machine")
        )

        # Add current location to VLANs query params
        location = virtual_machine.location
        if location:
            self.fields["untagged_vlan"].widget.add_query_param("location_id", location.pk)
            self.fields["tagged_vlans"].widget.add_query_param("location_id", location.pk)


class VMInterfaceCSVForm(CustomFieldModelCSVForm, StatusModelCSVFormMixin):
    virtual_machine = CSVModelChoiceField(queryset=VirtualMachine.objects.all(), to_field_name="name")
    parent_interface = CSVModelChoiceField(
        queryset=VMInterface.objects.all(), required=False, to_field_name="name", help_text="Parent interface"
    )
    bridge = CSVModelChoiceField(
        queryset=VMInterface.objects.all(), required=False, to_field_name="name", help_text="Bridge interface"
    )
    mode = CSVChoiceField(
        choices=InterfaceModeChoices,
        required=False,
        help_text="IEEE 802.1Q operational mode (for L2 interfaces)",
    )

    class Meta:
        model = VMInterface
        fields = VMInterface.csv_headers

    def clean_enabled(self):
        # Make sure enabled is True when it's not included in the uploaded data
        if "enabled" not in self.data:
            return True
        else:
            return self.cleaned_data["enabled"]


class VMInterfaceBulkEditForm(TagsBulkEditFormMixin, StatusModelBulkEditFormMixin, NautobotBulkEditForm):
    pk = forms.ModelMultipleChoiceField(queryset=VMInterface.objects.all(), widget=forms.MultipleHiddenInput())
    virtual_machine = forms.ModelChoiceField(
        queryset=VirtualMachine.objects.all(),
        required=False,
        disabled=True,
        widget=forms.HiddenInput(),
    )
    parent_interface = DynamicModelChoiceField(
        queryset=VMInterface.objects.all(), required=False, display_field="display_name"
    )
    bridge = DynamicModelChoiceField(
        queryset=VMInterface.objects.all(),
        required=False,
    )
    enabled = forms.NullBooleanField(required=False, widget=BulkEditNullBooleanSelect())
    mtu = forms.IntegerField(
        required=False,
        min_value=INTERFACE_MTU_MIN,
        max_value=INTERFACE_MTU_MAX,
        label="MTU",
    )
    description = forms.CharField(max_length=100, required=False)
    mode = forms.ChoiceField(
        choices=add_blank_choice(InterfaceModeChoices),
        required=False,
        widget=StaticSelect2(),
    )
    untagged_vlan = DynamicModelChoiceField(
        queryset=VLAN.objects.all(),
        required=False,
        brief_mode=False,
        query_params={
            "location_id": "null",
        },
    )
    tagged_vlans = DynamicModelMultipleChoiceField(
        queryset=VLAN.objects.all(),
        required=False,
        brief_mode=False,
        query_params={
            "location_id": "null",
        },
    )

    class Meta:
        nullable_fields = [
            "parent_interface",
            "bridge",
            "mtu",
            "description",
        ]

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        vm_id = self.initial.get("virtual_machine") or self.data.get("virtual_machine")

        # Restrict parent/bridge interface assignment by VM
        self.fields["parent_interface"].widget.add_query_param("virtual_machine_id", vm_id)
        self.fields["bridge"].widget.add_query_param("virtual_machine_id", vm_id)

        # Limit available VLANs based on the parent VirtualMachine
        if "virtual_machine" in self.initial:
            parent_obj = VirtualMachine.objects.filter(pk=self.initial["virtual_machine"]).first()

            location = getattr(parent_obj.cluster, "location", None)
            if location is not None:
                # Add current location to VLANs query params
                self.fields["untagged_vlan"].widget.add_query_param("location_id", location.pk)
                self.fields["tagged_vlans"].widget.add_query_param("location_id", location.pk)

        self.fields["parent_interface"].choices = ()
        self.fields["parent_interface"].widget.attrs["disabled"] = True
        self.fields["bridge"].choices = ()
        self.fields["bridge"].widget.attrs["disabled"] = True


class VMInterfaceBulkRenameForm(BulkRenameForm):
    pk = forms.ModelMultipleChoiceField(queryset=VMInterface.objects.all(), widget=forms.MultipleHiddenInput())


class VMInterfaceFilterForm(NautobotFilterForm, StatusModelFilterFormMixin):
    model = VMInterface
    cluster_id = DynamicModelMultipleChoiceField(queryset=Cluster.objects.all(), required=False, label="Cluster")
    virtual_machine_id = DynamicModelMultipleChoiceField(
        queryset=VirtualMachine.objects.all(),
        required=False,
        label="Virtual machine",
        query_params={"cluster_id": "$cluster_id"},
    )
    enabled = forms.NullBooleanField(required=False, widget=StaticSelect2(choices=BOOLEAN_WITH_BLANK_CHOICES))
    tag = TagFilterField(model)


#
# Bulk VirtualMachine component creation
#


class VirtualMachineBulkAddComponentForm(CustomFieldModelBulkEditFormMixin, BootstrapMixin, forms.Form):
    pk = forms.ModelMultipleChoiceField(queryset=VirtualMachine.objects.all(), widget=forms.MultipleHiddenInput())
    name_pattern = ExpandableNameField(label="Name")

    class Meta:
        nullable_fields = []


class VMInterfaceBulkCreateForm(
    form_from_model(VMInterface, ["enabled", "mtu", "description", "mode", "tags"]),
    VirtualMachineBulkAddComponentForm,
):
    status = DynamicModelChoiceField(
        queryset=Status.objects.all(),
        query_params={"content_types": VMInterface._meta.label_lower},
    )

    field_order = (
        "name_pattern",
        "status",
        "enabled",
        "mtu",
        "description",
        "mode",
        "tags",
    )
