from itertools import count, groupby
import json
from urllib.parse import quote_plus, unquote_plus

from django.apps import apps
from django.core.exceptions import FieldDoesNotExist
from django.core.serializers import serialize
from django.utils.tree import Node
from taggit.managers import _TaggableManager

from nautobot.core.models.constants import NATURAL_KEY_SLUG_SEPARATOR


def array_to_string(array):
    """
    Generate an efficient, human-friendly string from a set of integers. Intended for use with ArrayField.
    For example:
        [0, 1, 2, 10, 14, 15, 16] => "0-2, 10, 14-16"
    """
    group = (list(x) for _, x in groupby(sorted(array), lambda x, c=count(): next(c) - x))
    return ", ".join("-".join(map(str, (g[0], g[-1])[: len(g)])) for g in group)


def get_all_concrete_models(base_class):
    """Get a list of all non-abstract models that inherit from the given base_class."""
    models = []
    for appconfig in apps.get_app_configs():
        for model in appconfig.get_models():
            if issubclass(model, base_class) and not model._meta.abstract:
                models.append(model)
    return sorted(models, key=lambda model: (model._meta.app_label, model._meta.model_name))


def is_taggable(obj):
    """
    Return True if the instance can have Tags assigned to it; False otherwise.
    """
    if hasattr(obj, "tags"):
        if issubclass(obj.tags.__class__, _TaggableManager):
            return True
    return False


def pretty_print_query(query):
    """
    Given a `Q` object, display it in a more human-readable format.

    Args:
        query (Q): Query to display.

    Returns:
        str: Pretty-printed query logic

    Example:
        >>> print(pretty_print_query(Q))
        (
          site__slug='ams01' OR site__slug='bkk01' OR (
            site__slug='can01' AND status__slug='active'
          ) OR (
            site__slug='del01' AND (
              NOT (site__slug='del01' AND status__slug='decommissioning')
            )
          )
        )
    """

    def pretty_str(self, node=None, depth=0):
        """Improvement to default `Node.__str__` with a more human-readable style."""
        template = f"(\n{'  ' * (depth + 1)}"
        if self.negated:
            template += "NOT (%s)"
        else:
            template += "%s"
        template += f"\n{'  ' * depth})"
        children = []

        # If we don't have a node, we are the node!
        if node is None:
            node = self

        # Iterate over children. They will be either a Q object (a Node subclass) or a 2-tuple.
        for child in node.children:
            # Trust that we can stringify the child if it is a Node instance.
            if isinstance(child, Node):
                children.append(pretty_str(child, depth=depth + 1))
            # If a 2-tuple, stringify to key=value
            else:
                key, value = child
                children.append(f"{key}={value!r}")

        return template % (f" {self.connector} ".join(children))

    # Use pretty_str() as the string generator vs. just stringify the `Q` object.
    return pretty_str(query)


def serialize_object(obj, extra=None, exclude=None):
    """
    Return a generic JSON representation of an object using Django's built-in serializer. (This is used for things like
    change logging, not the REST API.) Optionally include a dictionary to supplement the object data. A list of keys
    can be provided to exclude them from the returned dictionary. Private fields (prefaced with an underscore) are
    implicitly excluded.
    """
    json_str = serialize("json", [obj])
    data = json.loads(json_str)[0]["fields"]

    # Include custom_field_data as "custom_fields"
    if hasattr(obj, "_custom_field_data"):
        data["custom_fields"] = data.pop("_custom_field_data")

    # Include any tags. Check for tags cached on the instance; fall back to using the manager.
    if is_taggable(obj):
        tags = getattr(obj, "_tags", []) or obj.tags.all()
        data["tags"] = [tag.name for tag in tags]

    # Append any extra data
    if extra is not None:
        data.update(extra)

    # Copy keys to list to avoid 'dictionary changed size during iteration' exception
    for key in list(data):
        # Private fields shouldn't be logged in the object change
        if isinstance(key, str) and key.startswith("_"):
            data.pop(key)

        # Explicitly excluded keys
        if isinstance(exclude, (list, tuple)) and key in exclude:
            data.pop(key)

    return data


def serialize_object_v2(obj):
    """
    Return a JSON serialized representation of an object using obj's serializer.
    """
    from nautobot.core.api.exceptions import SerializerNotFound
    from nautobot.core.api.utils import get_serializer_for_model

    # Try serializing obj(model instance) using its API Serializer
    try:
        serializer_class = get_serializer_for_model(obj.__class__)
        data = serializer_class(obj, context={"request": None}).data
    except SerializerNotFound:
        # Fall back to generic JSON representation of obj
        data = serialize_object(obj)

    return data


def find_models_with_matching_fields(app_models, field_names, field_attributes=None):
    """
    Find all models that have fields with the specified names, and return them grouped by app.

    Args:
        app_models: A list of model classes to search through.
        field_names: A list of names of fields that must be present in order for the model to be considered
        field_attributes: Optional dictionary of attributes to filter the fields by.

    Return:
        A dictionary where the keys are app labels and the values are sets of model names.
    """
    registry_items = {}
    field_attributes = field_attributes or {}
    for model_class in app_models:
        app_label, model_name = model_class._meta.label_lower.split(".")
        for field_name in field_names:
            try:
                field = model_class._meta.get_field(field_name)
                if all((getattr(field, item, None) == value for item, value in field_attributes.items())):
                    registry_items.setdefault(app_label, set()).add(model_name)
            except FieldDoesNotExist:
                pass
    registry_items = {key: sorted(value) for key, value in registry_items.items()}
    return registry_items


def construct_natural_key_slug(values):
    """
    Convert the given list of natural key values to a single URL-path-usable string.

    - Non-URL-safe characters are percent-encoded.
    - Null (`None`) values are percent-encoded as a literal null character `%00`.

    Reversible by `deconstruct_natural_key_slug()`.
    """
    values = [str(value) if value is not None else "\0" for value in values]
    # . and : are generally "safe enough" to use in URL parameters, and are common in some natural key fields,
    # so we don't quote them by default (although `deconstruct_natural_key_slug` will work just fine if you do!)
    # / is a bit trickier to handle in URL paths, so for now we *do* quote it, even though it appears in IPAddress, etc.
    values = NATURAL_KEY_SLUG_SEPARATOR.join(quote_plus(value, safe=".:") for value in values)
    return values


def deconstruct_natural_key_slug(slug):
    """
    Convert the given natural key slug string back to a list of distinct values.

    - Percent-encoded characters are converted back to their raw values
    - Single literal null characters `%00` are converted back to a Python `None`.

    Inverse operation of `construct_natural_key_slug()`.
    """
    values = [unquote_plus(value) for value in slug.split(NATURAL_KEY_SLUG_SEPARATOR)]
    values = [value if value != "\0" else None for value in values]
    return values
