import uuid

from django.db import models
from django.utils.functional import classproperty

from nautobot.core.models.managers import BaseManager
from nautobot.core.models.querysets import RestrictedQuerySet
from nautobot.core.models.utils import construct_natural_key_slug


class BaseModel(models.Model):
    """
    Base model class that all models should inherit from.

    This abstract base provides globally common fields and functionality.

    Here we define the primary key to be a UUID field and set its default to
    automatically generate a random UUID value. Note however, this does not
    operate in the same way as a traditional auto incrementing field for which
    the value is issued by the database upon initial insert. In the case of
    the UUID field, Django creates the value upon object instantiation. This
    means the canonical pattern in Django of checking `self.pk is None` to tell
    if an object has been created in the actual database does not work because
    the object will always have the value populated prior to being saved to the
    database for the first time. An alternate pattern of checking `not self.present_in_database`
    can be used for the same purpose in most cases.
    """

    id = models.UUIDField(primary_key=True, default=uuid.uuid4, unique=True, editable=False)

    objects = BaseManager.from_queryset(RestrictedQuerySet)()

    @property
    def present_in_database(self):
        """
        True if the record exists in the database, False if it does not.
        """
        return not self._state.adding

    class Meta:
        abstract = True

    def validated_save(self):
        """
        Perform model validation during instance save.

        This is a convenience method that first calls `self.full_clean()` and then `self.save()`
        which in effect enforces model validation prior to saving the instance, without having
        to manually make these calls seperately. This is a slight departure from Django norms,
        but is intended to offer an optional, simplified interface for performing this common
        workflow. The intended use is for user defined Jobs and scripts run via the `nautobot-server nbshell`
        command.
        """
        self.full_clean()
        self.save()

    def natural_key(self) -> list:
        """
        Smarter default implementation of natural key construction.

        1. Handles nullable foreign keys (https://github.com/wq/django-natural-keys/issues/18)
        2. Handles variadic natural-keys (e.g. Location model - [name, parent__name, parent__parent__name, ...].)
        """
        vals = []
        for lookups in [lookup.split("__") for lookup in self.natural_key_field_lookups]:
            val = self
            for lookup in lookups:
                val = getattr(val, lookup)
                if val is None:
                    break
            vals.append(val)
        # Strip trailing Nones from vals
        while vals and vals[-1] is None:
            vals.pop()
        return vals

    @property
    def natural_key_slug(self) -> str:
        """
        Automatic "slug" string derived from this model's natural key, suitable for use in URLs etc.

        A less naïve implementation than django-natural-keys provides by default, based around URL percent-encoding.
        """
        return construct_natural_key_slug(self.natural_key())

    @classproperty  # https://github.com/PyCQA/pylint-django/issues/240
    def natural_key_field_lookups(cls):  # pylint: disable=no-self-argument
        """
        List of lookups (possibly including nested lookups for related models) that make up this model's natural key.

        BaseModel provides a "smart" implementation that tries to determine this automatically,
        but you can also explicitly set `natural_key_field_names` on a given model subclass if desired.

        This property is based on a consolidation of `django-natural-keys` `ForeignKeyModel.get_natural_key_info()`,
        `ForeignKeyModel.get_natural_key_def()`, and `ForeignKeyModel.get_natural_key_fields()`.

        Unlike `get_natural_key_def()`, this doesn't auto-exclude all AutoField and BigAutoField fields,
        but instead explicitly discounts the `id` field (only) as a candidate.
        """
        # First, figure out which local fields comprise the natural key:
        natural_key_field_names = []
        if hasattr(cls, "natural_key_field_names"):
            natural_key_field_names = cls.natural_key_field_names
        else:
            # Does this model have any new-style UniqueConstraints? If so, pick the first one
            for constraint in cls._meta.constraints:
                if isinstance(constraint, models.UniqueConstraint):
                    natural_key_field_names = constraint.fields
                    break
            else:
                # Else, does this model have any old-style unique_together? If so, pick the first one.
                if cls._meta.unique_together:
                    natural_key_field_names = cls._meta.unique_together[0]
                else:
                    # Else, do we have any individual unique=True fields? If so, pick the first one.
                    unique_fields = [field for field in cls._meta.fields if field.unique and field.name != "id"]
                    if unique_fields:
                        natural_key_field_names = (unique_fields[0].name,)

        if not natural_key_field_names:
            raise AttributeError(
                f"Unable to identify an intrinsic natural-key definition for {cls.__name__}. "
                "If there isn't at least one UniqueConstraint, unique_together, or field with unique=True, "
                "you probably need to explicitly declare the 'natural_key_field_names' for this model, "
                "or potentially override the default 'natural_key_field_lookups' implementation for this model."
            )

        # Next, for any natural key fields that have related models, get the natural key for the related model if known
        natural_key_field_lookups = []
        for field_name in natural_key_field_names:
            field = cls._meta.get_field(field_name)
            if getattr(field, "remote_field", None) is None:
                # Not a related field, so the field name is the field lookup
                natural_key_field_lookups.append(field_name)
                continue

            related_model = field.remote_field.model
            related_natural_key_field_lookups = None
            if hasattr(related_model, "natural_key_field_lookups"):
                # TODO: generic handling for self-referential case, as seen in Location
                related_natural_key_field_lookups = related_model.natural_key_field_lookups
            else:
                # Related model isn't a Nautobot model and so doesn't have a `natural_key_field_lookups`.
                # The common case we've encountered so far is the contenttypes.ContentType model:
                if related_model._meta.app_label == "contenttypes" and related_model._meta.model_name == "contenttype":
                    related_natural_key_field_lookups = ["app_label", "model"]
                # Additional special cases can be added here

            if not related_natural_key_field_lookups:
                raise AttributeError(
                    f"Unable to determine the related natural-key fields for {related_model.__name__} "
                    f"(as referenced from {cls.__name__}.{field_name}). If the related model is a non-Nautobot "
                    "model (such as ContentType) then it may be appropriate to add special-case handling for this "
                    "model in BaseModel.natural_key_field_lookups; alternately you may be able to solve this for "
                    f"a single special case by explicitly defining {cls.__name__}.natural_key_field_lookups."
                )

            for field_lookup in related_natural_key_field_lookups:
                natural_key_field_lookups.append(f"{field_name}__{field_lookup}")

        return natural_key_field_lookups

    @classmethod
    def natural_key_args_to_kwargs(cls, args):
        """
        Helper function to map a list of natural key field values to actual kwargs suitable for lookup and filtering.

        Based on `django-natural-keys` `NaturalKeyQuerySet.natural_key_kwargs()` method.
        """
        args = list(args)
        natural_key_field_lookups = cls.natural_key_field_lookups
        # Because `natural_key` strips trailing `None` from the natural key to handle the variadic-natural-key case,
        # we may need to add trailing `None` back on to make the number of args match back up.
        while len(args) < len(natural_key_field_lookups):
            args.append(None)
        # However, if we have *too many* args, that's just incorrect usage:
        if len(args) > len(natural_key_field_lookups):
            raise ValueError(
                f"Wrong number of natural-key args for {cls.__name__}.natural_key_args_to_kwargs() -- "
                f"expected no more than {len(natural_key_field_lookups)} but got {len(args)}."
            )
        return dict(zip(natural_key_field_lookups, args))
