import abc
import gevent.event

from mxcubecore.BaseHardwareObjects import HardwareObject


class AbstractCharacterisation(HardwareObject):
    __metaclass__ = abc.ABCMeta

    def __init__(self, name):
        HardwareObject.__init__(self, name)
        self.processing_done_event = gevent.event.Event()

    @abc.abstractmethod
    def characterise(self, _input):
        """
        Args:
            input (object) Characterisation input object

        Returns:
            (str) The Characterisation result
        """
        pass

    @abc.abstractmethod
    def get_html_report(self, output):
        """
        Args:
            output (object) Characterisation output object

        Returns:
            (str) The path to the html result report generated by the characterisation
            software
        """
        pass

    @abc.abstractmethod
    def input_from_params(self, ref_parameters, char_params, path_str):
        """
        Args:
            ref_parameters: (A named tuple or object with following fields):
              'id',
              'prefix',
              'run_number',
              'template',
              'first_image',
              'num_images',
              'osc_start',
              'osc_range',
              'overlap',
              'exp_time',
              'num_passes',
              'comments',
              'path',
              'centred_positions',
              'energy',
              'resolution',
              'transmission',
              'shutterless',
              'inverse_beam',
              'screening_id'

        Args:
            char_params: (A named tuple or object with following fields):
              # Optimisation parameters
              'aimed_resolution'
              'aimed_multiplicity'
              'aimed_i_sigma'
              'aimed_completness'
              'strategy_complexity'
              'induce_burn'
              'use_permitted_rotation'
              'permitted_phi_start'
              'permitted_phi_end'

              # Crystal
              'max_crystal_vdim'
              'min_crystal_vdim'
              'max_crystal_vphi'
              'min_crystal_vphi'
              'space_group'

              # Characterisation type
              'use_min_dose'
              'use_min_time'
              'min_dose'
              'min_time'
              'account_rad_damage'
              'not_use_low_res'
              'auto_res'
              'opt_sad'
              'sad_res'
              'determine_rad_params'

              # Radiation damage model
              'rad_suscept'
              'beta'
              'sigma'

        Args:
          path_str (Template string representing path to each image)

        Returns:
            Input for characterisation software
        """
        pass

    @abc.abstractmethod
    def dc_from_output(self, output):
        """
        Create a data collection from characterisation result

        Args:
            output (object) Characterisation result object

        Returns:
            (queue_model_objects.DataCollection)
        """
        pass

    @abc.abstractmethod
    def get_default_characterisation_parameters(self, defualt_input_file):
        """
        Args:
            defualt_input_file (str): Path to file containing default input

        Returns:
            (queue_model_objects.CharacterisationsParameters): object with default
            parameters.
        """
        pass

    def prepare_input(self, _input):
        """
        Method called by characterise before characterisation starts, can be used
        to manipulate edna_input object before characterisation starts.

        Example: to set a site specific output directory

        Args:
           _input (object) Characterisation input object
        """
        pass

    def is_running(self):
        """
        Returns:
            (bool) True if process is running otherwise False
        """
        return not self.processing_done_event.is_set()
