from ._Utils import _Utils
from ._Globals import anyAuthenticatedSession

class Deployment:
  _requirementsTxt = None
  _deployName = None
  _deployFunc = None
  _deployArgs = None
  _pythonVersion = '3.9' # Default version

  STATUS_DEPLOYABLE = 'Ready to Deploy'
  STATUS_NOT_DEPLOYABLE = 'Not Ready to Deploy'

  ALLOWED_PY_VERSIONS = ['3.6', '3.7', '3.8', '3.9']
  CODE_STYLE = "font-family: monospace; font-size: 0.95em; font-weight: medium; color: #714488;"

  MAX_REQUIREMENTS_TXT = 20_000

  # Keep these kwargs in sync with __init__.py/.Deployment(...)
  def __init__(self, name = None, deploy_function = None, deploy_args = {}, python_version = None):
    if name != None: self.set_name(name)
    if deploy_function != None: self.set_deploy_function(deploy_function, deploy_args or {})
    if deploy_args != None: self.set_deploy_arguments(deploy_args)
    if python_version != None: self.set_python_version(python_version)

  def _repr_markdown_(self):
    return self._describe()

  def set_name(self, name):
    import re
    if not re.match('^[a-zA-Z0-9_]+$', name):
      raise Exception("Deployment names should be alphanumeric with underscores.")
    self._deployName = name
    return self

  def set_python_version(self, version):
    if version not in self.ALLOWED_PY_VERSIONS:
      return self._selfError(f'Python version should be one of {self.ALLOWED_PY_VERSIONS}.')
    self._pythonVersion = version
    return self

  def set_requirements_txt(self):
    from ipywidgets import FileUpload
    from IPython.display import display, clear_output
    import re
    upload = FileUpload(accept='.txt', multiple=False)
    display(upload)
    def onUploadChange(change):
        clear_output(wait=True)
        for k,v in change['new'].items():
          content = v['content'].decode('utf-8')
          if len(content) < self.MAX_REQUIREMENTS_TXT:
            self._requirementsTxt = content
            session = anyAuthenticatedSession()
            if session:
              session._getJson("jupyter/v1/deployments/prep_environment", {
                "environment": {
                  "requirementsTxt": self._requirementsTxt,
                  "pythonVersion": self._pythonVersion
                }
              })
          else:
            _Utils._printError("The requirements.txt file is too large.")
        _Utils._printMk(self._describe())
    upload.observe(onUploadChange, names=['value'])
    return None

  def set_deploy_function(self, func, args = {}):
    self._deployFunc = func
    self._deployArgs = args
    return self

  def set_deploy_arguments(self, args):
    if type(args) is not dict:
      return self._selfError('Args should be a dictionary.')
    self._deployArgs = args
    return self

  def set_deploy_argument(self, name, val):
    self._deployArgs[name] = val
    return self
  
  def remove_deploy_argument(self, name):
    self._deployArgs.pop(name, None)
    return self

  def _deploy(self, mbMain):
    status = self._getStatus()
    if status['status'] != self.STATUS_DEPLOYABLE:
      _Utils._printError('Unable to deploy.')
      return self
    resp = mbMain._getJsonOrPrintError("jupyter/v1/deployments/create", {
      "deployment": {
        "name": self._deployName,
        "pyState": {
          **self._getFuncProps(self._deployFunc, self._deployArgs),
          "requirementsTxt": self._requirementsTxt,
          "pythonVersion": self._pythonVersion
        }}})
    if resp and "deployOverviewUrl" in resp:
      if "message" in resp: _Utils._printMk(resp["message"])
      _Utils._printMk(f'<a href="{resp["deployOverviewUrl"]}" target="_blank">View status and integration options.</a>')
    else:
      _Utils._printMk(f'Error while deploying. ({resp})')
    return None

  def _selfError(self, txt):
    _Utils._printError(txt)
    return None

  def _describe(self):
    nonStr = '(None)'
    def codeWrap(txt):
      if txt == nonStr: return nonStr
      return self._wrapStyle(txt, self.CODE_STYLE)

    status = self._getStatus()
    statusWithStyle = self._wrapStyle(status["status"], status["style"])
    md = ""
    if self._deployName != None: md += f'**{self._deployName}**: '
    md += f'{statusWithStyle}\n\n'
    statusList = "\n".join([f'* {n}' for n in status["notes"]])
    if len(statusList) > 0: md += statusList + "\n\n"

    md += "| Property | Value |\n" + "|:-|:-|\n"
    funcProps = self._getFuncProps(self._deployFunc, self._deployArgs)
    funcSig = nonStr
    argsDesc = nonStr
    if funcProps != None:
      if 'name' in funcProps and 'argNames' in funcProps:
        funcSig = f"{funcProps['name']}({', '.join(funcProps['argNames'])})"
      if 'deployVarsDesc' in funcProps:
        argsDesc = "<br/>".join([f'{k}: {v}' for k,v in funcProps['deployVarsDesc'].items()])
    md += f"| Function | {codeWrap(funcSig)} |\n"
    md += f"| Arguments | {codeWrap(argsDesc)} |\n"
    md += f"| Python Version | {codeWrap(self._pythonVersion or nonStr)} |\n"

    deps = nonStr
    if self._requirementsTxt and len(self._requirementsTxt) > 0:
      depsList = self._requirementsTxt.splitlines()
      maxDepsShown = 7
      if len(depsList) > maxDepsShown:
        deps = "<br/>".join([d for d in depsList[:maxDepsShown]])
        numLeft = len(depsList) - maxDepsShown
        deps += f'<br/><span style="font-style: italic;">...and {numLeft} more.</span>'
      else:
        deps = "<br/>".join([d for d in depsList])
    md += f"| requirements.txt | {codeWrap(deps)} |\n"
    return md

  def _getFuncProps(self, func, args):
    import inspect
    errors = []
    props = {}
    if not callable(func):
      errors.append('The deploy_function parameter does not appear to be a function.')
    else:
      props['name'] = func.__name__
      props['source'] = inspect.getsource(func)
      props['argNames'] = list(func.__code__.co_varnames[:func.__code__.co_argcount] or [])
      argCheck = self._checkArgs(props['argNames'], args)
      if 'error' in argCheck:
        errors.append(argCheck['error'])
    if 'argNames' in props and 'input_row' not in props['argNames']:
      errors.append("One of the function's arguments should be 'input_row'.")
    props['deployVars'] = self._pickleValues(args)
    props['deployVarsDesc'] = self._strValues(args)
    if len(errors) > 0: props['errors'] = errors
    return props

  def _getStatus(self):
    notes = []
    if not self._deployName:
      cmd = self._wrapStyle("dep.set_name('name')", self.CODE_STYLE)
      notes.append(f'Run {cmd} to specify the deployment\'s name.')
    if not self._deployFunc:
      cmd = self._wrapStyle("dep.set_deploy_function(func, args = {\"arg1\": value1, ...})", self.CODE_STYLE)
      notes.append(f'Run {cmd} to specify the deployment\'s runtime.')
    else:
      funcProps = self._getFuncProps(self._deployFunc, self._deployArgs)
      if 'errors' in funcProps: notes.extend(funcProps['errors'])
    if not self._pythonVersion:
      cmd = self._wrapStyle("dep.set_python_version('version')", self.CODE_STYLE)
      notes.append(f'Run {cmd} to set the python version to one of {self.ALLOWED_PY_VERSIONS}.')
    # if not self._requirementsTxt:
    #   cmd = self._wrapStyle("dep.set_requirements_txt()", self.CODE_STYLE)
    #   notes.append(f'Run {cmd} to upload required dependencies.')
    if len(notes) > 0:
      return { "status": self.STATUS_NOT_DEPLOYABLE, "style": "color:gray; font-weight: bold;", "notes": notes }
    else:
      cmd = self._wrapStyle("mb.deploy(dep)", self.CODE_STYLE)
      notes.append(f'Run {cmd} to deploy this function to Modelbit.')
      return { "status": self.STATUS_DEPLOYABLE, "style": "color:green; font-weight: bold;", "notes": notes }

  def _checkArgs(self, argNames, args):
    if type(args) is not dict:
      return { "error": 'Args should be a dictionary.' }
    if "input_row" in args:
      return { "error": 'Args should not include "input_row". It will be supplied automatically at runtime.' }
    for k, v in args.items():
      if type(k) is not str:
        return { "error": f'Arg key "{str(k)}" should be a string.' }
    for n in argNames:
      if n != 'input_row' and n not in args:
        return { "error": f'Function argument "{str(n)}" is missing from the args dictionary.' }
    return {}

  def _wrapStyle(self, text, style):
    return f'<span style="{style}">{text}</span>'

  def _pickleValues(self, args):
    import pickle, codecs
    newDict = {}
    for k, v in args.items():
      newDict[k] = codecs.encode(pickle.dumps(v), "base64").decode()
    return newDict

  def _strValues(self, args):
    newDict = {}
    for k, v in args.items():
      newDict[k] = str(v)
    return newDict
