import sys
from optparse import OptionParser
from typing import List, Optional
from pathlib import Path

import editor
from gitlab import GitlabAuthenticationError, GitlabUpdateError

from mkmr.api import API
from mkmr.config import Config
from mkmr.utils import find_cache, init_repo, strtobool

from . import __version__


def main():
    parser = OptionParser(version=__version__)
    parser.add_option(
        "--token", dest="token", action="store", type="string", help="GitLab Personal Access Token"
    )
    parser.add_option(
        "-c",
        "--config",
        dest="config",
        action="store",
        type="string",
        default=None,
        help="Full path to configuration file",
    )
    parser.add_option(
        "-n",
        "--dry-run",
        dest="dry_run",
        action="store_true",
        default=False,
        help="show which merge requests mgmr would try to merge",
    )
    parser.add_option(
        "--timeout",
        dest="timeout",
        action="store",
        default=None,
        type="int",
        help="Set timeout for making calls to the gitlab API",
    )
    parser.add_option(
        "--overwrite",
        dest="overwrite",
        action="store_true",
        default=False,
        help="if --token is passed, overwrite private_token in configuration file",
    )
    parser.add_option(
        "--remote",
        dest="remote",
        action="store",
        type="string",
        default="upstream",
        help="which remote from which to operate on",
    )
    parser.add_option(
        "-y",
        "--yes",
        dest="yes",
        action="store_true",
        default=False,
        help="Assume yes to all prompts",
    )

    (options, args) = parser.parse_args(sys.argv)

    if len(args) < 2:
        print("no merge request given")
        sys.exit(1)

    if options.token is None and options.overwrite is True:
        print("--overwrite was passed, but no --token was passed along with it")
        sys.exit(1)

    # Annotate some variables
    #
    # A string that corresponds to either an integer that corresponds to the internal-id
    # of the merge request, or the name of a branch, the later can have a branch file that
    # contains the internal-id of the merge request.
    mrnum: str
    iid: int
    namespace: Path

    mrnum = args[1]

    # Initialize our repo object based on the local repo we have
    repo = init_repo()
    if repo is None:
        sys.exit(1)

    remote = API(repo, options.remote)

    try:
        config = Config(options, remote.host)
    except ValueError as e:
        print(e)
        sys.exit(1)

    gl = config.get_gitlab()

    namespace = (
        Path()
        / remote.host.replace("https://", "").replace("/", ".")
        / remote.user
        / remote.project
    )
    try:
        cachepath = find_cache()
        # This path should be, taking alpine/aports from gitlab.alpinelinux.org as example:
        # $XDG_CACHE_HOME/mkmr/gitlab.alpinelinux.org/alpine/aports/branches/$source_branch
        cachepath = cachepath / namespace / "branches" / mrnum
        mrnum = (
            cachepath.read_text()
        )  # Read the file, it should contain the internal-id, can raise FileNotFoundError
        iid = int(mrnum)  # Try converting it to int, it will raise ValueError if it isn't an int
    except ValueError:
        if mrnum.isdigit():
            iid = int(mrnum)
        else:
            if not options.quiet:
                print(
                    "File has invalid value '{}', it must be an integer".format(
                        cachepath.read_text()
                    )
                )
                cachepath.unlink()  # Delete the file as it has an invalid value
                sys.exit(1)
    except FileNotFoundError:
        # The file isn't there, so assume the user named a branch an integer like '123'
        # but is actually trying to edit a merge request that has internal-id '123'
        if mrnum.isdigit():
            iid = int(mrnum)
        else:
            print(
                "branch name given, {}/branches/{}, has no corresponding cache file".format(
                    namespace, mrnum
                )
            )
            sys.exit(1)

    project = gl.projects.get(
        remote.load_project_id(token=gl.private_token), retry_transient_errors=True, lazy=True
    )
    mr = project.mergerequests.get(iid, include_rebase_in_progress=True)

    # Annotate some variables
    discussion_Locked: bool  # Whether only team members can comment in the merge request
    input: str  # String the user can edit
    output: List[str]  # 'input' after presented to the user to modify, split by newline
    output_len: int  # number of lines there is in 'output'
    linenum: int  # needle to move which line we are parsing from 'output'

    # Annotate all the final_ variables, they are the result of parsing 'output'
    final_state: Optional[str]
    final_title: Optional[str]
    final_desc: Optional[str]
    final_labels: Optional[List[str]]
    final_remove_branch: Optional[bool]
    final_target_branch: Optional[str]
    final_lock_discussion: Optional[bool]
    final_squash: Optional[bool]
    final_collab: Optional[bool]

    # If discussion isn't locked then the value returned is a None instead of a false like it is
    # normally expected
    discussion_locked = mr.attributes["discussion_locked"]
    if not discussion_locked:
        discussion_locked = False
    else:
        discussion_locked = True

    # The input variable will hold all the values we allow the user to edit,
    # we show all of them to the user in a text editor (they decide it by
    # setting the EDITOR variable) and they modify as they wish, we send
    # them back to be parsed, if any errors are found we tell the user
    # and change the valid changes.
    input = "# Set to 'close' to close the merge request\n"
    input += "# set to 'reopen' to open a closed merge request\n"
    input += "State: {}\n".format(mr.attributes["state"])

    input += "\n# Line must\n"
    input += "Title: {}\n".format(mr.attributes["title"])

    input += "\n# Any long multi-line string, will consider everything until the first line\n"
    input += "# that starts with a hashtag (#)\n"
    input += "Description: {}\n".format(mr.attributes["description"])

    input += "\n# Whitespace-separated list of labels, have it empty to remove all labels\n"
    input += "Labels: {}\n".format(" ".join(mr.attributes["labels"]).strip())

    input += "\n# If this is true then the source branch of the Merge Request will de deleted\n"
    input += "# when this Merge Request is merged, takes 'true' or 'false'\n"
    input += "Remove source branch when merged: {}\n".format(
        mr.attributes["force_remove_source_branch"]
    )

    input += "\n# Name of the branch the branch the commit will be land on\n"
    input += "Target Branch: {}\n".format(mr.attributes["target_branch"])

    input += "\n# If 'true' then no users can comment, 'false' otherwise\n"
    input += "Lock Discussion: {}\n".format(discussion_locked)

    input += "\n# If 'true' then all commits will be squashed into a single one\n"
    input += "Squash on merge: {}\n".format(mr.attributes["squash"])

    input += "\n# If 'true' maintainers of the repository will be able to push commits\n"
    input += "# into your branch, this is required for people rebasing-and-merging\n"
    input += "Allow Maintainers to push: {}".format(mr.attributes["allow_maintainer_to_push"])

    # This is the result of the edits of the user, and what we will parse to get all the answers
    # we need from what the user edited. Use splitlines() to split it into a list we can walk at
    # the pace we require.
    output = editor.edit(contents=input).decode("utf-8").splitlines()
    output_len = len(output)
    linenum = 0

    final_state = None
    final_title = None
    final_desc = None
    final_labels = None
    final_remove_branch = None
    final_target_branch = None
    final_lock_discussion = None
    final_squash = None
    final_collab = None

    while linenum < output_len:
        if output[linenum].startswith("State: "):
            final_state = ":".join(output[linenum].split(":")[1:]).strip()
            if (
                final_state != mr.attributes["state"]
                and final_state != "close"
                and final_state != "reopen"
            ):
                print("'State' given must be either close or reopen")
                final_state = mr.attributes["state"]
            linenum += 1
            continue
        if output[linenum].startswith("Title: "):
            final_title = ":".join(output[linenum].split(":")[1:]).strip()
            if len(final_title) >= 255:
                print("The title must be at most 254 characters, keeping old title")
                final_title = mr.attributes["title"]
            if final_title is None or len(final_title) < 1:
                print("The title must be at least 1 character, keeping old title")
                final_title = mr.attributes["title"]
            linenum += 1
            continue
        if output[linenum].startswith("Description: "):
            final_desc = "{}".format(":".join(output[linenum].split(":")[1:]).strip())
            linenum += 1
            while not output[linenum].startswith("#"):
                final_desc += "\n{}".format(output[linenum])
                linenum += 1
            # This value is given to us by the GitLab V4 API and is the hard limit for a description
            if len(final_desc) > 1048575:
                print(
                    "The description must be at most 1048576 characters, keeping old description"
                )
                final_desc = mr.attributes["description"]
            continue
        if output[linenum].startswith("Labels: "):
            final_labels = ":".join(output[linenum].split(":")[1:]).strip().split()
            linenum += 1
            continue
        if output[linenum].startswith("Remove source branch when merged: "):
            final_remove_branch = strtobool(":".join(output[linenum].split(":")[1:]).strip())
            if final_remove_branch is None:
                print("'Remove source branch when merged' value given must be true or false")
                final_remove_branch = mr.attributes["force_remove_source_branch"]
            linenum += 1
            continue
        if output[linenum].startswith("Target Branch: "):
            final_target_branch = ":".join(output[linenum].split(":")[1:]).strip()
            if final_target_branch is None or len(final_target_branch) <= 0:
                print("Target branch must be at least 1 character long, keeping old target branch")
                final_target_branch = mr.attributes["target_branch"]
            linenum += 1
            continue
        if output[linenum].startswith("Lock Discussion: "):
            final_lock_discussion = strtobool(":".join(output[linenum].split(":")[1:]).strip())
            if final_lock_discussion is None:
                print("'Lock Discussion' value given must be true or false")
                final_lock_discussion = mr.attributes["discussion_locked"]
            linenum += 1
            continue
        if output[linenum].startswith("Squash on merge: "):
            final_squash = strtobool(":".join(output[linenum].split(":")[1:]).strip())
            if final_squash is None:
                print("'Squash on merge' must be true or false")
                final_squash = mr.attributes["squash"]
            linenum += 1
            continue
        if output[linenum].startswith("Allow Maintainers to push: "):
            final_collab = strtobool(":".join(output[linenum].split(":")[1:]).strip())
            if final_collab is None:
                print("'Allow Maintainers to push' value given must be true or false")
                final_collab = mr.attributes["allow_maintainer_to_push"]
            linenum += 1
            continue
        linenum += 1

    # The state can be one of the three following values:
    # close - closes the merge request, or
    # reopen - opens the merge request again, or
    # the old value, which happens when an invalid state is given to us
    setattr(mr, "state", final_state)

    # The title can be either a valid new title of up to 255 chars, or the old title.
    # The old title happens when an invalid title (len = 0) is given
    setattr(mr, "title", final_title)

    # final_desc will either be the value given to us by the user, or an empty string
    # which is completely valid as having no description is acceptable
    setattr(mr, "description", final_desc)

    # final_labels will either be a List[str] that holds the labels we want. Or it will
    # be None, passing None will remove all labels to the user and is a legitimate action
    setattr(mr, "labels", final_labels)

    # final_remove_branch can be either:
    # - True -> Remove the source branch
    # - False -> Don't remove the source branch
    setattr(mr, "force_remove_source_branch", final_remove_branch)

    # final_target_branch can be either a new value that we can pass to gitlab
    # or the old value, and there is no harming in updating the old value
    setattr(mr, "target_branch", final_target_branch)

    # final_lock_discussion can be either:
    # True -> New comments only by team members
    # False -> All users can be new comments
    setattr(mr, "discussion_locked", final_lock_discussion)

    # final_squash can be either:
    # - True -> Squash commits on merge
    # - False -> Don't squash commits on merge
    setattr(mr, "squash", final_squash)

    # final_collab can be either:
    # - True -> Allow maintainers to push
    # - False -> Don't allow maintainers to push
    setattr(mr, "allow_maintainer_to_push", final_collab)

    if options.dry_run is True:
        print("State:", final_state)
        print("Title:", final_title)
        print("Description:", final_desc)
        print("Labels:", final_labels)
        print("Remove source branch when merged:", final_remove_branch)
        print("Target Branch:", final_target_branch)
        print("Lock Discussion:", final_lock_discussion)
        print("Squash on merge:", final_squash)
        print("Allow Maintainers to push:", final_collab)
        sys.exit(0)

    try:
        mr.save()
    except GitlabAuthenticationError as e:
        print("Failed to update, authentication error\n\n{}".format(e))
    except GitlabUpdateError as e:
        print("Failed to update, update error\n\n{}".format(e))


if __name__ == "__main__":
    main()
