import asyncio
import json
import os

from enum import Enum
from stoq import Stoq, RequestMeta


class OutputMode(Enum):
    toElastic = 1
    toStdOut = 2
    silent = 3


class InputMode(Enum):
    dirmon = 1
    manual = 2


def make_stoq(input_path, output_mode: OutputMode = OutputMode.toElastic, input_mode: InputMode = InputMode.manual):
    es_username = os.environ.get("ES_USERNAME", "es")
    es_password = os.environ.get("ES_PASSWORD", "password")
    es_host = os.environ.get("ES_HOST", "es")
    es_index = os.environ.get("ES_INDEX", "malwaretl")
    stoq_home = os.environ.get("STOQ_HOME", "/app")
    plugin_home = os.path.join(stoq_home, "plugins")
    dispatchers = ["decompress_dispatch"]
    plugins = ["EMBER_format_lief",
               "entropy",
               "hash",
               "hash_ssdeep",
               "mimetype",
               "mraptor",
               "ole",
               "peinfo",
               "rtf",
               "symhash",
               "xdpcarve",
               "xyz"
               ]
    always_dispatch = ",".join(plugins)
    if output_mode is OutputMode.toElastic:
        connector = ["es-search"]
    elif output_mode is OutputMode.toStdOut:
        connector = ["stdout"]
    elif output_mode is OutputMode.silent:
        connector = None
    else:
        raise Exception("Unknown scan_mode for Stoq")

    if input_mode is InputMode.dirmon:
        providers = ["dirmon"]
    elif input_mode is InputMode.manual:
        providers = None
    else:
        raise Exception("unknown input_mode for stoq")

    plugin_opts = {"dirmon": {"source_dir": input_path},
                   "decompress": {'passwords': "infected",
                                  "always_dispatch": always_dispatch},
                   "decompress_dispatch": {"always_dispatch": always_dispatch},
                   "filedir": {"source_dir": input_path, "recursive": True},
                   "es-search": {"es_options": json.dumps({"http_auth": [es_username, es_password],
                                                           "verify_certs": False,
                                                           "use_ssl": True,
                                                           "port": 9200}),
                                 "es_host": es_host,
                                 "es_index": es_index,
                                 "index_by_month": True
                                }
                   }
    s = Stoq(
        base_dir=stoq_home,
        plugin_dir_list=[plugin_home],
        connectors=connector,
        providers=providers,
        dispatchers=dispatchers,
        plugin_opts=plugin_opts,
    )
    return s


def init_github_android(output_mode, input_mode=InputMode.manual):
    input_path = os.environ.get("GITHUB_ANDROID_PATH", "/malware/github-android-malware/")
    s = make_stoq(input_path, output_mode, input_mode)
    meta = RequestMeta(source="github-android-malware",
                       extra_data={"target": "android", "malicious": "true"}
                       )
    return s, meta


def init_das_malwerk(output_mode, input_mode=InputMode.manual):
    input_path = os.environ.get("DAS_MALWERK_PATH", "/malware/malwerk/")
    s = make_stoq(input_path, output_mode, input_mode)
    meta = RequestMeta(source="das-malwerk",
                       extra_data={"malicous": "true"}
                       )
    return s, meta


def init_malshare(output_mode, input_mode=InputMode.manual):
    input_path = os.environ.get("malshare_PATH", "/malware/malshare/")
    s = make_stoq(input_path, output_mode, input_mode)
    meta = RequestMeta(source="malshare",
                       extra_data={"malicous": "true"}
                       )
    return s, meta


def init_mbazaar(output_mode, input_mode=InputMode.manual):
    input_path = os.environ.get("MBAZAAR_PATH", "/malware/mbazaar/")
    s = make_stoq(input_path, output_mode, input_mode)
    meta = RequestMeta(source="malware-bazaar",
                       extra_data={"malicous": "true"}
                       )
    return s, meta


def init_thezoo(output_mode, input_mode=InputMode.manual):
    input_path = os.environ.get("theZoo_PATH", "/malware/theZoo/")
    s = make_stoq(input_path, output_mode, input_mode)
    meta = RequestMeta(source="theZoo",
                       extra_data={"malicous": "true"}
                       )
    return s, meta


def init_urlhaus(output_mode, input_mode=InputMode.manual):
    input_path = os.environ.get("URLHAUS_PATH", "/malware/urlhaus/")
    s = make_stoq(input_path, output_mode, input_mode)
    meta = RequestMeta(source="urlhaus",
                       extra_data={"malicous": "true"}
                       )
    return s, meta


def init_vxug(output_mode, input_mode=InputMode.manual):
    input_path = os.environ.get("VXUG_PATH", "/malware/vxug/")
    s = make_stoq(input_path, output_mode, input_mode)
    meta = RequestMeta(source="vxug",
                       extra_data={"malicous": "true"}
                       )
    return s, meta


def init_vxvault(output_mode, input_mode=InputMode.manual):
    input_path = os.environ.get("VXVAULT_PATH", "/malware/vxvault/")
    s = make_stoq(input_path, output_mode, input_mode)
    meta = RequestMeta(source="vxvault",
                       extra_data={"malicous": "true"}
                       )
    return s, meta


transformer_map = {
    "github-android": init_github_android,
    "das-malwerk": init_das_malwerk,
    "malshare": init_malshare,
    "mbazaar": init_mbazaar,
    "theZoo": init_thezoo,
    "urlhaus": init_urlhaus,
    "vxug": init_vxug,
    "vxvault": init_vxvault
}


def get_transformers(output_mode, input_mode=InputMode.manual):
    transformers = list()

    workers = os.environ.get("TRANSFORMERS", "*")
    if workers == "*":
        for key in transformer_map:
            transformers.append(transformer_map[key](output_mode, input_mode))
    else:
        for worker in workers.split(","):
            if worker in transformer_map:
                transformers.append(transformer_map[worker](output_mode, input_mode))
    return transformers


def run(output_mode, input_mode=InputMode.manual):
    # configure plugins and outputs here
    transformers = get_transformers(output_mode, input_mode)
    loop = asyncio.get_event_loop()
    loop.run_until_complete(asyncio.gather(*[transformer.run(request_meta=meta) for transformer, meta in transformers]))


if __name__ == "__main__":
    scan_mode = os.environ.get("SCAN_MODE", "false").lower() == "true"
    run(scan_mode)
