import asyncio
import json
import os

from stoq import Stoq, RequestMeta


def make_stoq(input_path, scan_mode=False):
    es_username = os.environ.get("ES_USERNAME", "es")
    es_password = os.environ.get("ES_PASSWORD", "password")
    es_host = os.environ.get("ES_HOST", "es")
    es_index = os.environ.get("ES_INDEX", "malwaretl")
    stoq_home = os.environ.get("STOQ_HOME", "/app")
    plugin_home = os.path.join(stoq_home, "plugins")
    providers = ["dirmon"]
    dispatchers = ["decompress_dispatch"]
    plugins = ["EMBER_format_lief",
               "entropy",
               "hash",
               "hash_ssdeep",
               "lief",
               "mimetype",
               "mraptor",
               "ole",
               "peinfo",
               "rtf",
               "symhash",
               "xdpcarve",
               "xyz"
               ]
    always_dispatch = ",".join(plugins)
    connectors = ['es-search']
    plugin_opts = {"dirmon": {"source_dir": input_path},
                   "decompress": {'passwords': "infected",
                                  "always_dispatch": always_dispatch,
                                  "maximum_size": 100_000_000},
                   "decompress_dispatch": {"always_dispatch": always_dispatch},
                   "filedir": {"source_dir": input_path, "recursive": True},
                   "es-search": {"es_options": json.dumps({"http_auth": [es_username, es_password],
                                                           "verify_certs": False,
                                                           "use_ssl": True,
                                                           "port": 9200}),
                                 "es_host": es_host,
                                 "es_index": es_index,
                                 "index_by_month": True
                                }
                   }
    if scan_mode:
        s = Stoq(
                base_dir=stoq_home,
                plugin_dir_list=[plugin_home],
                connectors=connectors,
                dispatchers=dispatchers,
                plugin_opts=plugin_opts,
        )
    else:
        s = Stoq(
                base_dir=stoq_home,
                plugin_dir_list=[plugin_home],
                providers=providers,
                connectors=connectors,
                always_dispatch=plugins,
                plugin_opts=plugin_opts,
        )
    return s


def init_github_android(scan_mode):
    input_path = os.environ.get("GITHUB_ANDROID_PATH", "/malware/github-android-malware/")
    s = make_stoq(input_path, scan_mode)
    meta = RequestMeta(source="github-android-malware",
                       extra_data={"target": "android", "malicious": "true"}
                       )
    return s, meta


def init_das_malwerk(scan_mode):
    input_path = os.environ.get("DAS_MALWERK_PATH", "/malware/malwerk/")
    s = make_stoq(input_path, scan_mode)
    meta = RequestMeta(source="das-malwerk",
                       extra_data={"malicous": "true"}
                       )
    return s, meta


def init_malshare(scan_mode):
    input_path = os.environ.get("malshare_PATH", "/malware/malshare/")
    s = make_stoq(input_path, scan_mode)
    meta = RequestMeta(source="malshare",
                       extra_data={"malicous": "true"}
                       )
    return s, meta


def init_mbazaar(scan_mode):
    input_path = os.environ.get("MBAZAAR_PATH", "/malware/mbazaar/")
    s = make_stoq(input_path, scan_mode)
    meta = RequestMeta(source="malware-bazaar",
                       extra_data={"malicous": "true"}
                       )
    return s, meta


def init_thezoo(scan_mode):
    input_path = os.environ.get("theZoo_PATH", "/malware/theZoo/")
    s = make_stoq(input_path, scan_mode)
    meta = RequestMeta(source="theZoo",
                       extra_data={"malicous": "true"}
                       )
    return s, meta


def init_urlhaus(scan_mode):
    input_path = os.environ.get("URLHAUS_PATH", "/malware/urlhaus/")
    s = make_stoq(input_path, scan_mode)
    meta = RequestMeta(source="urlhaus",
                       extra_data={"malicous": "true"}
                       )
    return s, meta


def init_vxug(scan_mode):
    input_path = os.environ.get("VXUG_PATH", "/malware/vxug/")
    s = make_stoq(input_path, scan_mode)
    meta = RequestMeta(source="vxug",
                       extra_data={"malicous": "true"}
                       )
    return s, meta


def init_vxvault(scan_mode):
    input_path = os.environ.get("VXVAULT_PATH", "/malware/vxvault/")
    s = make_stoq(input_path, scan_mode)
    meta = RequestMeta(source="vxvault",
                       extra_data={"malicous": "true"}
                       )
    return s, meta


transformer_map = {
    "github-android": init_github_android,
    "das-malwerk": init_das_malwerk,
    "malshare": init_malshare,
    "mbazaar": init_mbazaar,
    "theZoo": init_thezoo,
    "urlhaus": init_urlhaus,
    "vxug": init_vxug,
    "vxvault": init_vxvault
}


def get_transformers(scan_mode):
    transformers = list()

    workers = os.environ.get("TRANSFORMERS", "*")
    if workers == "*":
        for key in transformer_map:
            transformers.append(transformer_map[key](scan_mode))
    else:
        for worker in workers.split(","):
            if worker in transformer_map:
                transformers.append(transformer_map[worker](scan_mode))
    return transformers


def run(scan_mode):
    # configure plugins and outputs here
    transformers = get_transformers(scan_mode)
    loop = asyncio.get_event_loop()
    loop.run_until_complete(asyncio.gather(*[transformer.run(request_meta=meta) for transformer, meta in transformers]))


if __name__ == "__main__":
    scan_mode = os.environ.get("SCAN_MODE", "false").lower() == "true"
    run(scan_mode)
