"""Levo's main CLI module."""
import logging
from typing import Dict, Optional, Tuple

import click
from click_loglevel import LogLevel
from schemathesis.cli import callbacks as schemathesis_cli_callbacks

from levocli.commands.constants import (
    AUTH_OPTION_HELP,
    CONTEXT_SETTINGS,
    DEBUG_SETTINGS_HELP,
    HEADER_OPTION_HELP,
    NO_REPORTS_TO_SAAS_HELP,
)
from levocli.commands.test_plan import test, test_plan

from . import callbacks, runners
from .config import TestConformanceCommandConfig
from .docker_utils import warn_on_invalid_env_and_mounts
from .logger import configure_logging, set_log_level
from .login import login_or_refresh

# Configure global logging settings on startup.
configure_logging()

# On startup, if running in Docker, warn on improper mounts
warn_on_invalid_env_and_mounts()


@click.group(context_settings=CONTEXT_SETTINGS)
@click.version_option()
def levo():
    """Command line tool for running automated tests generated by Levo's Platform."""
    return


@levo.command(short_help="Login into Levo's SaaS portal.")
@click.option(
    "-v",
    "--verbosity",
    type=LogLevel(),
    default=logging.WARN,
    help=DEBUG_SETTINGS_HELP,
)
def login(verbosity: LogLevel):
    """Authenticate the CLI with Levo's SaaS portal."""
    set_log_level(int(verbosity))
    login_or_refresh()
    click.secho(
        "\nYour account has been authenticated. Levo is now ready to be used!",
        fg="green",
    )


@levo.command(
    short_help="Perform schema conformance tests against API endpoints.",
    context_settings={
        **CONTEXT_SETTINGS,  # type: ignore
        "ignore_unknown_options": True,
        "allow_extra_args": True,
    },
)
@click.option(
    "--schema",
    help="--schema must specify a valid URL or file path (accessible from the CLI container) that points to an Open API / Swagger specification.",
    type=str,
    required=True,
    callback=callbacks.validate_schema,
)
@click.option(
    "--target-url",
    help="--target-url must specify a valid URL pointing to a live host that implements the endpoints"
    " specified by --schema.",
    type=str,
    required=True,
    callback=callbacks.validate_url,
)
@click.option(
    "--auth",
    "-a",
    help=(
        "Use in conjunction with '--auth'."
        " For basic authentication this specifies the target server's user and password. Example: USER:PASSWORD ."
    ),
    type=str,
    callback=schemathesis_cli_callbacks.validate_auth,
    hidden=True,  # Please see: https://app.clickup.com/t/1x0xk9m
)
@click.option(
    "--auth-type",
    "-A",
    type=click.Choice(["basic"], case_sensitive=False),
    default="basic",
    help="The authentication mechanism to be used. Only 'basic' currently supported.",
    show_default=True,
    hidden=True,  # Please see: https://app.clickup.com/t/1x0xk9m
)
@click.option(
    "--disable-reporting-to-saas",
    is_flag=True,
    help=NO_REPORTS_TO_SAAS_HELP,
)
@click.option(
    "--header",
    "-H",
    "headers",
    help=HEADER_OPTION_HELP,
    multiple=True,
    type=str,
    callback=callbacks.validate_headers,
)
@click.option(
    "--show-errors-tracebacks",
    help="Show full tracebacks for internal errors.",
    is_flag=True,
    is_eager=True,
    default=False,
    show_default=True,
)
@click.option(
    "-v",
    "--verbosity",
    type=LogLevel(),
    default=logging.WARN,
    help=DEBUG_SETTINGS_HELP,
)
@click.pass_context
def test_conformance(
    ctx: click.Context,
    target_url: str,
    schema: str,
    disable_reporting_to_saas: bool,
    auth: Optional[Tuple[str, str]],
    auth_type: str,
    headers: Dict[str, str],
    verbosity: LogLevel,
    show_errors_tracebacks: bool = False,
) -> None:
    """Perform schema conformance tests against API endpoints specified in the target-url."""
    set_log_level(int(verbosity))
    if not disable_reporting_to_saas:
        login_or_refresh()
    config = TestConformanceCommandConfig(
        target_url=target_url,
        schema=schema,
        auth=auth,
        auth_type=auth_type,
        report_to_saas=not disable_reporting_to_saas,
        headers=headers,
        passthru=ctx.args,
        show_errors_tracebacks=show_errors_tracebacks,
    )
    try:
        runners.schemathesis.cli_entrypoint(config)
    except Exception as e:
        click.echo()
        click.secho(e, fg="red")
        raise click.exceptions.Exit(1)


"""Sub commands are added here"""
levo.add_command(test)
levo.add_command(test_plan)
