"""Levo's main CLI module."""
import logging
import tempfile
from typing import Dict, Optional, Tuple

import click
from click_loglevel import LogLevel
from schemathesis.cli import callbacks as schemathesis_cli_callbacks

from . import callbacks, modules
from .config import TestConformanceCommandConfig, TestPlanCommandConfig
from .docker_utils import warn_on_invalid_env_and_mounts
from .logger import configure_logging, set_log_level
from .login import login_or_refresh

CONTEXT_SETTINGS = {"help_option_names": ["-h", "--help"]}

# Configure global logging settings on startup.
configure_logging()

# On startup, if running in Docker, warn on improper mounts
warn_on_invalid_env_and_mounts()


@click.group(context_settings=CONTEXT_SETTINGS)
@click.version_option()
def levo():
    """Command line tool for running the automated tests generated by Levo's Platform."""
    return


@levo.command(short_help="Login into Levo's SaaS portal.")
@click.option(
    "-v",
    "--verbosity",
    type=LogLevel(),
    default=logging.WARN,
    help="It accepts all of the Python logging log level names CRITICAL, ERROR, WARNING, INFO,"
    " DEBUG, and NOTSET, all case insensitive",
)
def login(verbosity: LogLevel):
    """Authenticate the CLI with Levo's SaaS portal."""
    set_log_level(int(verbosity))
    login_or_refresh()
    click.secho(
        "\nYour account has been authenticated. Levo is now ready to be used!",
        fg="green",
    )


@levo.command(
    short_help="Perform schema conformance tests against API endpoints.",
    context_settings={
        **CONTEXT_SETTINGS,  # type: ignore
        "ignore_unknown_options": True,
        "allow_extra_args": True,
    },
)
@click.option(
    "--schema",
    help="--schema must specify a valid URL or file path that points to an Open API / Swagger specification.",
    type=str,
    required=True,
    callback=callbacks.validate_schema,
)
@click.option(
    "--target-url",
    help="--target-url must specify a valid URL pointing to a live host that implements the endpoints"
    " specified by --schema.",
    type=str,
    required=True,
    callback=callbacks.validate_target_url,
)
@click.option(
    "--auth",
    "-a",
    help="Server user and password. Example: USER:PASSWORD",
    type=str,
    callback=schemathesis_cli_callbacks.validate_auth,
)
@click.option(
    "--auth-type",
    "-A",
    type=click.Choice(["basic"], case_sensitive=False),
    default="basic",
    help="The authentication mechanism to be used. Defaults to 'basic'.",
    show_default=True,
)
@click.option(
    "--disable-reporting-to-saas",
    is_flag=True,
)
@click.option(
    "--header",
    "-H",
    "headers",
    help='Custom header that will be used in all requests to the server. Example: -H "Authorization: Bearer 123"',
    multiple=True,
    type=str,
    callback=callbacks.validate_headers,
)
@click.option(
    "--show-errors-tracebacks",
    help="Show full tracebacks for internal errors.",
    is_flag=True,
    is_eager=True,
    default=False,
    show_default=True,
)
@click.option(
    "-v",
    "--verbosity",
    type=LogLevel(),
    default=logging.WARN,
    help="It accepts all of the Python logging log level names CRITICAL, ERROR, WARNING, INFO,"
    " DEBUG, and NOTSET, all case insensitive",
)
@click.pass_context
def test_conformance(
    ctx: click.Context,
    target_url: str,
    schema: str,
    disable_reporting_to_saas: bool,
    auth: Optional[Tuple[str, str]],
    auth_type: str,
    headers: Dict[str, str],
    verbosity: LogLevel,
    show_errors_tracebacks: bool = False,
) -> None:
    """Perform schema conformance tests against API endpoints specified in the target-url."""
    set_log_level(int(verbosity))
    login_or_refresh()
    config = TestConformanceCommandConfig(
        target_url=target_url,
        schema=schema,
        auth=auth,
        auth_type=auth_type,
        report_to_saas=not disable_reporting_to_saas,
        headers=headers,
        passthru=ctx.args,
        show_errors_tracebacks=show_errors_tracebacks,
    )
    modules.schemathesis.cli_entrypoint(config)


@levo.command(
    short_help="Fetches a test plan from Levo's SaaS and runs it against the target-url.",
    context_settings={
        **CONTEXT_SETTINGS,  # type: ignore
        "ignore_unknown_options": True,
        "allow_extra_args": True,
    },
)
@click.option(
    "--target-url",
    help="--target-url must specify a valid URL pointing to a live host that implements the endpoints"
    " that are present in the test plan.",
    type=str,
    required=True,
    callback=callbacks.validate_target_url,
)
@click.option(
    "--auth",
    "-a",
    help="Server user and password. Example: USER:PASSWORD",
    type=str,
    callback=schemathesis_cli_callbacks.validate_auth,
)
@click.option(
    "--auth-type",
    "-A",
    type=click.Choice(["basic", "token", "apikey"], case_sensitive=False),
    default="basic",
    help="The authentication mechanism to be used. Defaults to 'basic'.",
    show_default=True,
)
@click.option(
    "--test-plans-catalog",
    type=str,
    default=None,
    help="A path to a directory with local test plans.",
)
@click.option(
    "--disable-reporting-to-saas",
    is_flag=True,
)
@click.option("--plan-lrn", type=str)
@click.option(
    "--plan-name",
    type=str,
    default=None,
    help="Name of the plan to be picked from the test plans catalog. Only used with --test-plans-catalog",
)
@click.option(
    "--header",
    "-H",
    "headers",
    help='Custom header that will be used in all requests to the server. Example: -H "Authorization: Bearer 123"',
    multiple=True,
    type=str,
    callback=callbacks.validate_headers,
)
@click.option(
    "--show-errors-tracebacks",
    help="Show full tracebacks for internal errors.",
    is_flag=True,
    is_eager=True,
    default=False,
    show_default=True,
)
@click.option(
    "-v",
    "--verbosity",
    type=LogLevel(),
    default=logging.WARN,
    help="It accepts all of the Python logging log level names CRITICAL, ERROR, WARNING, INFO,"
    " DEBUG, and NOTSET, all case insensitive",
)
def test(
    plan_lrn: str,
    target_url: str,
    auth: Optional[Tuple[str, str]],
    auth_type: str,
    test_plans_catalog: Optional[str],
    plan_name: Optional[str],
    disable_reporting_to_saas: bool,
    headers: Dict[str, str],
    verbosity: LogLevel,
    show_errors_tracebacks: bool = False,
) -> None:
    """Fetch the test plan with the given lrn and run it against the server with target-url."""
    set_log_level(int(verbosity))
    login_or_refresh()
    config = TestPlanCommandConfig(
        target_url=target_url,
        plan_lrn=plan_lrn,
        auth=auth,
        auth_type=auth_type,
        test_plans_catalog=test_plans_catalog,
        plan_name=plan_name,
        report_to_saas=not disable_reporting_to_saas,
        headers=headers,
        show_errors_tracebacks=show_errors_tracebacks,
    )
    modules.plans.cli_entrypoint(config)


@levo.command(
    short_help="Exports test plan from Levo's SaaS into local file system.",
    context_settings={
        **CONTEXT_SETTINGS,  # type: ignore
        "ignore_unknown_options": True,
        "allow_extra_args": True,
    },
)
@click.option("--plan-lrn", type=str, required=True)
@click.option(
    "--local-dir",
    type=str,
    default=None,
    help="A path to a local directory where the test plan has to be written to.",
)
@click.option(
    "-v",
    "--verbosity",
    type=LogLevel(),
    default=logging.WARN,
    help="It accepts all of the Python logging log level names CRITICAL, ERROR, WARNING, INFO,"
    " DEBUG, and NOTSET, all case insensitive",
)
def export(
    plan_lrn: str,
    local_dir: Optional[str],
    verbosity: LogLevel,
) -> None:
    set_log_level(int(verbosity))
    login_or_refresh()

    path = local_dir if local_dir else tempfile.mkdtemp()
    modules.plans.export_plan(plan_lrn, path)
