"""Levo's main CLI module."""
from typing import Optional, Tuple

import click
from schemathesis.cli import callbacks as schemathesis_cli_callbacks

from . import callbacks, modules
from .config import TestConformanceCommandConfig, TestPlanCommandConfig
from .docker_utils import warn_on_invalid_env_and_mounts
from .login import login_or_refresh

CONTEXT_SETTINGS = {"help_option_names": ["-h", "--help"]}

# On startup, if running in Docker, warn on improper mounts
warn_on_invalid_env_and_mounts()


@click.group(context_settings=CONTEXT_SETTINGS)
@click.version_option()
def levo():
    """Command line tool for running the automated tests generated by Levo's Platform."""
    return


@levo.command(short_help="Login into Levo's SaaS portal.")
def login():
    """Authenticate the CLI with Levo's SaaS portal."""
    login_or_refresh()


@levo.command(
    short_help="Perform schema conformance tests against API endpoints.",
    context_settings={
        **CONTEXT_SETTINGS,
        "ignore_unknown_options": True,
        "allow_extra_args": True,
    },
)
@click.option(
    "--schema",
    help="--schema must specify a valid URL or file path that points to an Open API / Swagger specification.",
    type=str,
    required=True,
    callback=callbacks.validate_schema,
)
@click.option(
    "--target-url",
    help="--target-url must specify a valid URL pointing to a live host that implements the endpoints"
    " specified by --schema.",
    type=str,
    required=True,
    callback=callbacks.validate_target_url,
)
@click.option(
    "--auth",
    "-a",
    help="Server user and password. Example: USER:PASSWORD",
    type=str,
    callback=schemathesis_cli_callbacks.validate_auth,
)
@click.option(
    "--auth-type",
    "-A",
    type=click.Choice(["basic"], case_sensitive=False),
    default="basic",
    help="The authentication mechanism to be used. Defaults to 'basic'.",
    show_default=True,
)
@click.option(
    "--disable-reporting-to-saas",
    is_flag=True,
)
@click.pass_context
def test_conformance(
    ctx: click.Context,
    target_url: str,
    schema: str,
    disable_reporting_to_saas: bool,
    auth: Optional[Tuple[str, str]],
    auth_type: str,
) -> None:
    """Perform schema conformance tests against API endpoints specified in the target-url."""
    config = TestConformanceCommandConfig(
        target_url=target_url,
        schema=schema,
        auth=auth,
        auth_type=auth_type,
        report_to_saas=not disable_reporting_to_saas,
        passthru=ctx.args,
    )
    modules.schemathesis.cli_entrypoint(config)


@levo.command(
    short_help="Fetches a test plan from Levo's SaaS and runs it against the target-url.",
    context_settings={
        **CONTEXT_SETTINGS,
        "ignore_unknown_options": True,
        "allow_extra_args": True,
    },
)
@click.option(
    "--target-url",
    help="--target-url must specify a valid URL pointing to a live host that implements the endpoints"
    "that are present in the test plan.",
    type=str,
    required=True,
    callback=callbacks.validate_target_url,
)
@click.option(
    "--auth",
    "-a",
    help="Server user and password. Example: USER:PASSWORD",
    type=str,
    callback=schemathesis_cli_callbacks.validate_auth,
)
@click.option(
    "--auth-type",
    "-A",
    type=click.Choice(["basic"], case_sensitive=False),
    default="basic",
    help="The authentication mechanism to be used. Defaults to 'basic'.",
    show_default=True,
)
@click.option(
    "--test-plans-catalog",
    type=str,
    default=None,
    help="A path to a directory with local test plans.",
)
@click.option(
    "--disable-reporting-to-saas",
    is_flag=True,
)
@click.option("--plan-lrn", type=str, required=True)
def test(
    plan_lrn: str,
    target_url: str,
    auth: Optional[Tuple[str, str]],
    auth_type: str,
    test_plans_catalog: Optional[str],
    disable_reporting_to_saas: bool,
) -> None:
    """Fetch the test plan with the given lrn and run it against the server with target-url."""
    config = TestPlanCommandConfig(
        target_url=target_url,
        auth=auth,
        auth_type=auth_type,
        test_plans_catalog=test_plans_catalog,
        report_to_saas=not disable_reporting_to_saas,
        plan_lrn=plan_lrn,
    )
    modules.plans.cli_entrypoint(config)
