import asyncio
import logging
import json
from .runner import is_process_closed, regist_server, unregist_server
from .tcp_callback_base import TcpCallbackBase
from .tcp_proto_base import TcpProtoBase
from .tcp_session_mgr import TcpSession, TcpSessionMgr
from .kcp_stream import start_server


class TcpServer:
    def __init__(self,
                 host: str,
                 port: int,
                 cb: TcpCallbackBase,
                 proto: TcpProtoBase,
                 heartbeat_interval: float = 0,
                 max_no_msg_count: int = 0,
                 loop=None):
        self.loop = loop or asyncio.get_running_loop()
        self.host = host
        self.port = port
        self.server = None
        self.session_mgr = TcpSessionMgr(cb,
                                         proto,
                                         heartbeat_interval,
                                         max_no_msg_count)
        self._start_server = asyncio.start_server

    # def use_kcp(self, conv: int, timeout_time=60 * 1000, kcp_loop_interval=0.01, nodelay=1, interval=20, resend=3, nc=1):
    #     def _start_server(client_connected_cb: callable,
    #                       host=None, port=None,
    #                       loop=asyncio.get_event_loop()):
    #         return start_server(conv, client_connected_cb,
    #                             host, port, loop, timeout_time,
    #                             kcp_loop_interval=kcp_loop_interval,
    #                             nodelay=nodelay, interval=interval, resend=resend, nc=nc)
    #     self._start_server = _start_server

    # def use_tcp(self):
    #     self._start_server = asyncio.start_server

    async def _start(self):
        try:
            if is_process_closed():
                return
            self.server = await self._start_server(
                client_connected_cb=self.session_mgr.on_new_session,
                host=self.host,
                port=self.port,
                loop=self.loop)
            if not regist_server(self):
                self.server.close()
            await self.server.start_serving()
        except Exception as e:
            logging.error(e.with_traceback(None))

    async def _close(self):
        if self.server is not None and self.server.is_serving():
            self.session_mgr.close()
            self.server.close()

    def start(self):
        self.loop.create_task(self._start())

    def close(self, unregist=True):
        self.loop.create_task(self._close())
        if unregist:
            unregist_server(self)

    def send_msg(self, session_id: int, data: bytes):
        if is_process_closed():
            return
        session: TcpSession = self.session_mgr.sessions.get(session_id)
        if session is None:
            logging.error(
                f"[send_msg] not such session! session_id = {session_id}!")
            return False
        return session.send_msg(data)

    def send_json(self, session_id: int, data: dict):
        if is_process_closed():
            return
        session: TcpSession = self.session_mgr.sessions.get(session_id)
        if session is None:
            logging.error(
                f"[send_msg] not such session! session_id = {session_id}!")
            return
        session.send_msg(json.dumps(data).encode())

    async def flush_session(self, session_id: int):
        session: TcpSession = self.session_mgr.sessions.get(session_id)
        if session is None:
            return
        await session.drain()
