# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['kyber_py']

package_data = \
{'': ['*']}

install_requires = \
['pycryptodome==3.14.1']

setup_kwargs = {
    'name': 'kyber-py',
    'version': '0.1.3',
    'description': 'Implementation of the kyber crystal algorithm made by https://github.com/jack4818/kyber-py',
    'long_description': "# CRYSTALS-Kyber Python Implementation\n\nThis repository contains a pure python implementation of CRYSTALS-Kyber \nfollowing (at the time of writing) the most recent \n[specification](https://pq-crystals.org/kyber/data/kyber-specification-round3-20210804.pdf)\n(v3.02)\n\n## Disclaimer\n\n:warning: **Under no circumstances should this be used for a cryptographic application.** :warning:\n\nI have written `kyber-py` as a way to learn about the way Kyber works, and to\ntry and create a clean, well commented implementation which people can learn \nfrom.\n\nThis code is not constant time, or written to be performant. Rather, it was \nwritten so that reading though Algorithms 1-9 in the \n[specification](https://pq-crystals.org/kyber/data/kyber-specification-round3-20210804.pdf)\nclosely matches the code which is seen in `kyber.py`.\n\n### KATs\n\nThis implementation currently passes all KAT tests from the reference implementation. \nFor more information, see the unit tests in [`test_kyber.py`](test_kyber.py).\n\n**Note**: there is a discrepancy between the specification and reference implementation.\nTo ensure all KATs pass, I have to generate the public key **before** the random\nbytes $z = \\mathcal{B}^{32}$ in algorithm 7 of the \n[specification](https://pq-crystals.org/kyber/data/kyber-specification-round3-20210804.pdf)\n(v3.02).\n\n### Dependencies\n\nOriginally this was planned to have zero dependencies, however to make this work\npass the KATs, I needed a deterministic CSRNG. The reference implementation uses\nAES256 CTR DRBG. I have implemented this in [`aes256_ctr_drbg.py`](aes256_ctr_drbg.py). \nHowever, I have not implemented AES itself, instead I import this from `pycryptodome`.\n\nTo install dependencies, run `pip -r install requirements`.\n\nIf you're happy to use system randomness (`os.urandom`) then you don't need\nthis dependency.\n\n## Using kyber-py\n\nThere are three functions exposed on the `Kyber` class which are intended\nfor use:\n\n- `Kyber.keygen()`: generate a keypair `(pk, sk)`\n- `Kyber.enc(pk)`: generate a challenge and a shared key `(c, K)`\n- `Kyber.dec(sk, c)`: generate the shared key `K`\n\nTo use `Kyber()` it must be initialised with a dictionary of the \nprotocol parameters. An example can be seen in `DEFAULT_PARAMETERS`.\n\nAdditionally, the class has been initialised with these default parameters, \nso you can simply import the NIST level you want to play with:\n\n#### Example\n\n```python\n>>> from kyber import Kyber512\n>>> pk, sk = Kyber512.keygen()\n>>> c, key = Kyber512.enc(pk)\n>>> _key = Kyber512.dec(c, sk)\n>>> assert key == _key\n```\n\nThe above example would also work with `Kyber768` and `Kyber1024`.\n\n### Benchmarks\n\n**TODO**: Better benchmarks? Although this was never about speed haha.\n\nFor now, here are some approximate benchmarks:\n\n|  1000 Iterations         | Kyber512 | Kyber768 | Kyber1024 |\n|--------------------------|----------|----------|-----------|\n| `KeyGen()`               |  6.868s  | 10.820s  | 16.172s   |\n| `Enc()`                  | 10.677s  | 16.094s  | 22.341s   |\n| `Dec()`                  | 16.822s  | 25.979s  | 33.524s   |\n\nAll times recorded using a Intel Core i7-9750H CPU. \n\n## Future Plans\n\n* Add documentation on `NTT` transform for polynomials\n* Add documentation for working with DRBG and setting the seed\n\n### Include Dilithium\n\nUsing [`polynomials.py`](polynomials.py) and [`modules.py`](modules.py) \nthis work could be extended to\nhave a pure python implementation of CRYSTALS-Dilithium too.\n\nI suppose then this repo should be called `crystals-py` but I wont\nget ahead of myself.\n\n## Discussion of Implementation\n\n### Kyber\n\n```\nTODO:\n\nAdd some more information about how working with Kyber works with this\nlibrary...\n```\n\n### Polynomials\n\nThe file [`polynomials.py`](polynomials.py) contains the classes \n`PolynomialRing` and \n`Polynomial`. This implements the univariate polynomial ring\n\n$$\nR_q = \\mathbb{F}_q[X] /(X^n + 1) \n$$\n\nThe implementation is inspired by `SageMath` and you can create the\nring $R_{11} = \\mathbb{F}_{11}[X] /(X^8 + 1)$ in the following way:\n\n#### Example\n\n```python\n>>> R = PolynomialRing(11, 8)\n>>> x = R.gen()\n>>> f = 3*x**3 + 4*x**7\n>>> g = R.random_element(); g\n5 + x^2 + 5*x^3 + 4*x^4 + x^5 + 3*x^6 + 8*x^7\n>>> f*g\n8 + 9*x + 10*x^3 + 7*x^4 + 2*x^5 + 5*x^6 + 10*x^7\n>>> f + f\n6*x^3 + 8*x^7\n>>> g - g\n0\n```\n\nWe additionally include functions for `PolynomialRing` and `Polynomial`\nto move from bytes to polynomials (and back again). \n\n- `PolynomialRing`\n  - `parse(bytes)` takes $3n$ bytes and produces a random polynomial in $R_q$\n  - `decode(bytes, l)` takes $\\ell n$ bits and produces a polynomial in $R_q$\n  - `cbd(beta, eta)` takes $\\eta \\cdot n / 4$ bytes and produces a polynomial in $R_q$ with coefficents taken from a centered binomial distribution\n- `Polynomial`\n  - `self.encode(l)` takes the polynomial and returns a length $\\ell n / 8$ bytearray\n  \n#### Example\n\n```python\n>>> R = PolynomialRing(11, 8)\n>>> f = R.random_element()\n>>> # If we do not specify `l` then it is computed for us (minimal value)\n>>> f_bytes = f.encode()\n>>> f_bytes.hex()\n'06258910'\n>>> R.decode(f_bytes) == f\nTrue\n>>> # We can also set `l` ourselves\n>>> f_bytes = f.encode(l=10)\n>>> f_bytes.hex()\n'00180201408024010000'\n>>> R.decode(f_bytes, l=10) == f\nTrue\n```\n\nLastly, we define a `self.compress(d)` and `self.decompress(d)` method for\npolynomials following page 2 of the \n[specification](https://pq-crystals.org/kyber/data/kyber-specification-round3-20210804.pdf)\n\n$$\n\\textsf{compress}_q(x, d) = \\lceil (2^d / q) \\cdot x \\rfloor \\textrm{mod}^+ 2^d,\n$$\n\n$$\n\\textsf{decompress}_q(x, d) = \\lceil (q / 2^d) \\cdot x \\rfloor.\n$$\n\nThe functions `compress` and `decompress` are defined for the coefficients \nof a polynomial and a polynomial is (de)compressed by acting the function\non every coefficient. \nSimilarly, an element of a module is (de)compressed by acting the\nfunction on every polynomial.\n\n#### Example\n\n```python\n>>> R = PolynomialRing(11, 8)\n>>> f = R.random_element()\n>>> f\n9 + 3*x + 5*x^2 + 2*x^3 + 9*x^4 + 10*x^5 + 6*x^6 + x^7\n>>> f.compress(1)\nx + x^2 + x^6\n>>> f.decompress(1)\n6*x + 6*x^2 + 6*x^6\n```\n\n**Note**: compression is lossy! We do not get the same polynomial back \nby computing `f.compress(d).decompress(d)`. They are however *close*.\nSee the specification for more information.\n\n### Number Theoretic Transform\n\n```\nTODO:\n\nThis is now handled by `NTTHelper` which is passed to `PolynomialRing`\nand has functions which are accessed by `Polynomial`.\n\nTalk about what is available, and how they are used.\n```\n\n### Modules\n\nThe file [`modules.py`](modules.py) contains the classes `Module` and `Matrix`.\nA module is a generalisation of a vector space, where the field\nof scalars is replaced with a ring. In the case of Kyber, we \nneed the module with the ring $R_q$ as described above. \n\n`Matrix` allows elements of the module to be of size $m \\times n$\nbut for Kyber, we only need vectors of length $k$ and square\nmatricies of size $k \\times k$.\n\nAs an example of the operations we can perform with out `Module`\nlets revisit the ring from the previous example:\n\n#### Example\n\n```python\n>>> R = PolynomialRing(11, 8)\n>>> x = R.gen()\n>>>\n>>> M = Module(R)\n>>> # We create a matrix by feeding the coefficients to M\n>>> A = M([[x + 3*x**2, 4 + 3*x**7], [3*x**3 + 9*x**7, x**4]])\n>>> A\n[    x + 3*x^2, 4 + 3*x^7]\n[3*x^3 + 9*x^7,       x^4]\n>>> # We can add and subtract matricies of the same size\n>>> A + A\n[  2*x + 6*x^2, 8 + 6*x^7]\n[6*x^3 + 7*x^7,     2*x^4]\n>>> A - A\n[0, 0]\n[0, 0]\n>>> # A vector can be constructed by a list of coefficents\n>>> v = M([3*x**5, x])\n>>> v\n[3*x^5, x]\n>>> # We can compute the transpose\n>>> v.transpose()\n[3*x^5]\n[    x]\n>>> v + v\n[6*x^5, 2*x]\n>>> # We can also compute the transpose in place\n>>> v.transpose_self()\n[3*x^5]\n[    x]\n>>> v + v\n[6*x^5]\n[  2*x]\n>>> # Matrix multiplication follows python standards and is denoted by @\n>>> A @ v\n[8 + 4*x + 3*x^6 + 9*x^7]\n[        2 + 6*x^4 + x^5]\n```\n\nWe also carry through `Matrix.encode()` and \n`Module.decode(bytes, n_rows, n_cols)` \nwhich simply use the above functions defined for polynomials and run for each\nelement.\n\n#### Example\n\nWe can see how encoding / decoding a vector works in the following example.\nNote that we can swap the rows/columns to decode bytes into the transpose\nwhen working with a vector.\n\n```python\n>>> R = PolynomialRing(11, 8)\n>>> M = Module(R)\n>>> v = M([R.random_element() for _ in range(2)])\n>>> v_bytes = v.encode()\n>>> v_bytes.hex()\n'd'\n>>> M.decode(v_bytes, 1, 2) == v\nTrue\n>>> v_bytes = v.encode(l=10)\n>>> v_bytes.hex()\n'a014020100103004000040240a03009030080200'\n>>> M.decode(v_bytes, 1, 2, l=10) == v\nTrue\n>>> M.decode(v_bytes, 2, 1, l=10) == v.transpose()\nTrue\n>>> # We can also compress and decompress elements of the module\n>>> v\n[5 + 10*x + 4*x^2 + 2*x^3 + 8*x^4 + 3*x^5 + 2*x^6, 2 + 9*x + 5*x^2 + 3*x^3 + 9*x^4 + 3*x^5 + x^6 + x^7]\n>>> v.compress(1)\n[1 + x^2 + x^4 + x^5, x^2 + x^3 + x^5]\n>>> v.decompress(1)\n[6 + 6*x^2 + 6*x^4 + 6*x^5, 6*x^2 + 6*x^3 + 6*x^5]\n```\n\n## Baby Kyber\n\nA great resource for learning Kyber is available at\n[Approachable Cryptography](https://cryptopedia.dev/posts/kyber/).\n\nWe include code corresponding to their example in `baby_kyber.py`.\n",
    'author': 'Marc Partensky',
    'author_email': 'marc.partensky@gmail.com',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'None',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.10,<4.0',
}


setup(**setup_kwargs)
