"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.HtpasswdHashAlgorithm = exports.DEFAULT_BCRYPT_ROUNDS = void 0;
exports.addUserToHTPasswd = addUserToHTPasswd;
exports.changePasswordToHTPasswd = changePasswordToHTPasswd;
exports.generateHtpasswdLine = generateHtpasswdLine;
exports.lockAndRead = lockAndRead;
exports.parseHTPasswd = parseHTPasswd;
exports.sanityCheck = sanityCheck;
exports.verifyPassword = verifyPassword;
var _crypto = _interopRequireDefault(require("crypto"));
var _apacheMd = _interopRequireDefault(require("apache-md5"));
var _bcryptjs = _interopRequireDefault(require("bcryptjs"));
var _httpErrors = _interopRequireDefault(require("http-errors"));
var _fileLocking = require("@verdaccio/file-locking");
var _crypt = _interopRequireDefault(require("./crypt3"));
function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }
const DEFAULT_BCRYPT_ROUNDS = 10;
exports.DEFAULT_BCRYPT_ROUNDS = DEFAULT_BCRYPT_ROUNDS;
let HtpasswdHashAlgorithm;
exports.HtpasswdHashAlgorithm = HtpasswdHashAlgorithm;
(function (HtpasswdHashAlgorithm) {
  HtpasswdHashAlgorithm["md5"] = "md5";
  HtpasswdHashAlgorithm["sha1"] = "sha1";
  HtpasswdHashAlgorithm["crypt"] = "crypt";
  HtpasswdHashAlgorithm["bcrypt"] = "bcrypt";
})(HtpasswdHashAlgorithm || (exports.HtpasswdHashAlgorithm = HtpasswdHashAlgorithm = {}));
// this function neither unlocks file nor closes it
// it'll have to be done manually later
function lockAndRead(name, cb) {
  (0, _fileLocking.readFile)(name, {
    lock: true
  }, (err, res) => {
    if (err) {
      return cb(err);
    }
    return cb(null, res);
  });
}

/**
 * parseHTPasswd - convert htpasswd lines to object.
 * @param {string} input
 * @returns {object}
 */
function parseHTPasswd(input) {
  // The input is split on line ending styles that are both windows and unix compatible
  return input.split(/[\r]?[\n]/).reduce((result, line) => {
    const args = line.split(':', 3).map(str => str.trim());
    if (args.length > 1) {
      result[args[0]] = args[1];
    }
    return result;
  }, {});
}

/**
 * verifyPassword - matches password and it's hash.
 * @param {string} passwd
 * @param {string} hash
 * @returns {Promise<boolean>}
 */
async function verifyPassword(passwd, hash) {
  if (hash.match(/^\$2([aby])\$/)) {
    return await _bcryptjs.default.compare(passwd, hash);
  } else if (hash.indexOf('{PLAIN}') === 0) {
    return passwd === hash.slice(7);
  } else if (hash.indexOf('{SHA}') === 0) {
    return _crypto.default.createHash('sha1')
    // https://nodejs.org/api/crypto.html#crypto_hash_update_data_inputencoding
    .update(passwd, 'utf8').digest('base64') === hash.slice(5);
  }
  // for backwards compatibility, first check md5 then check crypt3
  return (0, _apacheMd.default)(passwd, hash) === hash || (0, _crypt.default)(passwd, hash) === hash;
}

/**
 * generateHtpasswdLine - generates line for htpasswd file.
 * @param {string} user
 * @param {string} passwd
 * @param {HtpasswdHashConfig} hashConfig
 * @returns {Promise<string>}
 */
async function generateHtpasswdLine(user, passwd, hashConfig) {
  let hash;
  switch (hashConfig.algorithm) {
    case HtpasswdHashAlgorithm.bcrypt:
      hash = await _bcryptjs.default.hash(passwd, hashConfig.rounds || DEFAULT_BCRYPT_ROUNDS);
      break;
    case HtpasswdHashAlgorithm.crypt:
      hash = (0, _crypt.default)(passwd);
      break;
    case HtpasswdHashAlgorithm.md5:
      hash = (0, _apacheMd.default)(passwd);
      break;
    case HtpasswdHashAlgorithm.sha1:
      hash = '{SHA}' + _crypto.default.createHash('sha1').update(passwd, 'utf8').digest('base64');
      break;
    default:
      throw (0, _httpErrors.default)('Unexpected hash algorithm');
  }
  const comment = 'autocreated ' + new Date().toJSON();
  return `${user}:${hash}:${comment}\n`;
}

/**
 * addUserToHTPasswd - Generate a htpasswd format for .htpasswd
 * @param {string} body
 * @param {string} user
 * @param {string} passwd
 * @returns {Promise<string>}
 */
async function addUserToHTPasswd(body, user, passwd, hashConfig) {
  if (user !== encodeURIComponent(user)) {
    const err = (0, _httpErrors.default)('username should not contain non-uri-safe characters');
    err.status = 409;
    throw err;
  }
  let newline = await generateHtpasswdLine(user, passwd, hashConfig);
  if (body.length && body[body.length - 1] !== '\n') {
    newline = '\n' + newline;
  }
  return body + newline;
}

/**
 * Sanity check for a user
 * @param {string} user
 * @param {object} users
 * @param {string} password
 * @param {Callback} verifyFn
 * @param {number} maxUsers
 * @returns {object}
 */
async function sanityCheck(user, password, verifyFn, users, maxUsers) {
  let err;

  // check for user or password
  if (!user || !password) {
    err = Error('username and password is required');
    err.status = 400;
    return err;
  }
  const hash = users[user];
  if (maxUsers < 0) {
    err = Error('user registration disabled');
    err.status = 409;
    return err;
  }
  if (hash) {
    const auth = await verifyFn(password, users[user]);
    if (auth) {
      err = Error('username is already registered');
      err.status = 409;
      return err;
    }
    err = Error('unauthorized access');
    err.status = 401;
    return err;
  } else if (Object.keys(users).length >= maxUsers) {
    err = Error('maximum amount of users reached');
    err.status = 403;
    return err;
  }
  return null;
}

/**
 * changePasswordToHTPasswd - change password for existing user
 * @param {string} body
 * @param {string} user
 * @param {string} passwd
 * @param {string} newPasswd
 * @param {HtpasswdHashConfig} hashConfig
 * @returns {Promise<string>}
 */
async function changePasswordToHTPasswd(body, user, passwd, newPasswd, hashConfig) {
  let lines = body.split('\n');
  const userLineIndex = lines.findIndex(line => line.split(':', 1).shift() === user);
  if (userLineIndex === -1) {
    throw new Error(`Unable to change password for user '${user}': user does not currently exist`);
  }
  const [username, hash] = lines[userLineIndex].split(':', 2);
  const passwordValid = await verifyPassword(passwd, hash);
  if (!passwordValid) {
    throw new Error(`Unable to change password for user '${user}': invalid old password`);
  }
  const updatedUserLine = await generateHtpasswdLine(username, newPasswd, hashConfig);
  lines.splice(userLineIndex, 1, updatedUserLine);
  return lines.join('\n');
}
//# sourceMappingURL=utils.js.map