import { IDisposable } from '@lumino/disposable';
import { IMessageHandler, Message } from '@lumino/messaging';
import { Widget } from '@lumino/widgets';
import { CellRenderer } from './cellrenderer';
import { DataModel } from './datamodel';
import { RendererMap } from './renderermap';
import { SectionList } from './sectionlist';
import { SelectionModel } from './selectionmodel';
import { ICellEditorController } from './celleditorcontroller';
/**
 * A widget which implements a high-performance tabular data grid.
 *
 * #### Notes
 * A data grid is implemented as a composition of child widgets. These
 * child widgets are considered an implementation detail. Manipulating
 * the child widgets of a data grid directly is undefined behavior.
 *
 * This class is not designed to be subclassed.
 */
export declare class DataGrid extends Widget {
    /**
     * Construct a new data grid.
     *
     * @param options - The options for initializing the data grid.
     */
    constructor(options?: DataGrid.IOptions);
    /**
     * Dispose of the resources held by the widgets.
     */
    dispose(): void;
    /**
     * Get the data model for the data grid.
     */
    get dataModel(): DataModel | null;
    /**
     * Set the data model for the data grid.
     *
     * #### Notes
     * This will automatically remove the current selection model.
     */
    set dataModel(value: DataModel | null);
    /**
     * Get the selection model for the data grid.
     */
    get selectionModel(): SelectionModel | null;
    /**
     * Set the selection model for the data grid.
     */
    set selectionModel(value: SelectionModel | null);
    /**
     * Get the key handler for the data grid.
     */
    get keyHandler(): DataGrid.IKeyHandler | null;
    /**
     * Set the key handler for the data grid.
     */
    set keyHandler(value: DataGrid.IKeyHandler | null);
    /**
     * Get the mouse handler for the data grid.
     */
    get mouseHandler(): DataGrid.IMouseHandler | null;
    /**
     * Set the mouse handler for the data grid.
     */
    set mouseHandler(value: DataGrid.IMouseHandler | null);
    /**
     * Get the style for the data grid.
     */
    get style(): DataGrid.Style;
    /**
     * Set the style for the data grid.
     */
    set style(value: DataGrid.Style);
    /**
     * Get the cell renderer map for the data grid.
     */
    get cellRenderers(): RendererMap;
    /**
     * Set the cell renderer map for the data grid.
     */
    set cellRenderers(value: RendererMap);
    /**
     * Get the header visibility for the data grid.
     */
    get headerVisibility(): DataGrid.HeaderVisibility;
    /**
     * Set the header visibility for the data grid.
     */
    set headerVisibility(value: DataGrid.HeaderVisibility);
    /**
     * Get the default sizes for the various sections of the data grid.
     */
    get defaultSizes(): DataGrid.DefaultSizes;
    /**
     * Set the default sizes for the various sections of the data grid.
     */
    set defaultSizes(value: DataGrid.DefaultSizes);
    /**
     * Get the minimum sizes for the various sections of the data grid.
     */
    get minimumSizes(): DataGrid.DefaultSizes;
    /**
     * Set the minimum sizes for the various sections of the data grid.
     */
    set minimumSizes(value: DataGrid.DefaultSizes);
    /**
     * Get the copy configuration for the data grid.
     */
    get copyConfig(): DataGrid.CopyConfig;
    /**
     * Set the copy configuration for the data grid.
     */
    set copyConfig(value: DataGrid.CopyConfig);
    /**
     * Get whether the last row is stretched.
     */
    get stretchLastRow(): boolean;
    /**
     * Set whether the last row is stretched.
     */
    set stretchLastRow(value: boolean);
    /**
     * Get whether the last column is stretched.
     */
    get stretchLastColumn(): boolean;
    /**
     * Set whether the last column is stretched.
     */
    set stretchLastColumn(value: boolean);
    /**
     * The virtual width of the row headers.
     */
    get headerWidth(): number;
    /**
     * The virtual height of the column headers.
     */
    get headerHeight(): number;
    /**
     * The virtual width of the grid body.
     *
     * #### Notes
     * This does *not* account for a stretched last column.
     */
    get bodyWidth(): number;
    /**
     * The virtual height of the grid body.
     *
     * #### Notes
     * This does *not* account for a stretched last row.
     */
    get bodyHeight(): number;
    /**
     * The virtual width of the entire grid.
     *
     * #### Notes
     * This does *not* account for a stretched last column.
     */
    get totalWidth(): number;
    /**
     * The virtual height of the entire grid.
     *
     * #### Notes
     * This does *not* account for a stretched last row.
     */
    get totalHeight(): number;
    /**
     * The actual width of the viewport.
     */
    get viewportWidth(): number;
    /**
     * The actual height of the viewport.
     */
    get viewportHeight(): number;
    /**
     * The width of the visible portion of the grid body.
     */
    get pageWidth(): number;
    /**
     * The height of the visible portion of the grid body.
     */
    get pageHeight(): number;
    /**
     * The current scroll X position of the viewport.
     */
    get scrollX(): number;
    /**
     * The current scroll Y position of the viewport.
     */
    get scrollY(): number;
    /**
     * The maximum scroll X position for the grid.
     */
    get maxScrollX(): number;
    /**
     * The maximum scroll Y position for the grid.
     */
    get maxScrollY(): number;
    /**
     * The viewport widget for the data grid.
     */
    get viewport(): Widget;
    /**
     * The cell editor controller object for the data grid.
     */
    get editorController(): ICellEditorController | null;
    set editorController(controller: ICellEditorController | null);
    /**
     * Whether the cell editing is enabled for the data grid.
     */
    get editingEnabled(): boolean;
    set editingEnabled(enabled: boolean);
    /**
     * Whether the grid cells are editable.
     *
     * `editingEnabled` flag must be on and grid must have required
     * selection model, editor controller and data model properties.
     */
    get editable(): boolean;
    /**
     * The rendering context for painting the data grid.
     */
    protected get canvasGC(): CanvasRenderingContext2D;
    /**
     * The row sections of the data grid.
     */
    protected get rowSections(): SectionList;
    /**
     * The column sections of the data grid.
     */
    protected get columnSections(): SectionList;
    /**
     * The row header sections of the data grid.
     */
    protected get rowHeaderSections(): SectionList;
    /**
     * The column header sections of the data grid.
     */
    protected get columnHeaderSections(): SectionList;
    /**
     * Scroll the grid to the specified row.
     *
     * @param row - The row index of the cell.
     *
     * #### Notes
     * This is a no-op if the row is already visible.
     */
    scrollToRow(row: number): void;
    /**
     * Scroll the grid to the specified column.
     *
     * @param column - The column index of the cell.
     *
     * #### Notes
     * This is a no-op if the column is already visible.
     */
    scrollToColumn(column: number): void;
    /**
     * Scroll the grid to the specified cell.
     *
     * @param row - The row index of the cell.
     *
     * @param column - The column index of the cell.
     *
     * #### Notes
     * This is a no-op if the cell is already visible.
     */
    scrollToCell(row: number, column: number): void;
    /**
     * Move cursor down/up/left/right while making sure it remains
     * within the bounds of selected rectangles
     *
     * @param direction - The direction of the movement.
     */
    moveCursor(direction: SelectionModel.CursorMoveDirection): void;
    /**
     * Scroll the grid to the current cursor position.
     *
     * #### Notes
     * This is a no-op if the cursor is already visible or
     * if there is no selection model installed on the grid.
     */
    scrollToCursor(): void;
    /**
     * Scroll the viewport by the specified amount.
     *
     * @param dx - The X scroll amount.
     *
     * @param dy - The Y scroll amount.
     */
    scrollBy(dx: number, dy: number): void;
    /**
     * Scroll the viewport by one page.
     *
     * @param dir - The desired direction of the scroll.
     */
    scrollByPage(dir: 'up' | 'down' | 'left' | 'right'): void;
    /**
     * Scroll the viewport by one cell-aligned step.
     *
     * @param dir - The desired direction of the scroll.
     */
    scrollByStep(dir: 'up' | 'down' | 'left' | 'right'): void;
    /**
     * Scroll to the specified offset position.
     *
     * @param x - The desired X position.
     *
     * @param y - The desired Y position.
     */
    scrollTo(x: number, y: number): void;
    /**
     * Get the row count for a particular region in the data grid.
     *
     * @param region - The row region of interest.
     *
     * @returns The row count for the specified region.
     */
    rowCount(region: DataModel.RowRegion): number;
    /**
     * Get the column count for a particular region in the data grid.
     *
     * @param region - The column region of interest.
     *
     * @returns The column count for the specified region.
     */
    columnCount(region: DataModel.ColumnRegion): number;
    /**
     * Get the row at a virtual offset in the data grid.
     *
     * @param region - The region which holds the row of interest.
     *
     * @param offset - The virtual offset of the row of interest.
     *
     * @returns The index of the row, or `-1` if the offset is out of range.
     *
     * #### Notes
     * This method accounts for a stretched last row.
     */
    rowAt(region: DataModel.RowRegion, offset: number): number;
    /**
     * Get the column at a virtual offset in the data grid.
     *
     * @param region - The region which holds the column of interest.
     *
     * @param offset - The virtual offset of the column of interest.
     *
     * @returns The index of the column, or `-1` if the offset is out of range.
     *
     * #### Notes
     * This method accounts for a stretched last column.
     */
    columnAt(region: DataModel.ColumnRegion, offset: number): number;
    /**
     * Get the offset of a row in the data grid.
     *
     * @param region - The region which holds the row of interest.
     *
     * @param index - The index of the row of interest.
     *
     * @returns The offset of the row, or `-1` if the index is out of range.
     *
     * #### Notes
     * A stretched last row has no effect on the return value.
     */
    rowOffset(region: DataModel.RowRegion, index: number): number;
    /**
     * Get the offset of a column in the data grid.
     *
     * @param region - The region which holds the column of interest.
     *
     * @param index - The index of the column of interest.
     *
     * @returns The offset of the column, or `-1` if the index is out of range.
     *
     * #### Notes
     * A stretched last column has no effect on the return value.
     */
    columnOffset(region: DataModel.ColumnRegion, index: number): number;
    /**
     * Get the size of a row in the data grid.
     *
     * @param region - The region which holds the row of interest.
     *
     * @param index - The index of the row of interest.
     *
     * @returns The size of the row, or `-1` if the index is out of range.
     *
     * #### Notes
     * This method accounts for a stretched last row.
     */
    rowSize(region: DataModel.RowRegion, index: number): number;
    /**
     * Get the size of a column in the data grid.
     *
     * @param region - The region which holds the column of interest.
     *
     * @param index - The index of the column of interest.
     *
     * @returns The size of the column, or `-1` if the index is out of range.
     *
     * #### Notes
     * This method accounts for a stretched last column.
     */
    columnSize(region: DataModel.ColumnRegion, index: number): number;
    /**
     * Resize a row in the data grid.
     *
     * @param region - The region which holds the row of interest.
     *
     * @param index - The index of the row of interest.
     *
     * @param size - The desired size of the row.
     */
    resizeRow(region: DataModel.RowRegion, index: number, size: number): void;
    /**
     * Resize a column in the data grid.
     *
     * @param region - The region which holds the column of interest.
     *
     * @param index - The index of the column of interest.
     *
     * @param size - The desired size of the column.
     */
    resizeColumn(region: DataModel.ColumnRegion, index: number, size: number): void;
    /**
     * Reset modified rows to their default size.
     *
     * @param region - The row region of interest.
     */
    resetRows(region: DataModel.RowRegion | 'all'): void;
    /**
     * Reset modified columns to their default size.
     *
     * @param region - The column region of interest.
     */
    resetColumns(region: DataModel.ColumnRegion | 'all'): void;
    /**
     * Auto sizes column widths based on their text content.
     * @param area which area to resize: 'body', 'row-header' or 'all'.
     * @param padding padding added to resized columns (pixels).
     * @param numCols specify cap on the number of column resizes (optional).
     */
    fitColumnNames(area?: DataGrid.ColumnFitType, padding?: number, numCols?: number): void;
    /**
     * Map a client position to local viewport coordinates.
     *
     * @param clientX - The client X position of the mouse.
     *
     * @param clientY - The client Y position of the mouse.
     *
     * @returns The local viewport coordinates for the position.
     */
    mapToLocal(clientX: number, clientY: number): {
        lx: number;
        ly: number;
    };
    /**
     * Map a client position to virtual grid coordinates.
     *
     * @param clientX - The client X position of the mouse.
     *
     * @param clientY - The client Y position of the mouse.
     *
     * @returns The virtual grid coordinates for the position.
     */
    mapToVirtual(clientX: number, clientY: number): {
        vx: number;
        vy: number;
    };
    /**
     * Hit test the viewport for the given client position.
     *
     * @param clientX - The client X position of the mouse.
     *
     * @param clientY - The client Y position of the mouse.
     *
     * @returns The hit test result, or `null` if the client
     *   position is out of bounds.
     *
     * #### Notes
     * This method accounts for a stretched last row and/or column.
     */
    hitTest(clientX: number, clientY: number): DataGrid.HitTestResult;
    /**
     * Copy the current selection to the system clipboard.
     *
     * #### Notes
     * The grid must have a data model and a selection model.
     *
     * The behavior can be configured via `DataGrid.copyConfig`.
     */
    copyToClipboard(): void;
    /**
     * Process a message sent to the widget.
     *
     * @param msg - The message sent to the widget.
     */
    processMessage(msg: Message): void;
    /**
     * Intercept a message sent to a message handler.
     *
     * @param handler - The target handler of the message.
     *
     * @param msg - The message to be sent to the handler.
     *
     * @returns `true` if the message should continue to be processed
     *   as normal, or `false` if processing should cease immediately.
     */
    messageHook(handler: IMessageHandler, msg: Message): boolean;
    /**
     * Handle the DOM events for the data grid.
     *
     * @param event - The DOM event sent to the data grid.
     *
     * #### Notes
     * This method implements the DOM `EventListener` interface and is
     * called in response to events on the data grid's DOM node. It
     * should not be called directly by user code.
     */
    handleEvent(event: Event): void;
    /**
     * A message handler invoked on an `'activate-request'` message.
     */
    protected onActivateRequest(msg: Message): void;
    /**
     * A message handler invoked on a `'before-attach'` message.
     */
    protected onBeforeAttach(msg: Message): void;
    /**
     * A message handler invoked on an `'after-detach'` message.
     */
    protected onAfterDetach(msg: Message): void;
    /**
     * A message handler invoked on a `'before-show'` message.
     */
    protected onBeforeShow(msg: Message): void;
    /**
     * A message handler invoked on a `'resize'` message.
     */
    protected onResize(msg: Widget.ResizeMessage): void;
    /**
     * Schedule a repaint of all of the grid content.
     */
    protected repaintContent(): void;
    /**
     * Schedule a repaint of specific grid content.
     */
    protected repaintRegion(region: DataModel.CellRegion, r1: number, c1: number, r2: number, c2: number): void;
    /**
     * Schedule a repaint of the overlay.
     */
    protected repaintOverlay(): void;
    /**
     * Ensure the canvas is at least the specified size.
     *
     * This method will retain the valid canvas content.
     */
    private _resizeCanvasIfNeeded;
    /**
     * Sync the scroll bars and scroll state with the viewport.
     *
     * #### Notes
     * If the visibility of either scroll bar changes, a synchronous
     * fit-request will be dispatched to the data grid to immediately
     * resize the viewport.
     */
    private _syncScrollState;
    /**
     * Sync the viewport to the given scroll position.
     *
     * #### Notes
     * This schedules a full repaint and syncs the scroll state.
     */
    private _syncViewport;
    /**
     * Process a message sent to the viewport
     */
    private _processViewportMessage;
    /**
     * A message hook invoked on a viewport `'resize'` message.
     */
    private _onViewportResize;
    /**
     * A message hook invoked on a viewport `'scroll-request'` message.
     */
    private _onViewportScrollRequest;
    /**
     * A message hook invoked on a viewport `'paint-request'` message.
     */
    private _onViewportPaintRequest;
    /**
     * A message hook invoked on a viewport `'overlay-paint-request'` message.
     */
    private _onViewportOverlayPaintRequest;
    /**
     * A message hook invoked on a viewport `'row-resize-request'` message.
     */
    private _onViewportRowResizeRequest;
    /**
     * A message hook invoked on a viewport `'column-resize-request'` message.
     */
    private _onViewportColumnResizeRequest;
    /**
     * Handle the `thumbMoved` signal from a scroll bar.
     */
    private _onThumbMoved;
    /**
     * Handle the `pageRequested` signal from a scroll bar.
     */
    private _onPageRequested;
    /**
     * Handle the `stepRequested` signal from a scroll bar.
     */
    private _onStepRequested;
    /**
     * A signal handler for the data model `changed` signal.
     */
    private _onDataModelChanged;
    /**
     * A signal handler for the selection model `changed` signal.
     */
    private _onSelectionsChanged;
    /**
     * Handle rows being inserted in the data model.
     */
    private _onRowsInserted;
    /**
     * Handle columns being inserted into the data model.
     */
    private _onColumnsInserted;
    /**
     * Handle rows being removed from the data model.
     */
    private _onRowsRemoved;
    /**
     * Handle columns being removed from the data model.
     */
    private _onColumnsRemoved;
    /**
     * Handle rows moving in the data model.
     */
    private _onRowsMoved;
    /**
     * Handle columns moving in the data model.
     */
    private _onColumnsMoved;
    /**
     * Handle cells changing in the data model.
     */
    private _onCellsChanged;
    /**
     * Handle a full data model reset.
     */
    private _onModelReset;
    /**
     * A signal handler for the renderer map `changed` signal.
     */
    private _onRenderersChanged;
    /**
     * Handle the `'keydown'` event for the data grid.
     */
    private _evtKeyDown;
    /**
     * Handle the `'mousedown'` event for the data grid.
     */
    private _evtMouseDown;
    /**
     * Handle the `'mousemove'` event for the data grid.
     */
    private _evtMouseMove;
    /**
     * Handle the `'mouseup'` event for the data grid.
     */
    private _evtMouseUp;
    /**
     * Handle the `'dblclick'` event for the data grid.
     */
    private _evtMouseDoubleClick;
    /**
     * Handle the `'mouseleave'` event for the data grid.
     */
    private _evtMouseLeave;
    /**
     * Handle the `'contextmenu'` event for the data grid.
     */
    private _evtContextMenu;
    /**
     * Handle the `'wheel'` event for the data grid.
     */
    private _evtWheel;
    /**
     * Release the mouse grab.
     */
    private _releaseMouse;
    /**
     * Refresh the dpi ratio.
     */
    private _refreshDPI;
    /**
     * Resize a row section immediately.
     */
    private _resizeRow;
    /**
     * Resize a column section immediately.
     */
    private _resizeColumn;
    /**
     * Resize a row header section immediately.
     */
    private _resizeRowHeader;
    /**
     * Resize a column header section immediately.
     */
    private _resizeColumnHeader;
    /**
     * Scroll immediately to the specified offset position.
     */
    private _scrollTo;
    /**
     * Blit content into the on-screen grid canvas.
     *
     * The rect should be expressed in viewport coordinates.
     *
     * This automatically accounts for the dpi ratio.
     */
    private _blitContent;
    /**
     * Paint the grid content for the given dirty rect.
     *
     * The rect should be expressed in valid viewport coordinates.
     *
     * This is the primary paint entry point. The individual `_draw*`
     * methods should not be invoked directly. This method dispatches
     * to the drawing methods in the correct order.
     */
    protected paintContent(rx: number, ry: number, rw: number, rh: number): void;
    /**
     * Resizes body column headers so their text fits
     * without clipping or wrapping.
     * @param dataModel
     */
    private _fitBodyColumnHeaders;
    /**
     * Resizes row header columns so their text fits
     * without clipping or wrapping.
     * @param dataModel
     */
    private _fitRowColumnHeaders;
    /**
     * Paint the overlay content for the entire grid.
     *
     * This is the primary overlay paint entry point. The individual
     * `_draw*` methods should not be invoked directly. This method
     * dispatches to the drawing methods in the correct order.
     */
    private _paintOverlay;
    /**
     * Draw the void region for the dirty rect.
     */
    private _drawVoidRegion;
    /**
     * Draw the body region which intersects the dirty rect.
     */
    private _drawBodyRegion;
    /**
     * Draw the row header region which intersects the dirty rect.
     */
    private _drawRowHeaderRegion;
    /**
     * Draw the column header region which intersects the dirty rect.
     */
    private _drawColumnHeaderRegion;
    /**
     * Draw the corner header region which intersects the dirty rect.
     */
    protected drawCornerHeaderRegion(rx: number, ry: number, rw: number, rh: number): void;
    /**
     * Draw the background for the given paint region.
     */
    private _drawBackground;
    /**
     * Draw the row background for the given paint region.
     */
    private _drawRowBackground;
    /**
     * Draw the column background for the given paint region.
     */
    private _drawColumnBackground;
    /**
     * Returns column size
     * @param region
     * @param index
     */
    private _getColumnSize;
    /**
     * Returns row size
     * @param region
     * @param index
     */
    private _getRowSize;
    /**
     * Draw the cells for the given paint region.
     */
    private _drawCells;
    private cellGroupInteresectsRegion;
    /**
     * Paint group cells.
     */
    private _paintMergedCells;
    /**
     * Draw the horizontal grid lines for the given paint region.
     */
    private _drawHorizontalGridLines;
    /**
     * Draw the vertical grid lines for the given paint region.
     */
    private _drawVerticalGridLines;
    /**
     * Draw the body selections for the data grid.
     */
    private _drawBodySelections;
    /**
     * Draw the row header selections for the data grid.
     */
    private _drawRowHeaderSelections;
    /**
     * Draw the column header selections for the data grid.
     */
    private _drawColumnHeaderSelections;
    /**
     * Draw the overlay cursor for the data grid.
     */
    private _drawCursor;
    /**
     * Draw the overlay shadows for the data grid.
     */
    private _drawShadows;
    private _viewport;
    private _vScrollBar;
    private _hScrollBar;
    private _scrollCorner;
    private _scrollX;
    private _scrollY;
    private _viewportWidth;
    private _viewportHeight;
    private _mousedown;
    private _keyHandler;
    private _mouseHandler;
    private _vScrollBarMinWidth;
    private _hScrollBarMinHeight;
    private _dpiRatio;
    private _canvas;
    private _buffer;
    private _overlay;
    private _canvasGC;
    private _bufferGC;
    private _overlayGC;
    private _rowSections;
    private _columnSections;
    private _rowHeaderSections;
    private _columnHeaderSections;
    private _dataModel;
    private _selectionModel;
    private _stretchLastRow;
    private _stretchLastColumn;
    private _style;
    private _cellRenderers;
    private _copyConfig;
    private _headerVisibility;
    private _editorController;
    private _editingEnabled;
}
/**
 * The namespace for the `DataGrid` class statics.
 */
export declare namespace DataGrid {
    /**
     * An object which defines the style for a data grid.
     *
     * #### Notes
     * All style colors support the full CSS color syntax.
     */
    type Style = {
        /**
         * The void color for the data grid.
         *
         * This is the base fill color for the entire data grid.
         */
        readonly voidColor?: string;
        /**
         * The background color for the body cells.
         *
         * This color is layered on top of the `voidColor`.
         */
        readonly backgroundColor?: string;
        /**
         * A function which returns the background color for a row.
         *
         * This color is layered on top of the `backgroundColor` and can
         * be used to implement "zebra striping" of the grid rows.
         */
        readonly rowBackgroundColor?: (index: number) => string;
        /**
         * A function which returns the background color for a column.
         *
         * This color is layered on top of the `backgroundColor` and can
         * be used to implement "zebra striping" of the grid columns.
         */
        readonly columnBackgroundColor?: (index: number) => string;
        /**
         * The color for the grid lines of the body cells.
         *
         * The grid lines are draw on top of the cell contents.
         */
        readonly gridLineColor?: string;
        /**
         * The color for the vertical grid lines of the body cells.
         *
         * This overrides the `gridLineColor` option.
         */
        readonly verticalGridLineColor?: string;
        /**
         * The color for the horizontal grid lines of the body cells.
         *
         * This overrides the `gridLineColor` option.
         */
        readonly horizontalGridLineColor?: string;
        /**
         * The background color for the header cells.
         *
         * This color is layered on top of the `voidColor`.
         */
        readonly headerBackgroundColor?: string;
        /**
         * The color for the grid lines of the header cells.
         *
         * The grid lines are draw on top of the cell contents.
         */
        readonly headerGridLineColor?: string;
        /**
         * The color for the vertical grid lines of the header cells.
         *
         * This overrides the `headerGridLineColor` option.
         */
        readonly headerVerticalGridLineColor?: string;
        /**
         * The color for the horizontal grid lines of the header cells.
         *
         * This overrides the `headerGridLineColor` option.
         */
        readonly headerHorizontalGridLineColor?: string;
        /**
         * The fill color for a selection.
         */
        readonly selectionFillColor?: string;
        /**
         * The border color for a selection.
         */
        readonly selectionBorderColor?: string;
        /**
         * The fill color for the cursor.
         */
        readonly cursorFillColor?: string;
        /**
         * The border color for the cursor.
         */
        readonly cursorBorderColor?: string;
        /**
         * The fill color for a header selection.
         */
        readonly headerSelectionFillColor?: string;
        /**
         * The border color for a header selection.
         */
        readonly headerSelectionBorderColor?: string;
        /**
         * The drop shadow effect when the grid is scrolled.
         */
        readonly scrollShadow?: {
            /**
             * The size of the shadow, in pixels.
             */
            readonly size: number;
            /**
             * The first color stop for the shadow.
             */
            readonly color1: string;
            /**
             * The second color stop for the shadow.
             */
            readonly color2: string;
            /**
             * The third color stop for the shadow.
             */
            readonly color3: string;
        };
    };
    /**
     * An object which defines the default sizes for a data grid.
     */
    type DefaultSizes = {
        /**
         * The default height of a row.
         */
        readonly rowHeight: number;
        /**
         * The default width of a column.
         */
        readonly columnWidth: number;
        /**
         * The default width of a row header.
         */
        readonly rowHeaderWidth: number;
        /**
         * The default height of a column header.
         */
        readonly columnHeaderHeight: number;
    };
    /**
     * An object which defines the minimum sizes for a data grid.
     */
    type MinimumSizes = {
        /**
         * The minimum height of a row.
         */
        readonly rowHeight: number;
        /**
         * The minimum width of a column.
         */
        readonly columnWidth: number;
        /**
         * The minimum width of a row header.
         */
        readonly rowHeaderWidth: number;
        /**
         * The minimum height of a column header.
         */
        readonly columnHeaderHeight: number;
    };
    /**
     * A type alias for the supported header visibility modes.
     */
    type HeaderVisibility = 'all' | 'row' | 'column' | 'none';
    /**
     * A type alias for the supported column auto resize modes.
     */
    type ColumnFitType = 'all' | 'row-header' | 'body';
    /**
     * A type alias for the arguments to a copy format function.
     */
    type CopyFormatArgs = {
        /**
         * The cell region for the value.
         */
        region: DataModel.CellRegion;
        /**
         * The row index of the value.
         */
        row: number;
        /**
         * The column index of the value.
         */
        column: number;
        /**
         * The value for the cell.
         */
        value: any;
        /**
         * The metadata for the cell.
         */
        metadata: DataModel.Metadata;
    };
    /**
     * A type alias for a copy format function.
     */
    type CopyFormatFunc = (args: CopyFormatArgs) => string;
    /**
     * A type alias for the data grid copy config.
     */
    type CopyConfig = {
        /**
         * The separator to use between values.
         */
        readonly separator: string;
        /**
         * The headers to included in the copied output.
         */
        readonly headers: 'none' | 'row' | 'column' | 'all';
        /**
         * The function for formatting the data values.
         */
        readonly format: CopyFormatFunc;
        /**
         * The cell count threshold for a copy to be considered "large".
         */
        readonly warningThreshold: number;
    };
    /**
     * An options object for initializing a data grid.
     */
    interface IOptions {
        /**
         * The style for the data grid.
         *
         * The default is `DataGrid.defaultStyle`.
         */
        style?: Style;
        /**
         * The default sizes for the data grid.
         *
         * The default is `DataGrid.defaultSizes`.
         */
        defaultSizes?: DefaultSizes;
        /**
         * The minimum sizes for the data grid.
         *
         * The default is `DataGrid.minimumSizes`.
         */
        minimumSizes?: MinimumSizes;
        /**
         * The header visibility for the data grid.
         *
         * The default is `'all'`.
         */
        headerVisibility?: HeaderVisibility;
        /**
         * The cell renderer map for the data grid.
         *
         * The default is an empty renderer map.
         */
        cellRenderers?: RendererMap;
        /**
         * The default cell renderer for the data grid.
         *
         * The default is a new `TextRenderer`.
         */
        defaultRenderer?: CellRenderer;
        /**
         * The copy configuration data for the grid.
         *
         * The default is `DataGrid.defaultCopyConfig`.
         */
        copyConfig?: CopyConfig;
        /**
         * Whether to stretch the last row of the grid.
         *
         * The default is `false`.
         */
        stretchLastRow?: boolean;
        /**
         * Whether to stretch the last column of the grid.
         *
         * The default is `false`.
         */
        stretchLastColumn?: boolean;
    }
    /**
     * An object which handles keydown events for the data grid.
     */
    interface IKeyHandler extends IDisposable {
        /**
         * Handle the key down event for the data grid.
         *
         * @param grid - The data grid of interest.
         *
         * @param event - The keydown event of interest.
         *
         * #### Notes
         * This will not be called if the mouse button is pressed.
         */
        onKeyDown(grid: DataGrid, event: KeyboardEvent): void;
    }
    /**
     * An object which handles mouse events for the data grid.
     */
    interface IMouseHandler extends IDisposable {
        /**
         * Release any resources acquired during a mouse press.
         *
         * #### Notes
         * This method is called when the mouse should be released
         * independent of a mouseup event, such as an early detach.
         */
        release(): void;
        /**
         * Handle the mouse hover event for the data grid.
         *
         * @param grid - The data grid of interest.
         *
         * @param event - The mouse hover event of interest.
         */
        onMouseHover(grid: DataGrid, event: MouseEvent): void;
        /**
         * Handle the mouse leave event for the data grid.
         *
         * @param grid - The data grid of interest.
         *
         * @param event - The mouse hover event of interest.
         */
        onMouseLeave(grid: DataGrid, event: MouseEvent): void;
        /**
         * Handle the mouse down event for the data grid.
         *
         * @param grid - The data grid of interest.
         *
         * @param event - The mouse down event of interest.
         */
        onMouseDown(grid: DataGrid, event: MouseEvent): void;
        /**
         * Handle the mouse move event for the data grid.
         *
         * @param grid - The data grid of interest.
         *
         * @param event - The mouse move event of interest.
         */
        onMouseMove(grid: DataGrid, event: MouseEvent): void;
        /**
         * Handle the mouse up event for the data grid.
         *
         * @param grid - The data grid of interest.
         *
         * @param event - The mouse up event of interest.
         */
        onMouseUp(grid: DataGrid, event: MouseEvent): void;
        /**
         * Handle the mouse double click event for the data grid.
         *
         * @param grid - The data grid of interest.
         *
         * @param event - The mouse double click event of interest.
         */
        onMouseDoubleClick(grid: DataGrid, event: MouseEvent): void;
        /**
         * Handle the context menu event for the data grid.
         *
         * @param grid - The data grid of interest.
         *
         * @param event - The context menu event of interest.
         */
        onContextMenu(grid: DataGrid, event: MouseEvent): void;
        /**
         * Handle the wheel event for the data grid.
         *
         * @param grid - The data grid of interest.
         *
         * @param event - The wheel event of interest.
         */
        onWheel(grid: DataGrid, event: WheelEvent): void;
    }
    /**
     * An object which holds the result of a grid hit test.
     */
    type HitTestResult = {
        /**
         * The region of the data grid that was hit.
         */
        readonly region: DataModel.CellRegion | 'void';
        /**
         * The row index of the cell that was hit.
         *
         * This is `-1` for the `void` region.
         */
        readonly row: number;
        /**
         * The column index of the cell that was hit.
         *
         * This is `-1` for the `void` region.
         */
        readonly column: number;
        /**
         * The X coordinate of the mouse in cell coordinates.
         *
         * This is `-1` for the `void` region.
         */
        readonly x: number;
        /**
         * The Y coordinate of the mouse in cell coordinates.
         *
         * This is `-1` for the `void` region.
         */
        readonly y: number;
        /**
         * The width of the cell.
         *
         * This is `-1` for the `void` region.
         */
        readonly width: number;
        /**
         * The height of the cell.
         *
         * This is `-1` for the `void` region.
         */
        readonly height: number;
    };
    /**
     * A generic format function for the copy handler.
     *
     * @param args - The format args for the function.
     *
     * @returns The string representation of the value.
     *
     * #### Notes
     * This function uses `String()` to coerce a value to a string.
     */
    function copyFormatGeneric(args: CopyFormatArgs): string;
    /**
     * The default theme for a data grid.
     */
    const defaultStyle: Style;
    /**
     * The default sizes for a data grid.
     */
    const defaultSizes: DefaultSizes;
    /**
     * The default minimum sizes for a data grid.
     */
    const minimumSizes: MinimumSizes;
    /**
     * The default copy config for a data grid.
     */
    const defaultCopyConfig: CopyConfig;
}
