import { DataModel } from './datamodel';
/**
 * An interface describing a merged cell group.
 * r1: start row
 * r2: end row
 * c1: start column
 * c2: end column
 */
export interface CellGroup {
    r1: number;
    r2: number;
    c1: number;
    c2: number;
}
/**
 * A collection of helper functions relating to merged cell groups
 */
export declare namespace CellGroup {
    /**
     * Checks if two cell-groups are intersecting
     * in the given axis.
     * @param group1
     * @param group2
     * @param axis
     */
    function areCellGroupsIntersectingAtAxis(group1: CellGroup, group2: CellGroup, axis: 'row' | 'column'): boolean;
    /**
     * Checks if cell-groups are intersecting.
     * @param group1
     * @param group2
     */
    function areCellGroupsIntersecting(group1: CellGroup, group2: CellGroup): boolean;
    /**
     * Retrieves the index of the cell-group to which
     * the cell at the given row, column belongs.
     * @param dataModel
     * @param rgn
     * @param row
     * @param column
     */
    function getGroupIndex(dataModel: DataModel, rgn: DataModel.CellRegion, row: number, column: number): number;
    /**
     * Returns a cell-group for the given row/index coordinates.
     * @param dataModel
     * @param rgn
     * @param row
     * @param column
     */
    function getGroup(dataModel: DataModel, rgn: DataModel.CellRegion, row: number, column: number): CellGroup | null;
    /**
     * Returns all cell groups which belong to
     * a given cell cell region.
     * @param dataModel
     * @param rgn
     */
    function getCellGroupsAtRegion(dataModel: DataModel, rgn: DataModel.CellRegion): CellGroup[];
    /**
     * Calculates and returns a merged cell-group from
     * two cell-group objects.
     * @param groups
     */
    function joinCellGroups(groups: CellGroup[]): CellGroup;
    /**
     * Merges a cell group with other cells groups in the
     * same region if they intersect.
     * @param dataModel the data model of the grid.
     * @param group the target cell group.
     * @param region the region of the cell group.
     * @returns a new cell group after merging has happened.
     */
    function joinCellGroupWithMergedCellGroups(dataModel: DataModel, group: CellGroup, region: DataModel.CellRegion): CellGroup;
    /**
     * Retrieves a list of cell groups intersecting at
     * a given row.
     * @param dataModel data model of the grid.
     * @param rgn the cell region.
     * @param row the target row to look for intersections at.
     * @returns all cell groups intersecting with the row.
     */
    function getCellGroupsAtRow(dataModel: DataModel, rgn: DataModel.CellRegion, row: number): CellGroup[];
    /**
     * Retrieves a list of cell groups intersecting at
     * a given column.
     * @param dataModel data model of the grid.
     * @param rgn the cell region.
     * @param column the target column to look for intersections at.
     * @returns all cell groups intersecting with the column.
     */
    function getCellGroupsAtColumn(dataModel: DataModel, rgn: DataModel.CellRegion, column: number): CellGroup[];
    /**
     * Merges a target cell group with any cell groups
     * it intersects with at a given row or column.
     * @param dataModel data model of the grid.
     * @param regions list of cell regions.
     * @param axis row or column.
     * @param group the target cell group.
     * @returns a new merged cell group.
     */
    function joinCellGroupsIntersectingAtAxis(dataModel: DataModel, regions: DataModel.CellRegion[], axis: 'row' | 'column', group: CellGroup): CellGroup;
}
