import { CellDataType, CellEditor, ICellEditOptions, ICellEditor } from './celleditor';
import { DataModel } from './datamodel';
/**
 * A type alias for cell editor override identifier.
 */
export type EditorOverrideIdentifier = CellDataType | DataModel.Metadata | 'default';
/**
 * An object which manages cell editing.
 */
export interface ICellEditorController {
    /**
     * Override cell editor for the cells matching the identifier.
     *
     * @param identifier - Cell identifier to use when matching cells.
     * if identifier is a CellDataType, then cell matching is done using data type of the cell,
     * if identifier is a Metadata, then partial match of the cell metadata with identifier is used for match,
     * if identifier is 'default' then override is used as default editor when no other editor is found suitable
     *
     * @param editor - The cell editor to use or resolver to use to get an editor for matching cells.
     */
    setEditor(identifier: EditorOverrideIdentifier, editor: ICellEditor | Resolver): void;
    /**
     * Start editing a cell.
     *
     * @param cell - The object holding cell configuration data.
     *
     * @param options - The cell editing options.
     */
    edit(cell: CellEditor.CellConfig, options?: ICellEditOptions): boolean;
    /**
     * Cancel editing.
     */
    cancel(): void;
}
/**
 * A type alias for a cell editor config function.
 *
 * This type is used to compute a value from a cell config object.
 */
export type ConfigFunc<T> = (config: CellEditor.CellConfig) => T;
/**
 * A type alias for a cell editor config option.
 *
 * A config option can be a static value or a config function.
 */
export type ConfigOption<T> = T | ConfigFunc<T>;
/**
 * A type alias for a cell editor resolver function.
 */
export type Resolver = ConfigFunc<ICellEditor | undefined>;
/**
 * Resolve a config option for a cell editor.
 *
 * @param option - The config option to resolve.
 *
 * @param config - The cell config object.
 *
 * @returns The resolved value for the option.
 */
export declare function resolveOption<T>(option: ConfigOption<T>, config: CellEditor.CellConfig): T;
/**
 * An object which manages cell editing. It stores editor overrides,
 * decides which editor to use for a cell, makes sure there is only one editor active.
 */
export declare class CellEditorController implements ICellEditorController {
    /**
     * Override cell editor for the cells matching the identifier.
     *
     * @param identifier - Cell identifier to use when matching cells.
     * if identifier is a CellDataType, then cell matching is done using data type of the cell,
     * if identifier is a Metadata, then partial match of the cell metadata with identifier is used for match,
     * if identifier is 'default' then override is used as default editor when no other editor is found suitable
     *
     * @param editor - The cell editor to use or resolver to use to get an editor for matching cells.
     */
    setEditor(identifier: EditorOverrideIdentifier, editor: ICellEditor | Resolver): void;
    /**
     * Start editing a cell.
     *
     * @param cell - The object holding cell configuration data.
     *
     * @param options - The cell editing options.
     */
    edit(cell: CellEditor.CellConfig, options?: ICellEditOptions): boolean;
    /**
     * Cancel editing.
     */
    cancel(): void;
    private _onCommit;
    private _onCancel;
    private _getDataTypeKey;
    private _objectToKey;
    private _metadataIdentifierToKey;
    private _metadataMatchesIdentifier;
    private _getMetadataBasedEditor;
    /**
     * Choose the most appropriate cell editor to use based on overrides / cell data type.
     *
     * If no match is found in overrides or based on cell data type, and if cell has a primitive
     * data type then TextCellEditor is used as default cell editor. If 'default' cell editor
     * is overridden, then it is used instead of TextCellEditor for default.
     */
    private _getEditor;
    private _editor;
    private _cell;
    private _typeBasedOverrides;
    private _metadataBasedOverrides;
}
