import { IDisposable } from '@lumino/disposable';
import { Message } from '@lumino/messaging';
import { Widget } from '@lumino/widgets';
import { DataGrid } from './datagrid';
import { SelectionModel } from './selectionmodel';
import { Signal } from '@lumino/signaling';
/**
 * A response object returned from cell input validator
 */
export interface ICellInputValidatorResponse {
    /**
     * Flag indicating cell input is valid or not
     */
    valid: boolean;
    /**
     * Validation error message. Set only when input is invalid
     */
    message?: string;
}
/**
 * An object which validates cell input values.
 */
export interface ICellInputValidator {
    /**
     * Validate cell input.
     *
     * @param cell - The object holding cell configuration data.
     *
     * @param value - The cell value input.
     *
     * @returns An object with validation result.
     */
    validate(cell: CellEditor.CellConfig, value: any): ICellInputValidatorResponse;
}
/**
 * An object returned from cell editor after a successful edit.
 */
export interface ICellEditResponse {
    /**
     * An object which holds the configuration data for a cell.
     */
    cell: CellEditor.CellConfig;
    /**
     * Value input.
     */
    value: any;
    /**
     * Cursor move direction based on keys pressed to end the edit.
     */
    cursorMovement: SelectionModel.CursorMoveDirection;
}
/**
 * An object implementing cell editing.
 */
export interface ICellEditor {
    /**
     * Start editing the cell.
     *
     * @param cell - The object holding cell configuration data.
     *
     * @param options - The cell editing options.
     */
    edit(cell: CellEditor.CellConfig, options?: ICellEditOptions): void;
    /**
     * Cancel editing the cell.
     */
    cancel(): void;
}
export type CellDataType = 'string' | 'number' | 'integer' | 'boolean' | 'date' | 'string:option' | 'number:option' | 'integer:option' | 'date:option' | 'string:dynamic-option' | 'number:dynamic-option' | 'integer:dynamic-option' | 'date:dynamic-option';
/**
 * An object containing cell editing options.
 */
export interface ICellEditOptions {
    /**
     * Cell editor to use for editing.
     *
     * #### Notes
     * This object is only used by cell editor controller.
     * If not set, controller picks the most suitable editor
     * for the particular cell configuration.
     */
    editor?: ICellEditor;
    /**
     * Cell input validator to use for value validation.
     */
    validator?: ICellInputValidator;
    /**
     * Callback method to call on cell edit commit.
     */
    onCommit?: (response: ICellEditResponse) => void;
    /**
     * Callback method to call on cell edit cancel.
     */
    onCancel?: () => void;
}
/**
 * A cell input validator object which always returns valid.
 */
export declare class PassInputValidator implements ICellInputValidator {
    /**
     * Validate cell input.
     *
     * @param cell - The object holding cell configuration data.
     *
     * @param value - The cell value input.
     *
     * @returns An object with validation result.
     */
    validate(cell: CellEditor.CellConfig, value: unknown): ICellInputValidatorResponse;
}
/**
 * Text cell input validator.
 */
export declare class TextInputValidator implements ICellInputValidator {
    /**
     * Validate cell input.
     *
     * @param cell - The object holding cell configuration data.
     *
     * @param value - The cell value input.
     *
     * @returns An object with validation result.
     */
    validate(cell: CellEditor.CellConfig, value: string): ICellInputValidatorResponse;
    /**
     * Minimum text length
     *
     * The default is Number.NaN, meaning no minimum constraint
     */
    minLength: number;
    /**
     * Maximum text length
     *
     * The default is Number.NaN, meaning no maximum constraint
     */
    maxLength: number;
    /**
     * Required text pattern as regular expression
     *
     * The default is null, meaning no pattern constraint
     */
    pattern: RegExp | null;
}
/**
 * Integer cell input validator.
 */
export declare class IntegerInputValidator implements ICellInputValidator {
    /**
     * Validate cell input.
     *
     * @param cell - The object holding cell configuration data.
     *
     * @param value - The cell value input.
     *
     * @returns An object with validation result.
     */
    validate(cell: CellEditor.CellConfig, value: number): ICellInputValidatorResponse;
    /**
     * Minimum value
     *
     * The default is Number.NaN, meaning no minimum constraint
     */
    min: number;
    /**
     * Maximum value
     *
     * The default is Number.NaN, meaning no maximum constraint
     */
    max: number;
}
/**
 * Real number cell input validator.
 */
export declare class NumberInputValidator implements ICellInputValidator {
    /**
     * Validate cell input.
     *
     * @param cell - The object holding cell configuration data.
     *
     * @param value - The cell value input.
     *
     * @returns An object with validation result.
     */
    validate(cell: CellEditor.CellConfig, value: number): ICellInputValidatorResponse;
    /**
     * Minimum value
     *
     * The default is Number.NaN, meaning no minimum constraint
     */
    min: number;
    /**
     * Maximum value
     *
     * The default is Number.NaN, meaning no maximum constraint
     */
    max: number;
}
/**
 * An abstract base class that provides the most of the functionality
 * needed by a cell editor. All of the built-in cell editors
 * for various cell types are derived from this base class. Custom cell editors
 * can be easily implemented by extending this class.
 */
export declare abstract class CellEditor implements ICellEditor, IDisposable {
    /**
     * Construct a new cell editor.
     */
    constructor();
    /**
     * Whether the cell editor is disposed.
     */
    get isDisposed(): boolean;
    /**
     * Dispose of the resources held by cell editor.
     */
    dispose(): void;
    /**
     * Start editing the cell.
     *
     * @param cell - The object holding cell configuration data.
     *
     * @param options - The cell editing options.
     */
    edit(cell: CellEditor.CellConfig, options?: ICellEditOptions): void;
    /**
     * Cancel editing the cell.
     */
    cancel(): void;
    /**
     * Start editing the cell. Usually an editor widget is created and
     *  added to `editorContainer`
     */
    protected abstract startEditing(): void;
    /**
     * Return the current input entered. This method throws exceptions
     * if input is invalid. Error message in exception is shown as notification.
     */
    protected abstract getInput(): any;
    /**
     * Whether the value input is valid.
     */
    protected get validInput(): boolean;
    /**
     * Validate the cell input. Shows validation error notification when input is invalid.
     */
    protected validate(): void;
    /**
     * Set validity flag.
     *
     * @param valid - Whether the input is valid.
     *
     * @param message - Notification message to show.
     *
     * If message is set to empty string (which is the default)
     * existing notification popup is removed if any.
     */
    protected setValidity(valid: boolean, message?: string): void;
    /**
     * Create and return a cell input validator based on configuration of the
     * cell being edited. If no suitable validator can be found, it returns undefined.
     */
    protected createValidatorBasedOnType(): ICellInputValidator | undefined;
    /**
     * Compute cell rectangle and return with other cell properties.
     */
    protected getCellInfo(cell: CellEditor.CellConfig): Private.ICellInfo;
    /**
     * Reposition cell editor by moving viewport occluder and cell editor container.
     */
    protected updatePosition(): void;
    /**
     * Commit the edited value.
     *
     * @param cursorMovement - Cursor move direction based on keys pressed to end the edit.
     *
     * @returns true on valid input, false otherwise.
     */
    protected commit(cursorMovement?: SelectionModel.CursorMoveDirection): boolean;
    /**
     * Create container elements needed to prevent editor widget overflow
     * beyond viewport and to position cell editor widget.
     */
    private _addContainer;
    /**
     * Remove validity notification popup.
     */
    private _closeValidityNotification;
    /**
     * A signal emitted when input changes.
     */
    protected inputChanged: Signal<this, void>;
    /**
     * Callback method to call on cell edit commit.
     */
    protected onCommit?: (response: ICellEditResponse) => void;
    /**
     * Callback method to call on cell edit cancel.
     */
    protected onCancel?: () => void;
    /**
     * Cell configuration data for the cell being edited.
     */
    protected cell: CellEditor.CellConfig;
    /**
     * Cell input validator to use for the cell being edited.
     */
    protected validator: ICellInputValidator | undefined;
    /**
     * The div element used to prevent editor widget overflow beyond grid viewport.
     */
    protected viewportOccluder: HTMLDivElement;
    /**
     * The div element used to contain and position editor widget.
     */
    protected editorContainer: HTMLDivElement;
    /**
     * Notification popup used to show validation error messages.
     */
    protected validityNotification: CellEditor.Notification | null;
    /**
     * Whether the cell editor is disposed.
     */
    private _disposed;
    /**
     * Whether the value input is valid.
     */
    private _validInput;
    /**
     * Grid wheel event handler.
     */
    private _gridWheelEventHandler;
}
/**
 * Abstract base class with shared functionality
 * for cell editors which use HTML Input widget as editor.
 */
export declare abstract class InputCellEditor extends CellEditor {
    /**
     * Handle the DOM events for the editor.
     *
     * @param event - The DOM event sent to the editor.
     */
    handleEvent(event: Event): void;
    /**
     * Dispose of the resources held by cell editor.
     */
    dispose(): void;
    /**
     * Start editing the cell.
     */
    protected startEditing(): void;
    protected deserialize(value: unknown): any;
    protected createWidget(): void;
    protected bindEvents(): void;
    private _unbindEvents;
    private _onKeyDown;
    private _onBlur;
    private _onInput;
    protected input: HTMLInputElement;
    protected abstract inputType: string;
}
/**
 * Cell editor for text cells.
 */
export declare class TextCellEditor extends InputCellEditor {
    /**
     * Return the current text input entered.
     */
    protected getInput(): string | null;
    protected inputType: string;
}
/**
 * Cell editor for real number cells.
 */
export declare class NumberCellEditor extends InputCellEditor {
    /**
     * Start editing the cell.
     */
    protected startEditing(): void;
    /**
     * Return the current number input entered. This method throws exception
     * if input is invalid.
     */
    protected getInput(): number | null;
    protected inputType: string;
}
/**
 * Cell editor for integer cells.
 */
export declare class IntegerCellEditor extends InputCellEditor {
    /**
     * Start editing the cell.
     */
    protected startEditing(): void;
    /**
     * Return the current integer input entered. This method throws exception
     * if input is invalid.
     */
    protected getInput(): number | null;
    protected inputType: string;
}
/**
 * Cell editor for date cells.
 */
export declare class DateCellEditor extends CellEditor {
    /**
     * Handle the DOM events for the editor.
     *
     * @param event - The DOM event sent to the editor.
     */
    handleEvent(event: Event): void;
    /**
     * Dispose of the resources held by cell editor.
     */
    dispose(): void;
    /**
     * Start editing the cell.
     */
    protected startEditing(): void;
    /**
     * Return the current date input entered.
     */
    protected getInput(): string | null;
    private _deserialize;
    private _createWidget;
    private _bindEvents;
    private _unbindEvents;
    private _onKeyDown;
    private _onBlur;
    private _input;
}
/**
 * Cell editor for boolean cells.
 */
export declare class BooleanCellEditor extends CellEditor {
    /**
     * Handle the DOM events for the editor.
     *
     * @param event - The DOM event sent to the editor.
     */
    handleEvent(event: Event): void;
    /**
     * Dispose of the resources held by cell editor.
     */
    dispose(): void;
    /**
     * Start editing the cell.
     */
    protected startEditing(): void;
    /**
     * Return the current boolean input entered.
     */
    protected getInput(): boolean | null;
    private _deserialize;
    private _createWidget;
    private _bindEvents;
    private _unbindEvents;
    private _onKeyDown;
    private _onBlur;
    private _input;
}
/**
 * Cell editor for option cells.
 *
 * It supports multiple option selection. If cell metadata contains
 * type attribute 'array', then it behaves as a multi select.
 * In that case cell data is expected to be list of string values.
 */
export declare class OptionCellEditor extends CellEditor {
    /**
     * Dispose of the resources held by cell editor.
     */
    dispose(): void;
    /**
     * Start editing the cell.
     */
    protected startEditing(): void;
    /**
     * Return the current option input.
     */
    protected getInput(): string | string[] | null;
    /**
     * Reposition cell editor.
     */
    protected updatePosition(): void;
    private _deserialize;
    private _createWidget;
    private _bindEvents;
    private _onKeyDown;
    private _onBlur;
    private _select;
    private _isMultiSelect;
}
/**
 * Cell editor for option cells whose value can be any value
 * from set of pre-defined options or values that can be input by user.
 */
export declare class DynamicOptionCellEditor extends CellEditor {
    /**
     * Handle the DOM events for the editor.
     *
     * @param event - The DOM event sent to the editor.
     */
    handleEvent(event: Event): void;
    /**
     * Dispose of the resources held by cell editor.
     */
    dispose(): void;
    /**
     * Start editing the cell.
     */
    protected startEditing(): void;
    /**
     * Return the current option input.
     */
    protected getInput(): string | null;
    private _deserialize;
    private _createWidget;
    private _bindEvents;
    private _unbindEvents;
    private _onKeyDown;
    private _onBlur;
    private _input;
}
/**
 * The namespace for the `CellEditor` class statics.
 */
export declare namespace CellEditor {
    /**
     * An object which holds the configuration data for a cell.
     */
    type CellConfig = {
        /**
         * The grid containing the cell.
         */
        readonly grid: DataGrid;
        /**
         * The row index of the cell.
         */
        readonly row: number;
        /**
         * The column index of the cell.
         */
        readonly column: number;
    };
    /**
     * A widget which implements a notification popup.
     */
    class Notification extends Widget {
        /**
         * Construct a new notification.
         *
         * @param options - The options for initializing the notification.
         */
        constructor(options: Notification.IOptions);
        /**
         * Handle the DOM events for the notification.
         *
         * @param event - The DOM event sent to the notification.
         *
         * #### Notes
         * This method implements the DOM `EventListener` interface and is
         * called in response to events on the notification's DOM node.
         *
         * This should not be called directly by user code.
         */
        handleEvent(event: Event): void;
        /**
         * Get the placement of the notification.
         */
        get placement(): Notification.Placement;
        /**
         * Set the placement of the notification.
         */
        set placement(value: Notification.Placement);
        /**
         * Get the current value of the message.
         */
        get message(): string;
        /**
         * Set the current value of the message.
         *
         */
        set message(value: string);
        /**
         * Get the node presenting the message.
         */
        get messageNode(): HTMLSpanElement;
        /**
         * A method invoked on a 'before-attach' message.
         */
        protected onBeforeAttach(msg: Message): void;
        /**
         * A method invoked on an 'after-detach' message.
         */
        protected onAfterDetach(msg: Message): void;
        /**
         * A method invoked on an 'update-request' message.
         */
        protected onUpdateRequest(msg: Message): void;
        /**
         * Handle the `'mousedown'` event for the notification.
         */
        private _evtMouseDown;
        private _target;
        private _message;
        private _placement;
    }
    /**
     * The namespace for the `Notification` class statics.
     */
    namespace Notification {
        /**
         * A type alias for a notification placement.
         */
        type Placement = 'top' | 'bottom' | 'left' | 'right';
        /**
         * An options object for creating a notification.
         */
        interface IOptions {
            /**
             * Target element to attach notification to.
             *
             */
            target: HTMLElement;
            /**
             * The message to show on notification.
             */
            message?: string;
            /**
             * The placement of the notification.
             *
             * The default is `'bottom'`.
             */
            placement?: Placement;
            /**
             * Duration in ms after which to close notification popup.
             *
             * The default is undefined, and notification is kept visible
             * Timeout value needs to be greater than zero
             */
            timeout?: number;
        }
        /**
         * Create the DOM node for notification.
         */
        function createNode(): HTMLElement;
    }
}
/**
 * A namespace for module-private functionality.
 */
declare namespace Private {
    /**
     * A type alias for cell properties.
     */
    type ICellInfo = {
        grid: DataGrid;
        row: number;
        column: number;
        data: any;
        x: number;
        y: number;
        width: number;
        height: number;
    };
}
export {};
