import { IDisposable } from '@lumino/disposable';
import { DataGrid } from './datagrid';
import { DataModel } from './datamodel';
/**
 * A basic implementation of a data grid mouse handler.
 *
 * #### Notes
 * This class may be subclassed and customized as needed.
 */
export declare class BasicMouseHandler implements DataGrid.IMouseHandler {
    /**
     * Dispose of the resources held by the mouse handler.
     */
    dispose(): void;
    /**
     * Whether the mouse handler is disposed.
     */
    get isDisposed(): boolean;
    /**
     * Release the resources held by the handler.
     */
    release(): void;
    /**
     * Handle the mouse hover event for the data grid.
     *
     * @param grid - The data grid of interest.
     *
     * @param event - The mouse hover event of interest.
     */
    onMouseHover(grid: DataGrid, event: MouseEvent): void;
    /**
     * Handle the mouse leave event for the data grid.
     *
     * @param grid - The data grid of interest.
     *
     * @param event - The mouse hover event of interest.
     */
    onMouseLeave(grid: DataGrid, event: MouseEvent): void;
    /**
     * Handle the mouse down event for the data grid.
     *
     * @param grid - The data grid of interest.
     *
     * @param event - The mouse down event of interest.
     */
    onMouseDown(grid: DataGrid, event: MouseEvent): void;
    /**
     * Handle the mouse move event for the data grid.
     *
     * @param grid - The data grid of interest.
     *
     * @param event - The mouse move event of interest.
     */
    onMouseMove(grid: DataGrid, event: MouseEvent): void;
    /**
     * Handle the mouse up event for the data grid.
     *
     * @param grid - The data grid of interest.
     *
     * @param event - The mouse up event of interest.
     */
    onMouseUp(grid: DataGrid, event: MouseEvent): void;
    /**
     * Handle the mouse double click event for the data grid.
     *
     * @param grid - The data grid of interest.
     *
     * @param event - The mouse up event of interest.
     */
    onMouseDoubleClick(grid: DataGrid, event: MouseEvent): void;
    /**
     * Handle the context menu event for the data grid.
     *
     * @param grid - The data grid of interest.
     *
     * @param event - The context menu event of interest.
     */
    onContextMenu(grid: DataGrid, event: MouseEvent): void;
    /**
     * Handle the wheel event for the data grid.
     *
     * @param grid - The data grid of interest.
     *
     * @param event - The wheel event of interest.
     */
    onWheel(grid: DataGrid, event: WheelEvent): void;
    /**
     * Convert a resize handle into a cursor.
     */
    cursorForHandle(handle: ResizeHandle): string;
    /**
     * Get the current pressData
     */
    get pressData(): PressData.PressData | null;
    private _disposed;
    protected _pressData: PressData.PressData | null;
}
/**
 * A type alias for the resize handle types.
 */
export type ResizeHandle = 'top' | 'left' | 'right' | 'bottom' | 'none' | 'hyperlink';
/**
 * The namespace for the pressdata.
 */
export declare namespace PressData {
    /**
     * A type alias for the row resize data.
     */
    type RowResizeData = {
        /**
         * The descriminated type for the data.
         */
        readonly type: 'row-resize';
        /**
         * The row region which holds the section being resized.
         */
        readonly region: DataModel.RowRegion;
        /**
         * The index of the section being resized.
         */
        readonly index: number;
        /**
         * The original size of the section.
         */
        readonly size: number;
        /**
         * The original client Y position of the mouse.
         */
        readonly clientY: number;
        /**
         * The disposable to clear the cursor override.
         */
        readonly override: IDisposable;
    };
    /**
     * A type alias for the column resize data.
     */
    type ColumnResizeData = {
        /**
         * The descriminated type for the data.
         */
        readonly type: 'column-resize';
        /**
         * The column region which holds the section being resized.
         */
        readonly region: DataModel.ColumnRegion;
        /**
         * The index of the section being resized.
         */
        readonly index: number;
        /**
         * The original size of the section.
         */
        readonly size: number;
        /**
         * The original client X position of the mouse.
         */
        readonly clientX: number;
        /**
         * The disposable to clear the cursor override.
         */
        readonly override: IDisposable;
    };
    /**
     * A type alias for the select data.
     */
    type SelectData = {
        /**
         * The descriminated type for the data.
         */
        readonly type: 'select';
        /**
         * The original region for the mouse press.
         */
        readonly region: DataModel.CellRegion;
        /**
         * The original row that was selected.
         */
        readonly row: number;
        /**
         * The original column that was selected.
         */
        readonly column: number;
        /**
         * The disposable to clear the cursor override.
         */
        readonly override: IDisposable;
        /**
         * The current local X position of the mouse.
         */
        localX: number;
        /**
         * The current local Y position of the mouse.
         */
        localY: number;
        /**
         * The timeout delay for the autoselect loop.
         */
        timeout: number;
    };
    /**
     * A type alias for the resize handler press data.
     */
    type PressData = RowResizeData | ColumnResizeData | SelectData;
}
