// Copyright (c) Jupyter Development Team.
// Distributed under the terms of the Modified BSD License.
/**
 * @packageDocumentation
 * @module documentsearch-extension
 */
import { ILabShell } from '@jupyterlab/application';
import { ICommandPalette, MainAreaWidget } from '@jupyterlab/apputils';
import { ISearchProviderRegistry, SearchDocumentModel, SearchDocumentView, SearchProviderRegistry } from '@jupyterlab/documentsearch';
import { ISettingRegistry } from '@jupyterlab/settingregistry';
import { ITranslator } from '@jupyterlab/translation';
import { Widget } from '@lumino/widgets';
const SEARCHABLE_CLASS = 'jp-mod-searchable';
var CommandIDs;
(function (CommandIDs) {
    /**
     * Start search in a document
     */
    CommandIDs.search = 'documentsearch:start';
    /**
     * Start search and replace in a document
     */
    CommandIDs.searchAndReplace = 'documentsearch:startWithReplace';
    /**
     * Find next search match
     */
    CommandIDs.findNext = 'documentsearch:highlightNext';
    /**
     * Find previous search match
     */
    CommandIDs.findPrevious = 'documentsearch:highlightPrevious';
    /**
     * End search in a document
     */
    CommandIDs.end = 'documentsearch:end';
})(CommandIDs || (CommandIDs = {}));
const labShellWidgetListener = {
    id: '@jupyterlab/documentsearch-extension:labShellWidgetListener',
    requires: [ILabShell, ISearchProviderRegistry],
    autoStart: true,
    activate: (app, labShell, registry) => {
        // If a given widget is searchable, apply the searchable class.
        // If it's not searchable, remove the class.
        const transformWidgetSearchability = (widget) => {
            if (!widget) {
                return;
            }
            if (registry.hasProvider(widget)) {
                widget.addClass(SEARCHABLE_CLASS);
            }
            else {
                widget.removeClass(SEARCHABLE_CLASS);
            }
        };
        // Update searchability of the active widget when the registry
        // changes, in case a provider for the current widget was added
        // or removed
        registry.changed.connect(() => transformWidgetSearchability(labShell.activeWidget));
        // Apply the searchable class only to the active widget if it is actually
        // searchable. Remove the searchable class from a widget when it's
        // no longer active.
        labShell.activeChanged.connect((_, args) => {
            const oldWidget = args.oldValue;
            if (oldWidget) {
                oldWidget.removeClass(SEARCHABLE_CLASS);
            }
            transformWidgetSearchability(args.newValue);
        });
    }
};
/**
 * Initialization data for the document-search extension.
 */
const extension = {
    id: '@jupyterlab/documentsearch-extension:plugin',
    provides: ISearchProviderRegistry,
    requires: [ITranslator],
    optional: [ICommandPalette, ISettingRegistry],
    autoStart: true,
    activate: (app, translator, palette, settingRegistry) => {
        const trans = translator.load('jupyterlab');
        let searchDebounceTime = 500;
        // Create registry
        const registry = new SearchProviderRegistry(translator);
        const searchViews = new Map();
        if (settingRegistry) {
            const loadSettings = settingRegistry.load(extension.id);
            const updateSettings = (settings) => {
                searchDebounceTime = settings.get('searchDebounceTime')
                    .composite;
            };
            Promise.all([loadSettings, app.restored])
                .then(([settings]) => {
                updateSettings(settings);
                settings.changed.connect(settings => {
                    updateSettings(settings);
                });
            })
                .catch((reason) => {
                console.error(reason.message);
            });
        }
        const isEnabled = () => {
            const widget = app.shell.currentWidget;
            if (!widget) {
                return false;
            }
            return registry.hasProvider(widget);
        };
        const getSearchWidget = (widget) => {
            if (!widget) {
                return;
            }
            const widgetId = widget.id;
            let searchView = searchViews.get(widgetId);
            if (!searchView) {
                const searchProvider = registry.getProvider(widget);
                if (!searchProvider) {
                    return;
                }
                const searchModel = new SearchDocumentModel(searchProvider, searchDebounceTime);
                const newView = new SearchDocumentView(searchModel, translator);
                searchViews.set(widgetId, newView);
                // find next, previous and end are now enabled
                [CommandIDs.findNext, CommandIDs.findPrevious, CommandIDs.end].forEach(id => {
                    app.commands.notifyCommandChanged(id);
                });
                /**
                 * Activate the target widget when the search panel is closing
                 */
                newView.closed.connect(() => {
                    if (!widget.isDisposed) {
                        widget.activate();
                    }
                });
                /**
                 * Remove from mapping when the search view is disposed.
                 */
                newView.disposed.connect(() => {
                    if (!widget.isDisposed) {
                        widget.activate();
                    }
                    searchViews.delete(widgetId);
                    // find next, previous and end are now disabled
                    [
                        CommandIDs.findNext,
                        CommandIDs.findPrevious,
                        CommandIDs.end
                    ].forEach(id => {
                        app.commands.notifyCommandChanged(id);
                    });
                });
                /**
                 * Dispose resources when the widget is disposed.
                 */
                widget.disposed.connect(() => {
                    newView.dispose();
                    searchModel.dispose();
                    searchProvider.dispose();
                });
                searchView = newView;
            }
            if (!searchView.isAttached) {
                Widget.attach(searchView, widget.node);
                if (widget instanceof MainAreaWidget) {
                    // Offset the position of the search widget to not cover the toolbar nor the content header.
                    // TODO this does not update once the search widget is displayed.
                    searchView.node.style.top = `${widget.toolbar.node.getBoundingClientRect().height +
                        widget.contentHeader.node.getBoundingClientRect().height}px`;
                }
                if (searchView.model.searchExpression) {
                    searchView.model.refresh();
                }
            }
            return searchView;
        };
        app.commands.addCommand(CommandIDs.search, {
            label: trans.__('Find…'),
            isEnabled: isEnabled,
            execute: args => {
                const searchWidget = getSearchWidget(app.shell.currentWidget);
                if (searchWidget) {
                    const searchText = args['searchText'];
                    if (searchText) {
                        searchWidget.setSearchText(searchText);
                    }
                    else {
                        searchWidget.setSearchText(searchWidget.model.initialQuery);
                    }
                    searchWidget.focusSearchInput();
                }
            }
        });
        app.commands.addCommand(CommandIDs.searchAndReplace, {
            label: trans.__('Find and Replace…'),
            isEnabled: isEnabled,
            execute: args => {
                const searchWidget = getSearchWidget(app.shell.currentWidget);
                if (searchWidget) {
                    const searchText = args['searchText'];
                    if (searchText) {
                        searchWidget.setSearchText(searchText);
                    }
                    else {
                        searchWidget.setSearchText(searchWidget.model.initialQuery);
                    }
                    const replaceText = args['replaceText'];
                    if (replaceText) {
                        searchWidget.setReplaceText(replaceText);
                    }
                    searchWidget.showReplace();
                    searchWidget.focusSearchInput();
                }
            }
        });
        app.commands.addCommand(CommandIDs.findNext, {
            label: trans.__('Find Next'),
            isEnabled: () => !!app.shell.currentWidget &&
                searchViews.has(app.shell.currentWidget.id),
            execute: async () => {
                var _a;
                const currentWidget = app.shell.currentWidget;
                if (!currentWidget) {
                    return;
                }
                await ((_a = searchViews.get(currentWidget.id)) === null || _a === void 0 ? void 0 : _a.model.highlightNext());
            }
        });
        app.commands.addCommand(CommandIDs.findPrevious, {
            label: trans.__('Find Previous'),
            isEnabled: () => !!app.shell.currentWidget &&
                searchViews.has(app.shell.currentWidget.id),
            execute: async () => {
                var _a;
                const currentWidget = app.shell.currentWidget;
                if (!currentWidget) {
                    return;
                }
                await ((_a = searchViews.get(currentWidget.id)) === null || _a === void 0 ? void 0 : _a.model.highlightPrevious());
            }
        });
        app.commands.addCommand(CommandIDs.end, {
            label: trans.__('End Search'),
            isEnabled: () => !!app.shell.currentWidget &&
                searchViews.has(app.shell.currentWidget.id),
            execute: async () => {
                var _a;
                const currentWidget = app.shell.currentWidget;
                if (!currentWidget) {
                    return;
                }
                (_a = searchViews.get(currentWidget.id)) === null || _a === void 0 ? void 0 : _a.close();
            }
        });
        // Add the command to the palette.
        if (palette) {
            [
                CommandIDs.search,
                CommandIDs.findNext,
                CommandIDs.findPrevious,
                CommandIDs.end
            ].forEach(command => {
                palette.addItem({
                    command,
                    category: trans.__('Main Area')
                });
            });
        }
        // Provide the registry to the system.
        return registry;
    }
};
export default [extension, labShellWidgetListener];
//# sourceMappingURL=index.js.map