// Copyright (c) Jupyter Development Team.
// Distributed under the terms of the Modified BSD License.
import { selectAll } from '@codemirror/commands';
import { findNext, gotoLine } from '@codemirror/search';
import { Clipboard, MainAreaWidget } from '@jupyterlab/apputils';
import { CodeViewerWidget } from '@jupyterlab/codeeditor';
import { MarkdownCodeBlocks, PathExt } from '@jupyterlab/coreutils';
import { nullTranslator } from '@jupyterlab/translation';
import { consoleIcon, copyIcon, cutIcon, LabIcon, markdownIcon, pasteIcon, redoIcon, textEditorIcon, undoIcon } from '@jupyterlab/ui-components';
import { find } from '@lumino/algorithm';
const autoClosingBracketsNotebook = 'notebook:toggle-autoclosing-brackets';
const autoClosingBracketsConsole = 'console:toggle-autoclosing-brackets';
/**
 * The command IDs used by the fileeditor plugin.
 */
export var CommandIDs;
(function (CommandIDs) {
    CommandIDs.createNew = 'fileeditor:create-new';
    CommandIDs.createNewMarkdown = 'fileeditor:create-new-markdown-file';
    CommandIDs.changeFontSize = 'fileeditor:change-font-size';
    CommandIDs.lineNumbers = 'fileeditor:toggle-line-numbers';
    CommandIDs.currentLineNumbers = 'fileeditor:toggle-current-line-numbers';
    CommandIDs.lineWrap = 'fileeditor:toggle-line-wrap';
    CommandIDs.currentLineWrap = 'fileeditor:toggle-current-line-wrap';
    CommandIDs.changeTabs = 'fileeditor:change-tabs';
    CommandIDs.matchBrackets = 'fileeditor:toggle-match-brackets';
    CommandIDs.currentMatchBrackets = 'fileeditor:toggle-current-match-brackets';
    CommandIDs.autoClosingBrackets = 'fileeditor:toggle-autoclosing-brackets';
    CommandIDs.autoClosingBracketsUniversal = 'fileeditor:toggle-autoclosing-brackets-universal';
    CommandIDs.createConsole = 'fileeditor:create-console';
    CommandIDs.replaceSelection = 'fileeditor:replace-selection';
    CommandIDs.restartConsole = 'fileeditor:restart-console';
    CommandIDs.runCode = 'fileeditor:run-code';
    CommandIDs.runAllCode = 'fileeditor:run-all';
    CommandIDs.markdownPreview = 'fileeditor:markdown-preview';
    CommandIDs.undo = 'fileeditor:undo';
    CommandIDs.redo = 'fileeditor:redo';
    CommandIDs.cut = 'fileeditor:cut';
    CommandIDs.copy = 'fileeditor:copy';
    CommandIDs.paste = 'fileeditor:paste';
    CommandIDs.selectAll = 'fileeditor:select-all';
    CommandIDs.invokeCompleter = 'completer:invoke-file';
    CommandIDs.selectCompleter = 'completer:select-file';
    CommandIDs.openCodeViewer = 'code-viewer:open';
    CommandIDs.changeTheme = 'fileeditor:change-theme';
    CommandIDs.changeLanguage = 'fileeditor:change-language';
    CommandIDs.find = 'fileeditor:find';
    CommandIDs.goToLine = 'fileeditor:go-to-line';
})(CommandIDs || (CommandIDs = {}));
/**
 * The name of the factory that creates editor widgets.
 */
export const FACTORY = 'Editor';
/**
 * A utility class for adding commands and menu items,
 * for use by the File Editor extension or other Editor extensions.
 */
export var Commands;
(function (Commands) {
    let config = {};
    let scrollPastEnd = true;
    /**
     * Accessor function that returns the createConsole function for use by Create Console commands
     */
    function getCreateConsoleFunction(commands, languages) {
        return async function createConsole(widget, args) {
            var _a, _b, _c;
            const options = args || {};
            const console = await commands.execute('console:create', {
                activate: options['activate'],
                name: (_a = widget.context.contentsModel) === null || _a === void 0 ? void 0 : _a.name,
                path: widget.context.path,
                // Default value is an empty string -> using OR operator
                preferredLanguage: widget.context.model.defaultKernelLanguage ||
                    ((_c = (_b = languages.findByFileName(widget.context.path)) === null || _b === void 0 ? void 0 : _b.name) !== null && _c !== void 0 ? _c : ''),
                ref: widget.id,
                insertMode: 'split-bottom'
            });
            widget.context.pathChanged.connect((sender, value) => {
                var _a;
                console.session.setPath(value);
                console.session.setName((_a = widget.context.contentsModel) === null || _a === void 0 ? void 0 : _a.name);
            });
        };
    }
    /**
     * Update the setting values.
     */
    function updateSettings(settings, commands) {
        var _a;
        config =
            (_a = settings.get('editorConfig').composite) !== null && _a !== void 0 ? _a : {};
        scrollPastEnd = settings.get('scrollPasteEnd').composite;
        // Trigger a refresh of the rendered commands
        commands.notifyCommandChanged(CommandIDs.lineNumbers);
        commands.notifyCommandChanged(CommandIDs.currentLineNumbers);
        commands.notifyCommandChanged(CommandIDs.lineWrap);
        commands.notifyCommandChanged(CommandIDs.currentLineWrap);
        commands.notifyCommandChanged(CommandIDs.changeTabs);
        commands.notifyCommandChanged(CommandIDs.matchBrackets);
        commands.notifyCommandChanged(CommandIDs.currentMatchBrackets);
        commands.notifyCommandChanged(CommandIDs.autoClosingBrackets);
        commands.notifyCommandChanged(CommandIDs.changeLanguage);
    }
    Commands.updateSettings = updateSettings;
    /**
     * Update the settings of the current tracker instances.
     */
    function updateTracker(tracker) {
        tracker.forEach(widget => {
            updateWidget(widget.content);
        });
    }
    Commands.updateTracker = updateTracker;
    /**
     * Update the settings of a widget.
     * Skip global settings for transient editor specific configs.
     */
    function updateWidget(widget) {
        const editor = widget.editor;
        editor.setOptions({ ...config, scrollPastEnd });
    }
    Commands.updateWidget = updateWidget;
    /**
     * Wrapper function for adding the default File Editor commands
     */
    function addCommands(commands, settingRegistry, trans, id, isEnabled, tracker, defaultBrowser, extensions, languages, consoleTracker, sessionDialogs) {
        /**
         * Add a command to change font size for File Editor
         */
        commands.addCommand(CommandIDs.changeFontSize, {
            execute: args => {
                var _a;
                const delta = Number(args['delta']);
                if (Number.isNaN(delta)) {
                    console.error(`${CommandIDs.changeFontSize}: delta arg must be a number`);
                    return;
                }
                const style = window.getComputedStyle(document.documentElement);
                const cssSize = parseInt(style.getPropertyValue('--jp-code-font-size'), 10);
                const currentSize = ((_a = config['customStyles']['fontSize']) !== null && _a !== void 0 ? _a : extensions.baseConfiguration['customStyles']['fontSize']) ||
                    cssSize;
                config.fontSize = currentSize + delta;
                return settingRegistry
                    .set(id, 'editorConfig', config)
                    .catch((reason) => {
                    console.error(`Failed to set ${id}: ${reason.message}`);
                });
            },
            label: args => {
                const delta = Number(args['delta']);
                if (Number.isNaN(delta)) {
                    console.error(`${CommandIDs.changeFontSize}: delta arg must be a number`);
                }
                if (delta > 0) {
                    return args.isMenu
                        ? trans.__('Increase Text Editor Font Size')
                        : trans.__('Increase Font Size');
                }
                else {
                    return args.isMenu
                        ? trans.__('Decrease Text Editor Font Size')
                        : trans.__('Decrease Font Size');
                }
            }
        });
        /**
         * Add the Line Numbers command
         */
        commands.addCommand(CommandIDs.lineNumbers, {
            execute: async () => {
                var _a;
                config.lineNumbers = !((_a = config.lineNumbers) !== null && _a !== void 0 ? _a : extensions.baseConfiguration.lineNumbers);
                try {
                    return await settingRegistry.set(id, 'editorConfig', config);
                }
                catch (reason) {
                    console.error(`Failed to set ${id}: ${reason.message}`);
                }
            },
            isEnabled,
            isToggled: () => { var _a; return (_a = config.lineNumbers) !== null && _a !== void 0 ? _a : extensions.baseConfiguration.lineNumbers; },
            label: trans.__('Show Line Numbers')
        });
        commands.addCommand(CommandIDs.currentLineNumbers, {
            label: trans.__('Show Line Numbers'),
            caption: trans.__('Show the line numbers for the current file.'),
            execute: () => {
                const widget = tracker.currentWidget;
                if (!widget) {
                    return;
                }
                const lineNumbers = !widget.content.editor.getOption('lineNumbers');
                widget.content.editor.setOption('lineNumbers', lineNumbers);
            },
            isEnabled,
            isToggled: () => {
                var _a;
                const widget = tracker.currentWidget;
                return ((_a = widget === null || widget === void 0 ? void 0 : widget.content.editor.getOption('lineNumbers')) !== null && _a !== void 0 ? _a : false);
            }
        });
        /**
         * Add the Word Wrap command
         */
        commands.addCommand(CommandIDs.lineWrap, {
            execute: async (args) => {
                var _a;
                config.lineWrap = (_a = args['mode']) !== null && _a !== void 0 ? _a : false;
                try {
                    return await settingRegistry.set(id, 'editorConfig', config);
                }
                catch (reason) {
                    console.error(`Failed to set ${id}: ${reason.message}`);
                }
            },
            isEnabled,
            isToggled: args => {
                var _a, _b;
                const lineWrap = (_a = args['mode']) !== null && _a !== void 0 ? _a : false;
                return (lineWrap ===
                    ((_b = config.lineWrap) !== null && _b !== void 0 ? _b : extensions.baseConfiguration.lineWrap));
            },
            label: trans.__('Word Wrap')
        });
        commands.addCommand(CommandIDs.currentLineWrap, {
            label: trans.__('Wrap Words'),
            caption: trans.__('Wrap words for the current file.'),
            execute: () => {
                const widget = tracker.currentWidget;
                if (!widget) {
                    return;
                }
                const oldValue = widget.content.editor.getOption('lineWrap');
                widget.content.editor.setOption('lineWrap', !oldValue);
            },
            isEnabled,
            isToggled: () => {
                var _a;
                const widget = tracker.currentWidget;
                return ((_a = widget === null || widget === void 0 ? void 0 : widget.content.editor.getOption('lineWrap')) !== null && _a !== void 0 ? _a : false);
            }
        });
        /**
         * Add command for changing tabs size or type in File Editor
         */
        commands.addCommand(CommandIDs.changeTabs, {
            label: args => {
                var _a;
                if (args.size) {
                    return trans.__('Spaces: %1', (_a = args.size) !== null && _a !== void 0 ? _a : '');
                }
                else {
                    return trans.__('Indent with Tab');
                }
            },
            execute: async (args) => {
                var _a;
                config.indentUnit =
                    args['size'] !== undefined
                        ? ((_a = args['size']) !== null && _a !== void 0 ? _a : '4').toString()
                        : 'Tab';
                try {
                    return await settingRegistry.set(id, 'editorConfig', config);
                }
                catch (reason) {
                    console.error(`Failed to set ${id}: ${reason.message}`);
                }
            },
            isToggled: args => {
                var _a;
                const currentIndentUnit = (_a = config.indentUnit) !== null && _a !== void 0 ? _a : extensions.baseConfiguration.indentUnit;
                return args['size']
                    ? args['size'] === currentIndentUnit
                    : 'Tab' == currentIndentUnit;
            }
        });
        /**
         * Add the Match Brackets command
         */
        commands.addCommand(CommandIDs.matchBrackets, {
            execute: async () => {
                var _a;
                config.matchBrackets = !((_a = config.matchBrackets) !== null && _a !== void 0 ? _a : extensions.baseConfiguration.matchBrackets);
                try {
                    return await settingRegistry.set(id, 'editorConfig', config);
                }
                catch (reason) {
                    console.error(`Failed to set ${id}: ${reason.message}`);
                }
            },
            label: trans.__('Match Brackets'),
            isEnabled,
            isToggled: () => { var _a; return (_a = config.matchBrackets) !== null && _a !== void 0 ? _a : extensions.baseConfiguration.matchBrackets; }
        });
        commands.addCommand(CommandIDs.currentMatchBrackets, {
            label: trans.__('Match Brackets'),
            caption: trans.__('Change match brackets for the current file.'),
            execute: () => {
                const widget = tracker.currentWidget;
                if (!widget) {
                    return;
                }
                const matchBrackets = !widget.content.editor.getOption('matchBrackets');
                widget.content.editor.setOption('matchBrackets', matchBrackets);
            },
            isEnabled,
            isToggled: () => {
                var _a;
                const widget = tracker.currentWidget;
                return ((_a = widget === null || widget === void 0 ? void 0 : widget.content.editor.getOption('matchBrackets')) !== null && _a !== void 0 ? _a : false);
            }
        });
        /**
         * Add the Auto Close Brackets for Text Editor command
         */
        commands.addCommand(CommandIDs.autoClosingBrackets, {
            execute: async (args) => {
                var _a, _b;
                config.autoClosingBrackets = !!((_a = args['force']) !== null && _a !== void 0 ? _a : !((_b = config.autoClosingBrackets) !== null && _b !== void 0 ? _b : extensions.baseConfiguration.autoClosingBrackets));
                try {
                    return await settingRegistry.set(id, 'editorConfig', config);
                }
                catch (reason) {
                    console.error(`Failed to set ${id}: ${reason.message}`);
                }
            },
            label: trans.__('Auto Close Brackets in Text Editor'),
            isToggled: () => {
                var _a;
                return (_a = config.autoClosingBrackets) !== null && _a !== void 0 ? _a : extensions.baseConfiguration.autoClosingBrackets;
            }
        });
        commands.addCommand(CommandIDs.autoClosingBracketsUniversal, {
            execute: () => {
                const anyToggled = commands.isToggled(CommandIDs.autoClosingBrackets) ||
                    commands.isToggled(autoClosingBracketsNotebook) ||
                    commands.isToggled(autoClosingBracketsConsole);
                // if any auto closing brackets options is toggled, toggle both off
                if (anyToggled) {
                    void commands.execute(CommandIDs.autoClosingBrackets, {
                        force: false
                    });
                    void commands.execute(autoClosingBracketsNotebook, { force: false });
                    void commands.execute(autoClosingBracketsConsole, { force: false });
                }
                else {
                    // both are off, turn them on
                    void commands.execute(CommandIDs.autoClosingBrackets, {
                        force: true
                    });
                    void commands.execute(autoClosingBracketsNotebook, { force: true });
                    void commands.execute(autoClosingBracketsConsole, { force: true });
                }
            },
            label: trans.__('Auto Close Brackets'),
            isToggled: () => commands.isToggled(CommandIDs.autoClosingBrackets) ||
                commands.isToggled(autoClosingBracketsNotebook) ||
                commands.isToggled(autoClosingBracketsConsole)
        });
        /**
         * Create a menu for the editor.
         */
        commands.addCommand(CommandIDs.changeTheme, {
            label: args => {
                var _a, _b, _c, _d;
                return (_d = (_c = (_b = ((_a = args.displayName) !== null && _a !== void 0 ? _a : args.theme)) !== null && _b !== void 0 ? _b : config.theme) !== null && _c !== void 0 ? _c : extensions.baseConfiguration.theme) !== null && _d !== void 0 ? _d : trans.__('Editor Theme');
            },
            execute: async (args) => {
                var _a;
                config.theme = (_a = args['theme']) !== null && _a !== void 0 ? _a : config.theme;
                try {
                    return await settingRegistry.set(id, 'editorConfig', config);
                }
                catch (reason) {
                    console.error(`Failed to set theme - ${reason.message}`);
                }
            },
            isToggled: args => { var _a; return args['theme'] === ((_a = config.theme) !== null && _a !== void 0 ? _a : extensions.baseConfiguration.theme); }
        });
        commands.addCommand(CommandIDs.find, {
            label: trans.__('Find…'),
            execute: () => {
                const widget = tracker.currentWidget;
                if (!widget) {
                    return;
                }
                const editor = widget.content.editor;
                editor.execCommand(findNext);
            },
            isEnabled
        });
        commands.addCommand(CommandIDs.goToLine, {
            label: trans.__('Go to Line…'),
            execute: args => {
                const widget = tracker.currentWidget;
                if (!widget) {
                    return;
                }
                const editor = widget.content.editor;
                const line = args['line'];
                const column = args['column'];
                if (line !== undefined || column !== undefined) {
                    editor.setCursorPosition({
                        line: (line !== null && line !== void 0 ? line : 1) - 1,
                        column: (column !== null && column !== void 0 ? column : 1) - 1
                    });
                }
                else {
                    editor.execCommand(gotoLine);
                }
            },
            isEnabled
        });
        commands.addCommand(CommandIDs.changeLanguage, {
            label: args => {
                var _a, _b;
                return (_b = ((_a = args['displayName']) !== null && _a !== void 0 ? _a : args['name'])) !== null && _b !== void 0 ? _b : trans.__('Change editor language.');
            },
            execute: args => {
                const name = args['name'];
                const widget = tracker.currentWidget;
                if (name && widget) {
                    const spec = languages.findByName(name);
                    if (spec) {
                        widget.content.model.mimeType = spec.mime;
                    }
                }
            },
            isEnabled,
            isToggled: args => {
                const widget = tracker.currentWidget;
                if (!widget) {
                    return false;
                }
                const mime = widget.content.model.mimeType;
                const spec = languages.findByMIME(mime);
                const name = spec && spec.name;
                return args['name'] === name;
            }
        });
        /**
         * Add the replace selection for text editor command
         */
        commands.addCommand(CommandIDs.replaceSelection, {
            execute: args => {
                var _a, _b;
                const text = args['text'] || '';
                const widget = tracker.currentWidget;
                if (!widget) {
                    return;
                }
                (_b = (_a = widget.content.editor).replaceSelection) === null || _b === void 0 ? void 0 : _b.call(_a, text);
            },
            isEnabled,
            label: trans.__('Replace Selection in Editor')
        });
        /**
         * Add the Create Console for Editor command
         */
        commands.addCommand(CommandIDs.createConsole, {
            execute: args => {
                const widget = tracker.currentWidget;
                if (!widget) {
                    return;
                }
                return getCreateConsoleFunction(commands, languages)(widget, args);
            },
            isEnabled,
            icon: consoleIcon,
            label: trans.__('Create Console for Editor')
        });
        /**
         * Restart the Console Kernel linked to the current Editor
         */
        commands.addCommand(CommandIDs.restartConsole, {
            execute: async () => {
                var _a;
                const current = (_a = tracker.currentWidget) === null || _a === void 0 ? void 0 : _a.content;
                if (!current || consoleTracker === null) {
                    return;
                }
                const widget = consoleTracker.find(widget => { var _a; return ((_a = widget.sessionContext.session) === null || _a === void 0 ? void 0 : _a.path) === current.context.path; });
                if (widget) {
                    return sessionDialogs.restart(widget.sessionContext);
                }
            },
            label: trans.__('Restart Kernel'),
            isEnabled: () => consoleTracker !== null && isEnabled()
        });
        /**
         * Add the Run Code command
         */
        commands.addCommand(CommandIDs.runCode, {
            execute: () => {
                var _a;
                // Run the appropriate code, taking into account a ```fenced``` code block.
                const widget = (_a = tracker.currentWidget) === null || _a === void 0 ? void 0 : _a.content;
                if (!widget) {
                    return;
                }
                let code = '';
                const editor = widget.editor;
                const path = widget.context.path;
                const extension = PathExt.extname(path);
                const selection = editor.getSelection();
                const { start, end } = selection;
                let selected = start.column !== end.column || start.line !== end.line;
                if (selected) {
                    // Get the selected code from the editor.
                    const start = editor.getOffsetAt(selection.start);
                    const end = editor.getOffsetAt(selection.end);
                    code = editor.model.sharedModel.getSource().substring(start, end);
                }
                else if (MarkdownCodeBlocks.isMarkdown(extension)) {
                    const text = editor.model.sharedModel.getSource();
                    const blocks = MarkdownCodeBlocks.findMarkdownCodeBlocks(text);
                    for (const block of blocks) {
                        if (block.startLine <= start.line && start.line <= block.endLine) {
                            code = block.code;
                            selected = true;
                            break;
                        }
                    }
                }
                if (!selected) {
                    // no selection, submit whole line and advance
                    code = editor.getLine(selection.start.line);
                    const cursor = editor.getCursorPosition();
                    if (cursor.line + 1 === editor.lineCount) {
                        const text = editor.model.sharedModel.getSource();
                        editor.model.sharedModel.setSource(text + '\n');
                    }
                    editor.setCursorPosition({
                        line: cursor.line + 1,
                        column: cursor.column
                    });
                }
                const activate = false;
                if (code) {
                    return commands.execute('console:inject', { activate, code, path });
                }
                else {
                    return Promise.resolve(void 0);
                }
            },
            isEnabled,
            label: trans.__('Run Selected Code')
        });
        /**
         * Add the Run All Code command
         */
        commands.addCommand(CommandIDs.runAllCode, {
            execute: () => {
                var _a;
                const widget = (_a = tracker.currentWidget) === null || _a === void 0 ? void 0 : _a.content;
                if (!widget) {
                    return;
                }
                let code = '';
                const editor = widget.editor;
                const text = editor.model.sharedModel.getSource();
                const path = widget.context.path;
                const extension = PathExt.extname(path);
                if (MarkdownCodeBlocks.isMarkdown(extension)) {
                    // For Markdown files, run only code blocks.
                    const blocks = MarkdownCodeBlocks.findMarkdownCodeBlocks(text);
                    for (const block of blocks) {
                        code += block.code;
                    }
                }
                else {
                    code = text;
                }
                const activate = false;
                if (code) {
                    return commands.execute('console:inject', { activate, code, path });
                }
                else {
                    return Promise.resolve(void 0);
                }
            },
            isEnabled,
            label: trans.__('Run All Code')
        });
        /**
         * Add markdown preview command
         */
        commands.addCommand(CommandIDs.markdownPreview, {
            execute: () => {
                const widget = tracker.currentWidget;
                if (!widget) {
                    return;
                }
                const path = widget.context.path;
                return commands.execute('markdownviewer:open', {
                    path,
                    options: {
                        mode: 'split-right'
                    }
                });
            },
            isVisible: () => {
                const widget = tracker.currentWidget;
                return ((widget && PathExt.extname(widget.context.path) === '.md') || false);
            },
            icon: markdownIcon,
            label: trans.__('Show Markdown Preview')
        });
        /**
         * Add the New File command
         *
         * Defaults to Text/.txt if file type data is not specified
         */
        commands.addCommand(CommandIDs.createNew, {
            label: args => {
                var _a, _b;
                if (args.isPalette) {
                    return (_a = args.paletteLabel) !== null && _a !== void 0 ? _a : trans.__('New Text File');
                }
                return (_b = args.launcherLabel) !== null && _b !== void 0 ? _b : trans.__('Text File');
            },
            caption: args => { var _a; return (_a = args.caption) !== null && _a !== void 0 ? _a : trans.__('Create a new text file'); },
            icon: args => {
                var _a;
                return args.isPalette
                    ? undefined
                    : LabIcon.resolve({
                        icon: (_a = args.iconName) !== null && _a !== void 0 ? _a : textEditorIcon
                    });
            },
            execute: args => {
                var _a;
                const cwd = args.cwd || defaultBrowser.model.path;
                return createNew(commands, cwd, (_a = args.fileExt) !== null && _a !== void 0 ? _a : 'txt');
            }
        });
        /**
         * Add the New Markdown File command
         */
        commands.addCommand(CommandIDs.createNewMarkdown, {
            label: args => args['isPalette']
                ? trans.__('New Markdown File')
                : trans.__('Markdown File'),
            caption: trans.__('Create a new markdown file'),
            icon: args => (args['isPalette'] ? undefined : markdownIcon),
            execute: args => {
                const cwd = args['cwd'] || defaultBrowser.model.path;
                return createNew(commands, cwd, 'md');
            }
        });
        /**
         * Add undo command
         */
        commands.addCommand(CommandIDs.undo, {
            execute: () => {
                var _a;
                const widget = (_a = tracker.currentWidget) === null || _a === void 0 ? void 0 : _a.content;
                if (!widget) {
                    return;
                }
                widget.editor.undo();
            },
            isEnabled: () => {
                var _a;
                if (!isEnabled()) {
                    return false;
                }
                const widget = (_a = tracker.currentWidget) === null || _a === void 0 ? void 0 : _a.content;
                if (!widget) {
                    return false;
                }
                // Ideally enable it when there are undo events stored
                // Reference issue #8590: Code mirror editor could expose the history of undo/redo events
                return true;
            },
            icon: undoIcon.bindprops({ stylesheet: 'menuItem' }),
            label: trans.__('Undo')
        });
        /**
         * Add redo command
         */
        commands.addCommand(CommandIDs.redo, {
            execute: () => {
                var _a;
                const widget = (_a = tracker.currentWidget) === null || _a === void 0 ? void 0 : _a.content;
                if (!widget) {
                    return;
                }
                widget.editor.redo();
            },
            isEnabled: () => {
                var _a;
                if (!isEnabled()) {
                    return false;
                }
                const widget = (_a = tracker.currentWidget) === null || _a === void 0 ? void 0 : _a.content;
                if (!widget) {
                    return false;
                }
                // Ideally enable it when there are redo events stored
                // Reference issue #8590: Code mirror editor could expose the history of undo/redo events
                return true;
            },
            icon: redoIcon.bindprops({ stylesheet: 'menuItem' }),
            label: trans.__('Redo')
        });
        /**
         * Add cut command
         */
        commands.addCommand(CommandIDs.cut, {
            execute: () => {
                var _a;
                const widget = (_a = tracker.currentWidget) === null || _a === void 0 ? void 0 : _a.content;
                if (!widget) {
                    return;
                }
                const editor = widget.editor;
                const text = getTextSelection(editor);
                Clipboard.copyToSystem(text);
                editor.replaceSelection && editor.replaceSelection('');
            },
            isEnabled: () => {
                var _a;
                if (!isEnabled()) {
                    return false;
                }
                const widget = (_a = tracker.currentWidget) === null || _a === void 0 ? void 0 : _a.content;
                if (!widget) {
                    return false;
                }
                // Enable command if there is a text selection in the editor
                return isSelected(widget.editor);
            },
            icon: cutIcon.bindprops({ stylesheet: 'menuItem' }),
            label: trans.__('Cut')
        });
        /**
         * Add copy command
         */
        commands.addCommand(CommandIDs.copy, {
            execute: () => {
                var _a;
                const widget = (_a = tracker.currentWidget) === null || _a === void 0 ? void 0 : _a.content;
                if (!widget) {
                    return;
                }
                const editor = widget.editor;
                const text = getTextSelection(editor);
                Clipboard.copyToSystem(text);
            },
            isEnabled: () => {
                var _a;
                if (!isEnabled()) {
                    return false;
                }
                const widget = (_a = tracker.currentWidget) === null || _a === void 0 ? void 0 : _a.content;
                if (!widget) {
                    return false;
                }
                // Enable command if there is a text selection in the editor
                return isSelected(widget.editor);
            },
            icon: copyIcon.bindprops({ stylesheet: 'menuItem' }),
            label: trans.__('Copy')
        });
        /**
         * Add paste command
         */
        commands.addCommand(CommandIDs.paste, {
            execute: async () => {
                var _a;
                const widget = (_a = tracker.currentWidget) === null || _a === void 0 ? void 0 : _a.content;
                if (!widget) {
                    return;
                }
                const editor = widget.editor;
                // Get data from clipboard
                const clipboard = window.navigator.clipboard;
                const clipboardData = await clipboard.readText();
                if (clipboardData) {
                    // Paste data to the editor
                    editor.replaceSelection && editor.replaceSelection(clipboardData);
                }
            },
            isEnabled: () => { var _a; return Boolean(isEnabled() && ((_a = tracker.currentWidget) === null || _a === void 0 ? void 0 : _a.content)); },
            icon: pasteIcon.bindprops({ stylesheet: 'menuItem' }),
            label: trans.__('Paste')
        });
        /**
         * Add select all command
         */
        commands.addCommand(CommandIDs.selectAll, {
            execute: () => {
                var _a;
                const widget = (_a = tracker.currentWidget) === null || _a === void 0 ? void 0 : _a.content;
                if (!widget) {
                    return;
                }
                const editor = widget.editor;
                editor.execCommand(selectAll);
            },
            isEnabled: () => { var _a; return Boolean(isEnabled() && ((_a = tracker.currentWidget) === null || _a === void 0 ? void 0 : _a.content)); },
            label: trans.__('Select All')
        });
    }
    Commands.addCommands = addCommands;
    function addCompleterCommands(commands, editorTracker, manager, translator) {
        const trans = (translator !== null && translator !== void 0 ? translator : nullTranslator).load('jupyterlab');
        commands.addCommand(CommandIDs.invokeCompleter, {
            label: trans.__('Display the completion helper.'),
            execute: () => {
                const id = editorTracker.currentWidget && editorTracker.currentWidget.id;
                if (id) {
                    return manager.invoke(id);
                }
            }
        });
        commands.addCommand(CommandIDs.selectCompleter, {
            label: trans.__('Select the completion suggestion.'),
            execute: () => {
                const id = editorTracker.currentWidget && editorTracker.currentWidget.id;
                if (id) {
                    return manager.select(id);
                }
            }
        });
        commands.addKeyBinding({
            command: CommandIDs.selectCompleter,
            keys: ['Enter'],
            selector: '.jp-FileEditor .jp-mod-completer-active'
        });
    }
    Commands.addCompleterCommands = addCompleterCommands;
    /**
     * Helper function to check if there is a text selection in the editor
     */
    function isSelected(editor) {
        const selectionObj = editor.getSelection();
        const { start, end } = selectionObj;
        const selected = start.column !== end.column || start.line !== end.line;
        return selected;
    }
    /**
     * Helper function to get text selection from the editor
     */
    function getTextSelection(editor) {
        const selectionObj = editor.getSelection();
        const start = editor.getOffsetAt(selectionObj.start);
        const end = editor.getOffsetAt(selectionObj.end);
        const text = editor.model.sharedModel.getSource().substring(start, end);
        return text;
    }
    /**
     * Function to create a new untitled text file, given the current working directory.
     */
    async function createNew(commands, cwd, ext = 'txt') {
        const model = await commands.execute('docmanager:new-untitled', {
            path: cwd,
            type: 'file',
            ext
        });
        if (model != undefined) {
            const widget = (await commands.execute('docmanager:open', {
                path: model.path,
                factory: FACTORY
            }));
            widget.isUntitled = true;
            return widget;
        }
    }
    /**
     * Wrapper function for adding the default launcher items for File Editor
     */
    function addLauncherItems(launcher, trans) {
        addCreateNewToLauncher(launcher, trans);
        addCreateNewMarkdownToLauncher(launcher, trans);
    }
    Commands.addLauncherItems = addLauncherItems;
    /**
     * Add Create New Text File to the Launcher
     */
    function addCreateNewToLauncher(launcher, trans) {
        launcher.add({
            command: CommandIDs.createNew,
            category: trans.__('Other'),
            rank: 1
        });
    }
    Commands.addCreateNewToLauncher = addCreateNewToLauncher;
    /**
     * Add Create New Markdown to the Launcher
     */
    function addCreateNewMarkdownToLauncher(launcher, trans) {
        launcher.add({
            command: CommandIDs.createNewMarkdown,
            category: trans.__('Other'),
            rank: 2
        });
    }
    Commands.addCreateNewMarkdownToLauncher = addCreateNewMarkdownToLauncher;
    /**
     * Add ___ File items to the Launcher for common file types associated with available kernels
     */
    function addKernelLanguageLauncherItems(launcher, trans, availableKernelFileTypes) {
        for (let ext of availableKernelFileTypes) {
            launcher.add({
                command: CommandIDs.createNew,
                category: trans.__('Other'),
                rank: 3,
                args: ext
            });
        }
    }
    Commands.addKernelLanguageLauncherItems = addKernelLanguageLauncherItems;
    /**
     * Wrapper function for adding the default items to the File Editor palette
     */
    function addPaletteItems(palette, trans) {
        addChangeTabsCommandsToPalette(palette, trans);
        addCreateNewCommandToPalette(palette, trans);
        addCreateNewMarkdownCommandToPalette(palette, trans);
        addChangeFontSizeCommandsToPalette(palette, trans);
    }
    Commands.addPaletteItems = addPaletteItems;
    /**
     * Add commands to change the tab indentation to the File Editor palette
     */
    function addChangeTabsCommandsToPalette(palette, trans) {
        const paletteCategory = trans.__('Text Editor');
        const args = {
            size: 4
        };
        const command = CommandIDs.changeTabs;
        palette.addItem({ command, args, category: paletteCategory });
        for (const size of [1, 2, 4, 8]) {
            const args = {
                size
            };
            palette.addItem({ command, args, category: paletteCategory });
        }
    }
    Commands.addChangeTabsCommandsToPalette = addChangeTabsCommandsToPalette;
    /**
     * Add a Create New File command to the File Editor palette
     */
    function addCreateNewCommandToPalette(palette, trans) {
        const paletteCategory = trans.__('Text Editor');
        palette.addItem({
            command: CommandIDs.createNew,
            args: { isPalette: true },
            category: paletteCategory
        });
    }
    Commands.addCreateNewCommandToPalette = addCreateNewCommandToPalette;
    /**
     * Add a Create New Markdown command to the File Editor palette
     */
    function addCreateNewMarkdownCommandToPalette(palette, trans) {
        const paletteCategory = trans.__('Text Editor');
        palette.addItem({
            command: CommandIDs.createNewMarkdown,
            args: { isPalette: true },
            category: paletteCategory
        });
    }
    Commands.addCreateNewMarkdownCommandToPalette = addCreateNewMarkdownCommandToPalette;
    /**
     * Add commands to change the font size to the File Editor palette
     */
    function addChangeFontSizeCommandsToPalette(palette, trans) {
        const paletteCategory = trans.__('Text Editor');
        const command = CommandIDs.changeFontSize;
        let args = { delta: 1 };
        palette.addItem({ command, args, category: paletteCategory });
        args = { delta: -1 };
        palette.addItem({ command, args, category: paletteCategory });
    }
    Commands.addChangeFontSizeCommandsToPalette = addChangeFontSizeCommandsToPalette;
    /**
     * Add New ___ File commands to the File Editor palette for common file types associated with available kernels
     */
    function addKernelLanguagePaletteItems(palette, trans, availableKernelFileTypes) {
        const paletteCategory = trans.__('Text Editor');
        for (let ext of availableKernelFileTypes) {
            palette.addItem({
                command: CommandIDs.createNew,
                args: { ...ext, isPalette: true },
                category: paletteCategory
            });
        }
    }
    Commands.addKernelLanguagePaletteItems = addKernelLanguagePaletteItems;
    /**
     * Wrapper function for adding the default menu items for File Editor
     */
    function addMenuItems(menu, tracker, consoleTracker, isEnabled) {
        // Add undo/redo hooks to the edit menu.
        menu.editMenu.undoers.redo.add({
            id: CommandIDs.redo,
            isEnabled
        });
        menu.editMenu.undoers.undo.add({
            id: CommandIDs.undo,
            isEnabled
        });
        // Add editor view options.
        menu.viewMenu.editorViewers.toggleLineNumbers.add({
            id: CommandIDs.currentLineNumbers,
            isEnabled
        });
        menu.viewMenu.editorViewers.toggleMatchBrackets.add({
            id: CommandIDs.currentMatchBrackets,
            isEnabled
        });
        menu.viewMenu.editorViewers.toggleWordWrap.add({
            id: CommandIDs.currentLineWrap,
            isEnabled
        });
        // Add a console creator the the file menu.
        menu.fileMenu.consoleCreators.add({
            id: CommandIDs.createConsole,
            isEnabled
        });
        // Add a code runner to the run menu.
        if (consoleTracker) {
            addCodeRunnersToRunMenu(menu, consoleTracker);
        }
    }
    Commands.addMenuItems = addMenuItems;
    /**
     * Add Create New ___ File commands to the File menu for common file types associated with available kernels
     */
    function addKernelLanguageMenuItems(menu, availableKernelFileTypes) {
        for (let ext of availableKernelFileTypes) {
            menu.fileMenu.newMenu.addItem({
                command: CommandIDs.createNew,
                args: ext,
                rank: 31
            });
        }
    }
    Commands.addKernelLanguageMenuItems = addKernelLanguageMenuItems;
    /**
     * Add a File Editor code runner to the Run menu
     */
    function addCodeRunnersToRunMenu(menu, consoleTracker) {
        const isEnabled = (current) => current.context &&
            !!consoleTracker.find(widget => { var _a; return ((_a = widget.sessionContext.session) === null || _a === void 0 ? void 0 : _a.path) === current.context.path; });
        menu.runMenu.codeRunners.restart.add({
            id: CommandIDs.restartConsole,
            isEnabled
        });
        menu.runMenu.codeRunners.run.add({
            id: CommandIDs.runCode,
            isEnabled
        });
        menu.runMenu.codeRunners.runAll.add({
            id: CommandIDs.runAllCode,
            isEnabled
        });
    }
    Commands.addCodeRunnersToRunMenu = addCodeRunnersToRunMenu;
    function addOpenCodeViewerCommand(app, editorServices, tracker, trans) {
        const openCodeViewer = async (args) => {
            var _a;
            const func = editorServices.factoryService.newDocumentEditor;
            const factory = options => {
                return func(options);
            };
            // Derive mimetype from extension
            let mimetype = args.mimeType;
            if (!mimetype && args.extension) {
                mimetype = editorServices.mimeTypeService.getMimeTypeByFilePath(`temp.${args.extension.replace(/\\.$/, '')}`);
            }
            const widget = CodeViewerWidget.createCodeViewer({
                factory,
                content: args.content,
                mimeType: mimetype
            });
            widget.title.label = args.label || trans.__('Code Viewer');
            widget.title.caption = widget.title.label;
            // Get the fileType based on the mimetype to determine the icon
            const fileType = find(app.docRegistry.fileTypes(), fileType => mimetype ? fileType.mimeTypes.includes(mimetype) : false);
            widget.title.icon = (_a = fileType === null || fileType === void 0 ? void 0 : fileType.icon) !== null && _a !== void 0 ? _a : textEditorIcon;
            if (args.widgetId) {
                widget.id = args.widgetId;
            }
            const main = new MainAreaWidget({ content: widget });
            await tracker.add(main);
            app.shell.add(main, 'main');
            return widget;
        };
        app.commands.addCommand(CommandIDs.openCodeViewer, {
            label: trans.__('Open Code Viewer'),
            execute: (args) => {
                return openCodeViewer(args);
            }
        });
    }
    Commands.addOpenCodeViewerCommand = addOpenCodeViewerCommand;
})(Commands || (Commands = {}));
//# sourceMappingURL=commands.js.map