// Copyright (c) Jupyter Development Team.
// Distributed under the terms of the Modified BSD License.
import { closeIcon } from '@jupyterlab/ui-components';
import { UUID } from '@lumino/coreutils';
import { Debouncer } from '@lumino/polling';
import React, { useState } from 'react';
const SETTING_NAME = 'languageServers';
const SERVER_SETTINGS = 'configuration';
/**
 * The React component of the setting field
 */
function BuildSettingForm(props) {
    const { [SERVER_SETTINGS]: serverSettingsSchema, ...otherSettingsSchema } = props.schema;
    const { [SERVER_SETTINGS]: serverSettings, serverName, ...otherSettings } = props.settings;
    const [currentServerName, setCurrentServerName] = useState(serverName);
    /**
     * Callback on server name field change event
     */
    const onServerNameChange = (e) => {
        props.updateSetting
            .invoke(props.serverHash, {
            serverName: e.target.value
        })
            .catch(console.error);
        setCurrentServerName(e.target.value);
    };
    const serverSettingWithType = {};
    Object.entries(serverSettings).forEach(([key, value]) => {
        const newProps = {
            property: key,
            type: typeof value,
            value
        };
        serverSettingWithType[UUID.uuid4()] = newProps;
    });
    const [propertyMap, setPropertyMap] = useState(serverSettingWithType);
    const defaultOtherSettings = {};
    Object.entries(otherSettingsSchema).forEach(([key, value]) => {
        if (key in otherSettings) {
            defaultOtherSettings[key] = otherSettings[key];
        }
        else {
            defaultOtherSettings[key] = value['default'];
        }
    });
    const [otherSettingsComposite, setOtherSettingsComposite] = useState(defaultOtherSettings);
    /**
     * Callback on additional setting field change event
     */
    const onOtherSettingsChange = (property, value, type) => {
        let settingValue = value;
        if (type === 'number') {
            settingValue = parseFloat(value);
        }
        const newProps = {
            ...otherSettingsComposite,
            [property]: settingValue
        };
        props.updateSetting.invoke(props.serverHash, newProps).catch(console.error);
        setOtherSettingsComposite(newProps);
    };
    /**
     * Callback on `Add property` button click event.
     */
    const addProperty = () => {
        const hash = UUID.uuid4();
        const newMap = {
            ...propertyMap,
            [hash]: { property: '', type: 'string', value: '' }
        };
        const payload = {};
        Object.values(newMap).forEach(value => {
            payload[value.property] = value.value;
        });
        props.updateSetting
            .invoke(props.serverHash, {
            [SERVER_SETTINGS]: payload
        })
            .catch(console.error);
        setPropertyMap(newMap);
    };
    /**
     * Callback on `Remove property` button click event.
     */
    const removeProperty = (entryHash) => {
        const newMap = {};
        Object.entries(propertyMap).forEach(([hash, value]) => {
            if (hash !== entryHash) {
                newMap[hash] = value;
            }
            const payload = {};
            Object.values(newMap).forEach(value => {
                payload[value.property] = value.value;
            });
            props.updateSetting
                .invoke(props.serverHash, {
                [SERVER_SETTINGS]: payload
            })
                .catch(console.error);
            setPropertyMap(newMap);
        });
    };
    /**
     * Save setting to the setting registry on field change event.
     */
    const setProperty = (hash, property) => {
        if (hash in propertyMap) {
            const newMap = { ...propertyMap, [hash]: property };
            const payload = {};
            Object.values(newMap).forEach(value => {
                payload[value.property] = value.value;
            });
            setPropertyMap(newMap);
            props.updateSetting
                .invoke(props.serverHash, {
                [SERVER_SETTINGS]: payload
            })
                .catch(console.error);
        }
    };
    const debouncedSetProperty = new Debouncer(setProperty);
    return (React.createElement("div", { className: "array-item" },
        React.createElement("div", { className: "form-group " },
            React.createElement("div", { className: "jp-FormGroup-content" },
                React.createElement("div", { className: "jp-objectFieldWrapper" },
                    React.createElement("fieldset", null,
                        React.createElement("div", { className: "form-group small-field" },
                            React.createElement("div", { className: "jp-modifiedIndicator jp-errorIndicator" }),
                            React.createElement("div", { className: "jp-FormGroup-content" },
                                React.createElement("h3", { className: "jp-FormGroup-fieldLabel jp-FormGroup-contentItem" }, props.trans.__('Server name:')),
                                React.createElement("div", { className: "jp-inputFieldWrapper jp-FormGroup-contentItem" },
                                    React.createElement("input", { className: "form-control", type: "text", required: true, value: currentServerName, onChange: e => {
                                            onServerNameChange(e);
                                        } })),
                                React.createElement("div", { className: "validationErrors" },
                                    React.createElement("div", null,
                                        React.createElement("ul", { className: "error-detail bs-callout bs-callout-info" },
                                            React.createElement("li", { className: "text-danger" }, props.trans.__('is a required property'))))))),
                        Object.entries(otherSettingsSchema).map(([property, value], idx) => {
                            return (React.createElement("div", { key: `${idx}-${property}`, className: "form-group small-field" },
                                React.createElement("div", { className: "jp-FormGroup-content" },
                                    React.createElement("h3", { className: "jp-FormGroup-fieldLabel jp-FormGroup-contentItem" }, value.title),
                                    React.createElement("div", { className: "jp-inputFieldWrapper jp-FormGroup-contentItem" },
                                        React.createElement("input", { className: "form-control", placeholder: "", type: value.type, value: otherSettingsComposite[property], onChange: e => onOtherSettingsChange(property, e.target.value, value.type) })),
                                    React.createElement("div", { className: "jp-FormGroup-description" }, value.description),
                                    React.createElement("div", { className: "validationErrors" }))));
                        }),
                        React.createElement("fieldset", null,
                            React.createElement("legend", null, serverSettingsSchema['title']),
                            Object.entries(propertyMap).map(([hash, property]) => {
                                return (React.createElement(PropertyFrom, { key: hash, hash: hash, property: property, removeProperty: removeProperty, setProperty: debouncedSetProperty }));
                            }),
                            React.createElement("span", null, serverSettingsSchema['description'])))))),
        React.createElement("div", { className: "jp-ArrayOperations" },
            React.createElement("button", { className: "jp-mod-styled jp-mod-reject", onClick: addProperty }, props.trans.__('Add property')),
            React.createElement("button", { className: "jp-mod-styled jp-mod-warn jp-FormGroup-removeButton", onClick: () => props.removeSetting(props.serverHash) }, props.trans.__('Remove server')))));
}
function PropertyFrom(props) {
    const [state, setState] = useState({ ...props.property });
    const TYPE_MAP = { string: 'text', number: 'number', boolean: 'checkbox' };
    const removeItem = () => {
        props.removeProperty(props.hash);
    };
    const changeName = (newName) => {
        const newState = { ...state, property: newName };
        props.setProperty.invoke(props.hash, newState).catch(console.error);
        setState(newState);
    };
    const changeValue = (newValue, type) => {
        let value = newValue;
        if (type === 'number') {
            value = parseFloat(newValue);
        }
        const newState = { ...state, value };
        props.setProperty.invoke(props.hash, newState).catch(console.error);
        setState(newState);
    };
    const changeType = (newType) => {
        let value;
        if (newType === 'boolean') {
            value = false;
        }
        else if (newType === 'number') {
            value = 0;
        }
        else {
            value = '';
        }
        const newState = { ...state, type: newType, value };
        setState(newState);
        props.setProperty.invoke(props.hash, newState).catch(console.error);
    };
    return (React.createElement("div", { key: props.hash, className: "form-group small-field" },
        React.createElement("div", { className: "jp-FormGroup-content jp-LSPExtension-FormGroup-content" },
            React.createElement("input", { className: "form-control", type: "text", required: true, placeholder: 'Property name', value: state.property, onChange: e => {
                    changeName(e.target.value);
                } }),
            React.createElement("select", { className: "form-control", value: state.type, onChange: e => changeType(e.target.value) },
                React.createElement("option", { value: "string" }, "String"),
                React.createElement("option", { value: "number" }, "Number"),
                React.createElement("option", { value: "boolean" }, "Boolean")),
            React.createElement("input", { className: "form-control", type: TYPE_MAP[state.type], required: false, placeholder: 'Property value', value: state.type !== 'boolean' ? state.value : undefined, checked: state.type === 'boolean' ? state.value : undefined, onChange: state.type !== 'boolean'
                    ? e => changeValue(e.target.value, state.type)
                    : e => changeValue(e.target.checked, state.type) }),
            React.createElement("button", { className: "jp-mod-minimal jp-Button", onClick: removeItem },
                React.createElement(closeIcon.react, null)))));
}
/**
 * React setting component
 */
class SettingRenderer extends React.Component {
    constructor(props) {
        super(props);
        /**
         * Remove a setting item by its hash
         *
         * @param hash - hash of the item to be removed.
         */
        this.removeSetting = (hash) => {
            if (hash in this.state.items) {
                const items = {};
                for (const key in this.state.items) {
                    if (key !== hash) {
                        items[key] = this.state.items[key];
                    }
                }
                this.setState(old => {
                    return { ...old, items };
                }, () => {
                    this.saveServerSetting();
                });
            }
        };
        /**
         * Update a setting item by its hash
         *
         * @param hash - hash of the item to be updated.
         * @param newSetting - new setting value.
         */
        this.updateSetting = (hash, newSetting) => {
            if (hash in this.state.items) {
                const items = {};
                for (const key in this.state.items) {
                    if (key === hash) {
                        items[key] = { ...this.state.items[key], ...newSetting };
                    }
                    else {
                        items[key] = this.state.items[key];
                    }
                }
                this.setState(old => {
                    return { ...old, items };
                }, () => {
                    this.saveServerSetting();
                });
            }
        };
        /**
         * Add setting item to the setting component.
         */
        this.addServerSetting = () => {
            let index = 0;
            let key = 'newKey';
            while (Object.values(this.state.items)
                .map(val => val.serverName)
                .includes(key)) {
                index += 1;
                key = `newKey-${index}`;
            }
            this.setState(old => ({
                ...old,
                items: {
                    ...old.items,
                    [UUID.uuid4()]: { ...this._defaultSetting, serverName: key }
                }
            }), () => {
                this.saveServerSetting();
            });
        };
        /**
         * Save the value of setting items to the setting registry.
         */
        this.saveServerSetting = () => {
            const settings = {};
            Object.values(this.state.items).forEach(item => {
                const { serverName, ...setting } = item;
                settings[serverName] = setting;
            });
            this._setting.set(SETTING_NAME, settings).catch(console.error);
        };
        this._setting = props.formContext.settings;
        this._trans = props.translator.load('jupyterlab');
        const schema = this._setting.schema['definitions'];
        this._defaultSetting = schema['languageServer']['default'];
        this._schema = schema['languageServer']['properties'];
        const title = props.schema.title;
        const desc = props.schema.description;
        const settings = props.formContext.settings;
        const compositeData = settings.get(SETTING_NAME).composite;
        let items = {};
        if (compositeData) {
            Object.entries(compositeData).forEach(([key, value]) => {
                if (value) {
                    const hash = UUID.uuid4();
                    items[hash] = { serverName: key, ...value };
                }
            });
        }
        this.state = { title, desc, items };
        this._debouncedUpdateSetting = new Debouncer(this.updateSetting.bind(this));
    }
    render() {
        return (React.createElement("div", null,
            React.createElement("fieldset", null,
                React.createElement("legend", null, this.state.title),
                React.createElement("p", { className: "field-description" }, this.state.desc),
                React.createElement("div", { className: "field field-array field-array-of-object" }, Object.entries(this.state.items).map(([hash, value], idx) => {
                    return (React.createElement(BuildSettingForm, { key: `${idx}-${hash}`, trans: this._trans, removeSetting: this.removeSetting, updateSetting: this._debouncedUpdateSetting, serverHash: hash, settings: value, schema: this._schema }));
                })),
                React.createElement("div", null,
                    React.createElement("button", { style: { margin: 2 }, className: "jp-mod-styled jp-mod-reject", onClick: this.addServerSetting }, this._trans.__('Add server'))))));
    }
}
/**
 * Custom setting renderer for language server extension.
 */
export function renderServerSetting(props, translator) {
    return React.createElement(SettingRenderer, { ...props, translator: translator });
}
//# sourceMappingURL=renderer.js.map