// Copyright (c) Jupyter Development Team.
// Distributed under the terms of the Modified BSD License.
/**
 * @packageDocumentation
 * @module lsp-extension
 */
import { CodeExtractorsManager, DocumentConnectionManager, FeatureManager, ILSPCodeExtractorsManager, ILSPDocumentConnectionManager, ILSPFeatureManager, LanguageServerManager, TextForeignCodeExtractor } from '@jupyterlab/lsp';
import { IRunningSessionManagers } from '@jupyterlab/running';
import { ISettingRegistry } from '@jupyterlab/settingregistry';
import { ITranslator } from '@jupyterlab/translation';
import { IFormRendererRegistry, pythonIcon } from '@jupyterlab/ui-components';
import { Signal } from '@lumino/signaling';
import { renderServerSetting } from './renderer';
const plugin = {
    activate,
    id: '@jupyterlab/lsp-extension:plugin',
    requires: [ISettingRegistry, ITranslator],
    optional: [IRunningSessionManagers, IFormRendererRegistry],
    provides: ILSPDocumentConnectionManager,
    autoStart: true
};
const featurePlugin = {
    id: '@jupyterlab/lsp-extension:feature',
    activate: () => new FeatureManager(),
    provides: ILSPFeatureManager,
    autoStart: true
};
const codeExtractorManagerPlugin = {
    id: ILSPCodeExtractorsManager.name,
    activate: app => {
        const extractorManager = new CodeExtractorsManager();
        const markdownCellExtractor = new TextForeignCodeExtractor({
            language: 'markdown',
            isStandalone: false,
            file_extension: 'md',
            cellType: ['markdown']
        });
        extractorManager.register(markdownCellExtractor, null);
        const rawCellExtractor = new TextForeignCodeExtractor({
            language: 'text',
            isStandalone: false,
            file_extension: 'txt',
            cellType: ['raw']
        });
        extractorManager.register(rawCellExtractor, null);
        return extractorManager;
    },
    provides: ILSPCodeExtractorsManager,
    autoStart: true
};
/**
 * Activate the lsp plugin.
 */
function activate(app, settingRegistry, translator, runningSessionManagers, settingRendererRegistry) {
    const LANGUAGE_SERVERS = 'languageServers';
    const languageServerManager = new LanguageServerManager({});
    const connectionManager = new DocumentConnectionManager({
        languageServerManager
    });
    const updateOptions = (settings) => {
        const options = settings.composite;
        const languageServerSettings = (options.languageServers ||
            {});
        if (options.activate === 'on' && !languageServerManager.isEnabled) {
            languageServerManager.enable().catch(console.error);
        }
        else if (options.activate === 'off' && languageServerManager.isEnabled) {
            languageServerManager.disable();
            return;
        }
        connectionManager.initialConfigurations = languageServerSettings;
        // TODO: if priorities changed reset connections
        connectionManager.updateConfiguration(languageServerSettings);
        connectionManager.updateServerConfigurations(languageServerSettings);
        connectionManager.updateLogging(options.logAllCommunication, options.setTrace);
    };
    settingRegistry.transform(plugin.id, {
        fetch: plugin => {
            const schema = plugin.schema.properties;
            const defaultValue = {};
            languageServerManager.sessions.forEach((_, key) => {
                defaultValue[key] = { rank: 50, configuration: {} };
            });
            schema[LANGUAGE_SERVERS]['default'] = defaultValue;
            return plugin;
        },
        compose: plugin => {
            const properties = plugin.schema.properties;
            const user = plugin.data.user;
            const serverDefaultSettings = properties[LANGUAGE_SERVERS]['default'];
            const serverUserSettings = user[LANGUAGE_SERVERS];
            let serverComposite = { ...serverDefaultSettings };
            if (serverUserSettings) {
                serverComposite = { ...serverComposite, ...serverUserSettings };
            }
            const composite = {
                [LANGUAGE_SERVERS]: serverComposite
            };
            Object.entries(properties).forEach(([key, value]) => {
                if (key !== LANGUAGE_SERVERS) {
                    if (key in user) {
                        composite[key] = user[key];
                    }
                    else {
                        composite[key] = value.default;
                    }
                }
            });
            plugin.data.composite = composite;
            return plugin;
        }
    });
    languageServerManager.sessionsChanged.connect(async () => {
        await settingRegistry.reload(plugin.id);
    });
    settingRegistry
        .load(plugin.id)
        .then(settings => {
        updateOptions(settings);
        settings.changed.connect(() => {
            updateOptions(settings);
        });
        languageServerManager.disable();
    })
        .catch((reason) => {
        console.error(reason.message);
    });
    // Add a sessions manager if the running extension is available
    if (runningSessionManagers) {
        addRunningSessionManager(runningSessionManagers, connectionManager, translator);
    }
    if (settingRendererRegistry) {
        const renderer = {
            fieldRenderer: (props) => {
                return renderServerSetting(props, translator);
            }
        };
        settingRendererRegistry.addRenderer(`${plugin.id}.${LANGUAGE_SERVERS}`, renderer);
    }
    return connectionManager;
}
export class RunningLanguageServer {
    constructor(connection, manager) {
        this._connection = new WeakSet([connection]);
        this._manager = manager;
        this._serverIdentifier = connection.serverIdentifier;
        this._serverLanguage = connection.serverLanguage;
    }
    /**
     * This is no-op because we do not do anything on server click event
     */
    open() {
        /** no-op */
    }
    icon() {
        return pythonIcon;
    }
    label() {
        var _a, _b;
        return `${(_a = this._serverIdentifier) !== null && _a !== void 0 ? _a : ''} (${(_b = this._serverLanguage) !== null && _b !== void 0 ? _b : ''})`;
    }
    shutdown() {
        for (const [key, value] of this._manager.connections.entries()) {
            if (this._connection.has(value)) {
                const { uri } = this._manager.documents.get(key);
                this._manager.unregisterDocument(uri);
            }
        }
        this._manager.disconnect(this._serverIdentifier);
    }
}
/**
 * Add the running terminal manager to the running panel.
 */
function addRunningSessionManager(managers, lsManager, translator) {
    const trans = translator.load('jupyterlab');
    const signal = new Signal(lsManager);
    lsManager.connected.connect(() => signal.emit(lsManager));
    lsManager.disconnected.connect(() => signal.emit(lsManager));
    lsManager.closed.connect(() => signal.emit(lsManager));
    lsManager.documentsChanged.connect(() => signal.emit(lsManager));
    let currentRunning = [];
    managers.add({
        name: trans.__('Language servers'),
        running: () => {
            const connections = new Set([...lsManager.connections.values()]);
            currentRunning = [...connections].map(conn => new RunningLanguageServer(conn, lsManager));
            return currentRunning;
        },
        shutdownAll: () => {
            currentRunning.forEach(item => {
                item.shutdown();
            });
        },
        refreshRunning: () => {
            return void 0;
        },
        runningChanged: signal,
        shutdownLabel: trans.__('Shut Down'),
        shutdownAllLabel: trans.__('Shut Down All'),
        shutdownAllConfirmationText: trans.__('Are you sure you want to permanently shut down all running language servers?')
    });
}
/**
 * Export the plugin as default.
 */
export default [plugin, featurePlugin, codeExtractorManagerPlugin];
//# sourceMappingURL=index.js.map