/*
 * Copyright (c) Jupyter Development Team.
 * Distributed under the terms of the Modified BSD License.
 */
import React from 'react';
import { addIcon, checkIcon, LabIcon } from '@jupyterlab/ui-components';
import { reduce } from '@lumino/algorithm';
import { nullTranslator } from '@jupyterlab/translation';
/**
 * The class name added to the cell-tags field.
 */
const CELL_TAGS_WIDGET_CLASS = 'jp-CellTags';
/**
 * The class name added to each tag element.
 */
const CELL_TAGS_ELEMENT_CLASS = 'jp-CellTags-Tag';
/**
 * The class name added to each applied tag element.
 */
const CELL_TAGS_ELEMENT_APPLIED_CLASS = 'jp-CellTags-Applied';
/**
 * The class name added to each unapplied tag element.
 */
const CELL_TAGS_ELEMENT_UNAPPLIED_CLASS = 'jp-CellTags-Unapplied';
/**
 * The class name added to the tag holder.
 */
const CELL_TAGS_HOLDER_CLASS = 'jp-CellTags-Holder';
/**
 * The class name added to the add-tag input.
 */
const CELL_TAGS_ADD_CLASS = 'jp-CellTags-Add';
/**
 * The class name added to an empty input.
 */
const CELL_TAGS_EMPTY_CLASS = 'jp-CellTags-Empty';
export class CellTagField {
    constructor(tracker, translator) {
        this._tracker = tracker;
        this._translator = translator || nullTranslator;
        this._trans = this._translator.load('jupyterlab');
        this._editing = false;
    }
    addTag(props, tag) {
        const data = props.formData;
        if (tag && !data.includes(tag)) {
            data.push(tag);
            props.formContext.updateMetadata({ [props.name]: data }, true);
        }
    }
    /**
     * Pull from cell metadata all the tags used in the notebook and update the
     * stored tag list.
     */
    pullTags() {
        var _a, _b;
        const notebook = (_a = this._tracker) === null || _a === void 0 ? void 0 : _a.currentWidget;
        const cells = (_b = notebook === null || notebook === void 0 ? void 0 : notebook.model) === null || _b === void 0 ? void 0 : _b.cells;
        if (cells === undefined) {
            return [];
        }
        const allTags = reduce(cells, (allTags, cell) => {
            var _a;
            const tags = (_a = cell.getMetadata('tags')) !== null && _a !== void 0 ? _a : [];
            return [...allTags, ...tags];
        }, []);
        return [...new Set(allTags)].filter(tag => tag !== '');
    }
    _emptyAddTag(target) {
        target.value = '';
        target.style.width = '';
        target.classList.add(CELL_TAGS_EMPTY_CLASS);
    }
    _onAddTagKeyDown(props, event) {
        const input = event.target;
        if (event.ctrlKey)
            return;
        if (event.key === 'Enter') {
            this.addTag(props, input.value);
        }
        else if (event.key === 'Escape') {
            this._emptyAddTag(input);
        }
    }
    _onAddTagFocus(event) {
        if (!this._editing) {
            event.target.blur();
        }
    }
    _onAddTagBlur(input) {
        if (this._editing) {
            this._editing = false;
            this._emptyAddTag(input);
        }
    }
    _onChange(event) {
        if (!event.target.value) {
            this._emptyAddTag(event.target);
        }
        else {
            event.target.classList.remove(CELL_TAGS_EMPTY_CLASS);
            const tmp = document.createElement('span');
            tmp.className = CELL_TAGS_ADD_CLASS;
            tmp.textContent = event.target.value;
            // set width to the pixel length of the text
            document.body.appendChild(tmp);
            event.target.style.setProperty('width', `calc(${tmp.getBoundingClientRect().width}px + var(--jp-add-tag-extra-width))`);
            document.body.removeChild(tmp);
        }
    }
    _onAddTagClick(props, event) {
        const elem = event.target.closest('div');
        const input = elem === null || elem === void 0 ? void 0 : elem.childNodes[0];
        if (!this._editing) {
            this._editing = true;
            input.value = '';
            input.focus();
        }
        else if (event.target !== input) {
            this.addTag(props, input.value);
        }
        event.preventDefault();
    }
    _onTagClick(props, tag) {
        const data = props.formData;
        if (data.includes(tag)) {
            data.splice(data.indexOf(tag), 1);
        }
        else {
            data.push(tag);
        }
        props.formContext.updateMetadata({ [props.name]: data }, true);
    }
    render(props) {
        const allTags = this.pullTags();
        return (React.createElement("div", { className: CELL_TAGS_WIDGET_CLASS },
            React.createElement("div", { className: "jp-FormGroup-fieldLabel jp-FormGroup-contentItem" }, "Cell Tags"),
            allTags &&
                allTags.map((tag, i) => (React.createElement("div", { key: i, className: `${CELL_TAGS_ELEMENT_CLASS} ${props.formData.includes(tag)
                        ? CELL_TAGS_ELEMENT_APPLIED_CLASS
                        : CELL_TAGS_ELEMENT_UNAPPLIED_CLASS}`, onClick: () => this._onTagClick(props, tag) },
                    React.createElement("div", { className: CELL_TAGS_HOLDER_CLASS },
                        React.createElement("span", null, tag),
                        props.formData.includes(tag) && (React.createElement(LabIcon.resolveReact, { icon: checkIcon, tag: "span", elementPosition: "center", height: "18px", width: "18px", marginLeft: "5px", marginRight: "-3px" })))))),
            React.createElement("div", { className: `${CELL_TAGS_ELEMENT_CLASS} ${CELL_TAGS_ELEMENT_UNAPPLIED_CLASS}` },
                React.createElement("div", { className: CELL_TAGS_HOLDER_CLASS, onMouseDown: (e) => this._onAddTagClick(props, e) },
                    React.createElement("input", { className: `${CELL_TAGS_ADD_CLASS} ${CELL_TAGS_EMPTY_CLASS}`, type: "text", placeholder: this._trans.__('Add Tag'), onKeyDown: (e) => this._onAddTagKeyDown(props, e), onFocus: (e) => this._onAddTagFocus(e), onBlur: (e) => this._onAddTagBlur(e.target), onChange: (e) => {
                            this._onChange(e);
                        } }),
                    React.createElement(LabIcon.resolveReact, { icon: addIcon, tag: "span", height: "18px", width: "18px", marginLeft: "5px", marginRight: "-3px" })))));
    }
}
//# sourceMappingURL=celltag.js.map