// Copyright (c) Jupyter Development Team.
// Distributed under the terms of the Modified BSD License.
import { jupyterHighlightStyle } from '@jupyterlab/codemirror';
import { nullTranslator } from '@jupyterlab/translation';
import { InputGroup } from '@jupyterlab/ui-components';
import { tags } from '@lezer/highlight';
import { JSONExt } from '@lumino/coreutils';
import * as React from 'react';
import Highlighter from 'react-highlight-words';
import { JSONTree } from 'react-json-tree';
import { StyleModule } from 'style-mod';
/**
 * Get the CodeMirror style for a given tag.
 */
function getStyle(tag) {
    var _a;
    return (_a = jupyterHighlightStyle.style([tag])) !== null && _a !== void 0 ? _a : '';
}
/**
 * A component that renders JSON data as a collapsible tree.
 */
export class Component extends React.Component {
    constructor() {
        super(...arguments);
        this.state = { filter: '', value: '' };
        this.timer = 0;
        this.handleChange = (event) => {
            const { value } = event.target;
            this.setState({ value });
            window.clearTimeout(this.timer);
            this.timer = window.setTimeout(() => {
                this.setState({ filter: value });
            }, 300);
        };
    }
    componentDidMount() {
        StyleModule.mount(document, jupyterHighlightStyle.module);
    }
    render() {
        const translator = this.props.translator || nullTranslator;
        const trans = translator.load('jupyterlab');
        const { data, metadata, forwardedRef } = this.props;
        const root = metadata && metadata.root ? metadata.root : 'root';
        const keyPaths = this.state.filter
            ? filterPaths(data, this.state.filter, [root])
            : [root];
        return (React.createElement("div", { className: "container", ref: forwardedRef },
            React.createElement(InputGroup, { className: "filter", type: "text", placeholder: trans.__('Filter…'), onChange: this.handleChange, value: this.state.value, rightIcon: "ui-components:search" }),
            React.createElement(JSONTree, { data: data, collectionLimit: 100, theme: {
                    extend: theme,
                    valueLabel: getStyle(tags.variableName),
                    valueText: getStyle(tags.string),
                    nestedNodeItemString: getStyle(tags.comment)
                }, invertTheme: false, keyPath: [root], getItemString: (type, data, itemType, itemString) => Array.isArray(data) ? (
                // Always display array type and the number of items i.e. "[] 2 items".
                React.createElement("span", null,
                    itemType,
                    " ",
                    itemString)) : Object.keys(data).length === 0 ? (
                // Only display object type when it's empty i.e. "{}".
                React.createElement("span", null, itemType)) : (null // Upstream typings don't accept null, but it should be ok
                ), labelRenderer: ([label, type]) => {
                    return (React.createElement("span", { className: getStyle(tags.keyword) },
                        React.createElement(Highlighter, { searchWords: [this.state.filter], textToHighlight: `${label}`, highlightClassName: "jp-mod-selected" })));
                }, valueRenderer: raw => {
                    let className = getStyle(tags.string);
                    if (typeof raw === 'number') {
                        className = getStyle(tags.number);
                    }
                    if (raw === 'true' || raw === 'false') {
                        className = getStyle(tags.keyword);
                    }
                    return (React.createElement("span", { className: className },
                        React.createElement(Highlighter, { searchWords: [this.state.filter], textToHighlight: `${raw}`, highlightClassName: "jp-mod-selected" })));
                }, shouldExpandNodeInitially: (keyPath, data, level) => metadata && metadata.expanded
                    ? true
                    : keyPaths.join(',').includes(keyPath.join(',')) })));
    }
}
// Provide an invalid theme object (this is on purpose!) to invalidate the
// react-json-tree's inline styles that override CodeMirror CSS classes
const theme = {
    scheme: 'jupyter',
    base00: 'invalid',
    base01: 'invalid',
    base02: 'invalid',
    base03: 'invalid',
    base04: 'invalid',
    base05: 'invalid',
    base06: 'invalid',
    base07: 'invalid',
    base08: 'invalid',
    base09: 'invalid',
    base0A: 'invalid',
    base0B: 'invalid',
    base0C: 'invalid',
    base0D: 'invalid',
    base0E: 'invalid',
    base0F: 'invalid',
    author: 'invalid'
};
function objectIncludes(data, query) {
    return JSON.stringify(data).includes(query);
}
function filterPaths(data, query, parent = ['root']) {
    if (JSONExt.isArray(data)) {
        return data.reduce((result, item, index) => {
            if (item && typeof item === 'object' && objectIncludes(item, query)) {
                return [
                    ...result,
                    [index, ...parent].join(','),
                    ...filterPaths(item, query, [index, ...parent])
                ];
            }
            return result;
        }, []);
    }
    if (JSONExt.isObject(data)) {
        return Object.keys(data).reduce((result, key) => {
            const item = data[key];
            if (item &&
                typeof item === 'object' &&
                (key.includes(query) || objectIncludes(item, query))) {
                return [
                    ...result,
                    [key, ...parent].join(','),
                    ...filterPaths(item, query, [key, ...parent])
                ];
            }
            return result;
        }, []);
    }
    return [];
}
//# sourceMappingURL=component.js.map