// Copyright (c) Jupyter Development Team.
// Distributed under the terms of the Modified BSD License.
import { nullTranslator } from '@jupyterlab/translation';
import { kernelIcon, terminalIcon, VDomModel, VDomRenderer } from '@jupyterlab/ui-components';
import React from 'react';
import { GroupItem, TextItem } from '@jupyterlab/statusbar';
/**
 * Half spacing between subitems in a status item.
 */
const HALF_SPACING = 4;
/**
 * A pure functional component for rendering kernel and terminal sessions.
 *
 * @param props: the props for the component.
 *
 * @returns a tsx component for the running sessions.
 */
function RunningSessionsComponent(props) {
    return (React.createElement(GroupItem, { spacing: HALF_SPACING, onClick: props.handleClick },
        React.createElement(GroupItem, { spacing: HALF_SPACING },
            React.createElement(TextItem, { source: props.terminals }),
            React.createElement(terminalIcon.react, { left: '1px', top: '3px', stylesheet: 'statusBar' })),
        React.createElement(GroupItem, { spacing: HALF_SPACING },
            React.createElement(TextItem, { source: props.sessions }),
            React.createElement(kernelIcon.react, { top: '2px', stylesheet: 'statusBar' }))));
}
/**
 * A VDomRenderer for a RunningSessions status item.
 */
export class RunningSessions extends VDomRenderer {
    /**
     * Create a new RunningSessions widget.
     */
    constructor(opts) {
        super(new RunningSessions.Model());
        this._serviceManager = opts.serviceManager;
        this._handleClick = opts.onClick;
        this.translator = opts.translator || nullTranslator;
        this._trans = this.translator.load('jupyterload');
        this._serviceManager.sessions.runningChanged.connect(this._onSessionsRunningChanged, this);
        this._serviceManager.terminals.runningChanged.connect(this._onTerminalsRunningChanged, this);
        this.addClass('jp-mod-highlighted');
    }
    /**
     * Render the running sessions widget.
     */
    render() {
        if (!this.model) {
            return null;
        }
        // TODO-TRANS: Should probably be handled differently.
        // This is more localizable friendly: "Terminals: %1 | Kernels: %2"
        this.title.caption = this._trans.__('%1 Terminals, %2 Kernel sessions', this.model.terminals, this.model.sessions);
        return (React.createElement(RunningSessionsComponent, { sessions: this.model.sessions, terminals: this.model.terminals, handleClick: this._handleClick }));
    }
    /**
     * Dispose of the status item.
     */
    dispose() {
        super.dispose();
        this._serviceManager.sessions.runningChanged.disconnect(this._onSessionsRunningChanged, this);
        this._serviceManager.terminals.runningChanged.disconnect(this._onTerminalsRunningChanged, this);
    }
    /**
     * Set the number of kernel sessions when the list changes.
     */
    _onSessionsRunningChanged(manager, sessions) {
        this.model.sessions = sessions.length;
    }
    /**
     * Set the number of terminal sessions when the list changes.
     */
    _onTerminalsRunningChanged(manager, terminals) {
        this.model.terminals = terminals.length;
    }
}
/**
 * A namespace for RunningSessions statics.
 */
(function (RunningSessions) {
    /**
     * A VDomModel for the RunningSessions status item.
     */
    class Model extends VDomModel {
        constructor() {
            super(...arguments);
            this._terminals = 0;
            this._sessions = 0;
        }
        /**
         * The number of active kernel sessions.
         */
        get sessions() {
            return this._sessions;
        }
        set sessions(sessions) {
            const oldSessions = this._sessions;
            this._sessions = sessions;
            if (oldSessions !== this._sessions) {
                this.stateChanged.emit(void 0);
            }
        }
        /**
         * The number of active terminal sessions.
         */
        get terminals() {
            return this._terminals;
        }
        set terminals(terminals) {
            const oldTerminals = this._terminals;
            this._terminals = terminals;
            if (oldTerminals !== this._terminals) {
                this.stateChanged.emit(void 0);
            }
        }
    }
    RunningSessions.Model = Model;
})(RunningSessions || (RunningSessions = {}));
//# sourceMappingURL=runningSessions.js.map