// Copyright (c) Jupyter Development Team.
// Distributed under the terms of the Modified BSD License.
/**
 * @packageDocumentation
 * @module mathjax-extension
 */
import { ILatexTypesetter } from '@jupyterlab/rendermime';
var CommandIDs;
(function (CommandIDs) {
    /**
     * Copy raw LaTeX to clipboard.
     */
    CommandIDs.copy = 'mathjax:clipboard';
    /**
     * Scale MathJax elements.
     */
    CommandIDs.scale = 'mathjax:scale';
})(CommandIDs || (CommandIDs = {}));
/**
 * The MathJax Typesetter.
 */
export class MathJaxTypesetter {
    constructor() {
        this._initialized = false;
    }
    async _ensureInitialized() {
        if (this._initialized) {
            return;
        }
        await import('mathjax-full/js/input/tex/require/RequireConfiguration');
        const { mathjax } = await import('mathjax-full/js/mathjax');
        const { CHTML } = await import('mathjax-full/js/output/chtml');
        const { TeX } = await import('mathjax-full/js/input/tex');
        const { TeXFont } = await import('mathjax-full/js/output/chtml/fonts/tex');
        const { AllPackages } = await import('mathjax-full/js/input/tex/AllPackages');
        const { SafeHandler } = await import('mathjax-full/js/ui/safe/SafeHandler');
        const { HTMLHandler } = await import('mathjax-full/js/handlers/html/HTMLHandler');
        const { browserAdaptor } = await import('mathjax-full/js/adaptors/browserAdaptor');
        const { AssistiveMmlHandler } = await import('mathjax-full/js/a11y/assistive-mml');
        mathjax.handlers.register(AssistiveMmlHandler(SafeHandler(new HTMLHandler(browserAdaptor()))));
        class EmptyFont extends TeXFont {
            constructor() {
                super(...arguments);
                this.defaultFonts = {};
            }
        }
        const chtml = new CHTML({
            // Override dynamically generated fonts in favor of our font css
            font: new EmptyFont()
        });
        const tex = new TeX({
            packages: AllPackages.concat('require'),
            inlineMath: [
                ['$', '$'],
                ['\\(', '\\)']
            ],
            displayMath: [
                ['$$', '$$'],
                ['\\[', '\\]']
            ],
            processEscapes: true,
            processEnvironments: true
        });
        this._mathDocument = mathjax.document(window.document, {
            InputJax: tex,
            OutputJax: chtml
        });
        this._initialized = true;
    }
    /**
     * Get an instance of the MathDocument object.
     */
    async mathDocument() {
        await this._ensureInitialized();
        return this._mathDocument;
    }
    /**
     * Typeset the math in a node.
     */
    async typeset(node) {
        try {
            await this._ensureInitialized();
        }
        catch (e) {
            console.error(e);
            return;
        }
        this._mathDocument.options.elements = [node];
        this._mathDocument.clear().render();
        delete this._mathDocument.options.elements;
    }
}
/**
 * The MathJax extension.
 */
const mathJaxPlugin = {
    id: '@jupyterlab/mathjax-extension:plugin',
    provides: ILatexTypesetter,
    activate: (app) => {
        const typesetter = new MathJaxTypesetter();
        app.commands.addCommand(CommandIDs.copy, {
            execute: async () => {
                const md = await typesetter.mathDocument();
                const oJax = md.outputJax;
                await navigator.clipboard.writeText(oJax.math.math);
            },
            label: 'MathJax Copy Latex'
        });
        app.commands.addCommand(CommandIDs.scale, {
            execute: async (args) => {
                const md = await typesetter.mathDocument();
                const scale = args['scale'] || 1.0;
                md.outputJax.options.scale = scale;
                md.rerender();
            },
            label: args => 'Mathjax Scale ' + (args['scale'] ? `x${args['scale']}` : 'Reset')
        });
        return typesetter;
    },
    autoStart: true
};
export default mathJaxPlugin;
//# sourceMappingURL=index.js.map