import { ServiceManager } from '@jupyterlab/services';
import { ITranslator } from '@jupyterlab/translation';
import { VDomModel } from '@jupyterlab/ui-components';
/**
 * Information about an extension.
 */
export interface IEntry {
    /**
     * The name of the extension.
     */
    name: string;
    /**
     * A short description of the extension.
     */
    description: string;
    /**
     * A representative link of the package.
     */
    homepage_url: string;
    /**
     * Whether the extension is currently installed.
     */
    installed?: boolean | null;
    /**
     * Whether the extension is allowed or not.
     */
    allowed: boolean;
    /**
     * Whether the extension is approved by the system administrators.
     */
    approved: boolean;
    /**
     * Whether the extension is currently enabled.
     */
    enabled: boolean;
    /**
     * The latest version of the extension.
     */
    latest_version: string;
    /**
     * The installed version of the extension.
     */
    installed_version: string;
    /**
     * A flag indicating the status of an installed extension.
     */
    status: 'ok' | 'warning' | 'error' | 'deprecated' | null;
    /**
     * The package type (prebuilt or source).
     */
    pkg_type: 'prebuilt' | 'source';
    /**
     * The information about extension installation.
     */
    install?: IInstall | null;
    /**
     * Package author.
     */
    author?: string;
    /**
     * Package license.
     */
    license?: string;
    /**
     * URL to the package bug tracker.
     */
    bug_tracker_url?: string;
    /**
     * URL to the package documentation.
     */
    documentation_url?: string;
    /**
     * URL to the package URL in the packager website.
     */
    package_manager_url?: string;
    /**
     * URL to the package code source.
     */
    repository_url?: string;
}
/**
 * Information about extension installation.
 */
export interface IInstall {
    /**
     * The used package manager (e.g. pip, conda...)
     */
    packageManager: string | undefined;
    /**
     * The package name as known by the package manager.
     */
    packageName: string | undefined;
    /**
     * The uninstallation instructions as a comprehensive
     * text for the end user.
     */
    uninstallInstructions: string | undefined;
}
/**
 * An object representing a server reply to performing an action.
 */
export interface IActionReply {
    /**
     * The status category of the reply.
     */
    status: 'ok' | 'warning' | 'error' | null;
    /**
     * An optional message when the status is not 'ok'.
     */
    message?: string;
    /**
     * Follow-up restart needed by the action
     */
    needs_restart: ('frontend' | 'kernel' | 'server')[];
}
/**
 * Extension actions that the server API accepts
 */
export type Action = 'install' | 'uninstall' | 'enable' | 'disable';
/**
 * Model for an extension list.
 */
export declare class ListModel extends VDomModel {
    constructor(serviceManager: ServiceManager.IManager, translator?: ITranslator);
    /**
     * Extension manager name.
     */
    readonly name: string;
    /**
     * Whether the extension manager support installation methods or not.
     */
    readonly canInstall: boolean;
    /**
     * Extensions installation path.
     */
    installPath: string | null;
    /**
     * A readonly array of the installed extensions.
     */
    get installed(): ReadonlyArray<IEntry>;
    /**
     * Whether the warning is disclaimed or not.
     */
    get isDisclaimed(): boolean;
    set isDisclaimed(v: boolean);
    /**
     * Whether the extension manager is enabled or not.
     */
    get isEnabled(): boolean;
    set isEnabled(v: boolean);
    get isLoadingInstalledExtensions(): boolean;
    get isSearching(): boolean;
    /**
     * A readonly array containing the latest search result
     */
    get searchResult(): ReadonlyArray<IEntry>;
    /**
     * The search query.
     *
     * Setting its value triggers a new search.
     */
    get query(): string;
    set query(value: string);
    /**
     * The current search page.
     *
     * Setting its value triggers a new search.
     *
     * ### Note
     * First page is 1.
     */
    get page(): number;
    set page(value: number);
    /**
     * The search pagination.
     *
     * Setting its value triggers a new search.
     */
    get pagination(): number;
    set pagination(value: number);
    /**
     * The last page of results in the current search.
     */
    get lastPage(): number;
    /**
     * Dispose the extensions list model.
     */
    dispose(): void;
    /**
     * Whether there are currently any actions pending.
     */
    hasPendingActions(): boolean;
    /**
     * Install an extension.
     *
     * @param entry An entry indicating which extension to install.
     */
    install(entry: IEntry): Promise<void>;
    /**
     * Uninstall an extension.
     *
     * @param entry An entry indicating which extension to uninstall.
     */
    uninstall(entry: IEntry): Promise<void>;
    /**
     * Enable an extension.
     *
     * @param entry An entry indicating which extension to enable.
     */
    enable(entry: IEntry): Promise<void>;
    /**
     * Disable an extension.
     *
     * @param entry An entry indicating which extension to disable.
     */
    disable(entry: IEntry): Promise<void>;
    /**
     * Refresh installed packages
     *
     * @param force Force refreshing the list of installed packages
     */
    refreshInstalled(force?: boolean): Promise<void>;
    /**
     * Search with current query.
     *
     * Sets searchError and totalEntries as appropriate.
     *
     * @returns The extensions matching the current query.
     */
    protected search(force?: boolean): Promise<void>;
    /**
     * Update the current model.
     *
     * This will query the packages repository, and the notebook server.
     *
     * Emits the `stateChanged` signal on successful completion.
     */
    protected update(force?: boolean): Promise<void>;
    /**
     * Send a request to the server to perform an action on an extension.
     *
     * @param action A valid action to perform.
     * @param entry The extension to perform the action on.
     */
    protected performAction(action: string, entry: IEntry): Promise<IActionReply>;
    /**
     * Add a pending action.
     *
     * @param pending A promise that resolves when the action is completed.
     */
    protected addPendingAction(pending: Promise<any>): void;
    actionError: string | null;
    /**
     * Contains an error message if an error occurred when querying installed extensions.
     */
    installedError: string | null;
    /**
     * Contains an error message if an error occurred when searching for extensions.
     */
    searchError: string | null;
    /**
     * Whether a reload should be considered due to actions taken.
     */
    promptReload: boolean;
    /**
     * The service manager to use for building.
     */
    protected serviceManager: ServiceManager.IManager;
    protected translator: ITranslator;
    private _isDisclaimed;
    private _isEnabled;
    private _isLoadingInstalledExtensions;
    private _isSearching;
    private _query;
    private _page;
    private _pagination;
    private _lastPage;
    private _installed;
    private _lastSearchResult;
    private _pendingActions;
    private _debouncedSearch;
}
/**
 * ListModel statics.
 */
export declare namespace ListModel {
    /**
     * Utility function to check whether an entry can be updated.
     *
     * @param entry The entry to check.
     */
    function entryHasUpdate(entry: IEntry): boolean;
}
