// Copyright (c) Jupyter Development Team.
// Distributed under the terms of the Modified BSD License.
import { showPopup, TextItem } from '@jupyterlab/statusbar';
import { nullTranslator } from '@jupyterlab/translation';
import { VDomModel, VDomRenderer } from '@jupyterlab/ui-components';
import React from 'react';
/**
 * A pure functional component for rendering the TabSpace status.
 */
function TabSpaceComponent(props) {
    const translator = props.translator || nullTranslator;
    const trans = translator.load('jupyterlab');
    const description = typeof props.tabSpace === 'number'
        ? trans.__('Spaces')
        : trans.__('Tab Indent');
    return (React.createElement(TextItem, { onClick: props.handleClick, source: typeof props.tabSpace === 'number'
            ? `${description}: ${props.tabSpace}`
            : description, title: trans.__('Change the indentation…') }));
}
/**
 * A VDomRenderer for a tabs vs. spaces status item.
 */
export class TabSpaceStatus extends VDomRenderer {
    /**
     * Create a new tab/space status item.
     */
    constructor(options) {
        super(new TabSpaceStatus.Model());
        this._popup = null;
        this._menu = options.menu;
        this.translator = options.translator || nullTranslator;
        this.addClass('jp-mod-highlighted');
    }
    /**
     * Render the TabSpace status item.
     */
    render() {
        var _a;
        if (!((_a = this.model) === null || _a === void 0 ? void 0 : _a.indentUnit)) {
            return null;
        }
        else {
            const tabSpace = this.model.indentUnit === 'Tab'
                ? null
                : parseInt(this.model.indentUnit, 10);
            return (React.createElement(TabSpaceComponent, { tabSpace: tabSpace, handleClick: () => this._handleClick(), translator: this.translator }));
        }
    }
    /**
     * Handle a click on the status item.
     */
    _handleClick() {
        const menu = this._menu;
        if (this._popup) {
            this._popup.dispose();
        }
        menu.aboutToClose.connect(this._menuClosed, this);
        this._popup = showPopup({
            body: menu,
            anchor: this,
            align: 'right'
        });
        // Update the menu items
        menu.update();
    }
    _menuClosed() {
        this.removeClass('jp-mod-clicked');
    }
}
/**
 * A namespace for TabSpace statics.
 */
(function (TabSpaceStatus) {
    /**
     * A VDomModel for the TabSpace status item.
     */
    class Model extends VDomModel {
        /**
         * Code editor indentation unit
         */
        get indentUnit() {
            return this._indentUnit;
        }
        set indentUnit(v) {
            if (v !== this._indentUnit) {
                this._indentUnit = v;
                this.stateChanged.emit();
            }
        }
    }
    TabSpaceStatus.Model = Model;
})(TabSpaceStatus || (TabSpaceStatus = {}));
//# sourceMappingURL=tabspacestatus.js.map