/*
 * Copyright (c) Jupyter Development Team.
 * Distributed under the terms of the Modified BSD License.
 */
import { IEditorMimeTypeService } from '@jupyterlab/codeeditor';
import { showPopup, TextItem } from '@jupyterlab/statusbar';
import { nullTranslator } from '@jupyterlab/translation';
import { VDomModel, VDomRenderer } from '@jupyterlab/ui-components';
import { Menu } from '@lumino/widgets';
import React from 'react';
/**
 * A pure function that returns a tsx component for an editor syntax item.
 *
 * @param props: the props for the component.
 *
 * @returns an editor syntax component.
 */
function EditorSyntaxComponent(props) {
    return React.createElement(TextItem, { source: props.language, onClick: props.handleClick });
}
/**
 * StatusBar item to change the language syntax highlighting of the file editor.
 */
export class EditorSyntaxStatus extends VDomRenderer {
    /**
     * Construct a new VDomRenderer for the status item.
     */
    constructor(options) {
        var _a;
        super(new EditorSyntaxStatus.Model(options.languages));
        /**
         * Create a menu for selecting the language of the editor.
         */
        this._handleClick = () => {
            const languageMenu = new Menu({ commands: this._commands });
            const command = 'fileeditor:change-language';
            if (this._popup) {
                this._popup.dispose();
            }
            this.model.languages
                .getLanguages()
                .sort((a, b) => {
                var _a, _b;
                const aName = (_a = a.displayName) !== null && _a !== void 0 ? _a : a.name;
                const bName = (_b = b.displayName) !== null && _b !== void 0 ? _b : b.name;
                return aName.localeCompare(bName);
            })
                .forEach(spec => {
                var _a;
                if (spec.name.toLowerCase().indexOf('brainf') === 0) {
                    return;
                }
                const args = {
                    name: spec.name,
                    displayName: (_a = spec.displayName) !== null && _a !== void 0 ? _a : spec.name
                };
                languageMenu.addItem({
                    command,
                    args
                });
            });
            this._popup = showPopup({
                body: languageMenu,
                anchor: this,
                align: 'left'
            });
        };
        this._popup = null;
        this._commands = options.commands;
        this.translator = (_a = options.translator) !== null && _a !== void 0 ? _a : nullTranslator;
        const trans = this.translator.load('jupyterlab');
        this.addClass('jp-mod-highlighted');
        this.title.caption = trans.__('Change text editor syntax highlighting');
    }
    /**
     * Render the status item.
     */
    render() {
        if (!this.model) {
            return null;
        }
        return (React.createElement(EditorSyntaxComponent, { language: this.model.language, handleClick: this._handleClick }));
    }
}
/**
 * A namespace for EditorSyntax statics.
 */
(function (EditorSyntaxStatus) {
    /**
     * A VDomModel for the current editor/mode combination.
     */
    class Model extends VDomModel {
        constructor(languages) {
            super();
            this.languages = languages;
            /**
             * If the editor mode changes, update the model.
             */
            this._onMIMETypeChange = (mode, change) => {
                var _a;
                const oldLanguage = this._language;
                const spec = this.languages.findByMIME(change.newValue);
                this._language = (_a = spec === null || spec === void 0 ? void 0 : spec.name) !== null && _a !== void 0 ? _a : IEditorMimeTypeService.defaultMimeType;
                this._triggerChange(oldLanguage, this._language);
            };
            this._language = '';
            this._editor = null;
        }
        /**
         * The current editor language. If no editor is present,
         * returns the empty string.
         */
        get language() {
            return this._language;
        }
        /**
         * The current editor for the application editor tracker.
         */
        get editor() {
            return this._editor;
        }
        set editor(editor) {
            var _a;
            const oldEditor = this._editor;
            if (oldEditor !== null) {
                oldEditor.model.mimeTypeChanged.disconnect(this._onMIMETypeChange);
            }
            const oldLanguage = this._language;
            this._editor = editor;
            if (this._editor === null) {
                this._language = '';
            }
            else {
                const spec = this.languages.findByMIME(this._editor.model.mimeType);
                this._language = (_a = spec === null || spec === void 0 ? void 0 : spec.name) !== null && _a !== void 0 ? _a : IEditorMimeTypeService.defaultMimeType;
                this._editor.model.mimeTypeChanged.connect(this._onMIMETypeChange);
            }
            this._triggerChange(oldLanguage, this._language);
        }
        /**
         * Trigger a rerender of the model.
         */
        _triggerChange(oldState, newState) {
            if (oldState !== newState) {
                this.stateChanged.emit(void 0);
            }
        }
    }
    EditorSyntaxStatus.Model = Model;
})(EditorSyntaxStatus || (EditorSyntaxStatus = {}));
//# sourceMappingURL=syntaxstatus.js.map