// Copyright (c) Jupyter Development Team.
// Distributed under the terms of the Modified BSD License.
import { VirtualDocument, WidgetLSPAdapter } from '@jupyterlab/lsp';
import { PromiseDelegate } from '@lumino/coreutils';
export class FileEditorAdapter extends WidgetLSPAdapter {
    constructor(editorWidget, options) {
        const { docRegistry, ...others } = options;
        super(editorWidget, others);
        this._readyDelegate = new PromiseDelegate();
        this.editor = editorWidget.content;
        this._docRegistry = docRegistry;
        // Ensure editor uniqueness
        this._virtualEditor = Object.freeze({
            getEditor: () => this.editor.editor,
            ready: () => Promise.resolve(this.editor.editor),
            reveal: () => Promise.resolve(this.editor.editor)
        });
        Promise.all([this.editor.context.ready, this.connectionManager.ready])
            .then(async () => {
            await this.initOnceReady();
            this._readyDelegate.resolve();
        })
            .catch(console.error);
    }
    /**
     * Promise that resolves once the adapter is initialized
     */
    get ready() {
        return this._readyDelegate.promise;
    }
    /**
     * Get current path of the document.
     */
    get documentPath() {
        return this.widget.context.path;
    }
    /**
     * Get the mime type of the document.
     */
    get mimeType() {
        var _a;
        const mimeTypeFromModel = this.editor.model.mimeType;
        const codeMirrorMimeType = Array.isArray(mimeTypeFromModel)
            ? (_a = mimeTypeFromModel[0]) !== null && _a !== void 0 ? _a : 'text/plain'
            : mimeTypeFromModel;
        const contentsModel = this.editor.context.contentsModel;
        // when MIME type is not known it defaults to 'text/plain',
        // so if it is different we can accept it as it is
        if (codeMirrorMimeType != 'text/plain') {
            return codeMirrorMimeType;
        }
        else if (contentsModel) {
            // a script that does not have a MIME type known by the editor
            // (no syntax highlight mode), can still be known by the document
            // registry (and this is arguably easier to extend).
            let fileType = this._docRegistry.getFileTypeForModel(contentsModel);
            return fileType.mimeTypes[0];
        }
        else {
            // "text/plain" this is
            return codeMirrorMimeType;
        }
    }
    /**
     * Get the file extension of the document.
     */
    get languageFileExtension() {
        let parts = this.documentPath.split('.');
        return parts[parts.length - 1];
    }
    /**
     * Get the CM editor
     */
    get ceEditor() {
        return this.editor.editor;
    }
    /**
     * Get the activated CM editor.
     */
    get activeEditor() {
        return this._virtualEditor;
    }
    /**
     * Get the inner HTMLElement of the document widget.
     */
    get wrapperElement() {
        return this.widget.node;
    }
    /**
     * Get current path of the document.
     */
    get path() {
        return this.widget.context.path;
    }
    /**
     *  Get the list of CM editors in the document, there is only one editor
     * in the case of file editor.
     */
    get editors() {
        var _a, _b;
        return [
            {
                ceEditor: this._virtualEditor,
                type: 'code',
                value: (_b = (_a = this.editor) === null || _a === void 0 ? void 0 : _a.model.sharedModel.getSource()) !== null && _b !== void 0 ? _b : ''
            }
        ];
    }
    /**
     * Dispose the widget.
     */
    dispose() {
        if (this.isDisposed) {
            return;
        }
        this.editor.model.mimeTypeChanged.disconnect(this.reloadConnection);
        super.dispose();
    }
    /**
     * Generate the virtual document associated with the document.
     */
    createVirtualDocument() {
        return new VirtualDocument({
            language: this.language,
            foreignCodeExtractors: this.options.foreignCodeExtractorsManager,
            path: this.documentPath,
            fileExtension: this.languageFileExtension,
            // notebooks are continuous, each cell is dependent on the previous one
            standalone: true,
            // notebooks are not supported by LSP servers
            hasLspSupportedFile: true
        });
    }
    /**
     * Get the index of editor from the cursor position in the virtual
     * document. Since there is only one editor, this method always return
     * 0
     *
     * @param position - the position of cursor in the virtual document.
     * @return  {number} - index of the virtual editor
     */
    getEditorIndexAt(position) {
        return 0;
    }
    /**
     * Get the index of input editor
     *
     * @param ceEditor - instance of the code editor
     */
    getEditorIndex(ceEditor) {
        return 0;
    }
    /**
     * Get the wrapper of input editor.
     *
     * @param ceEditor
     * @return  {HTMLElement}
     */
    getEditorWrapper(ceEditor) {
        return this.wrapperElement;
    }
    /**
     * Initialization function called once the editor and the LSP connection
     * manager is ready. This function will create the virtual document and
     * connect various signals.
     */
    async initOnceReady() {
        this.initVirtual();
        // connect the document, but do not open it as the adapter will handle this
        // after registering all features
        await this.connectDocument(this.virtualDocument, false);
        this.editor.model.mimeTypeChanged.connect(this.reloadConnection, this);
    }
}
//# sourceMappingURL=fileeditorlspadapter.js.map