// Copyright (c) Jupyter Development Team.
// Distributed under the terms of the Modified BSD License.
/**
 * @packageDocumentation
 * @module completer-extension
 */
import { CompletionProviderManager, ContextCompleterProvider, ICompletionProviderManager, KernelCompleterProvider } from '@jupyterlab/completer';
import { ISettingRegistry } from '@jupyterlab/settingregistry';
import { IFormRendererRegistry } from '@jupyterlab/ui-components';
import { renderAvailableProviders } from './renderer';
const COMPLETION_MANAGER_PLUGIN = '@jupyterlab/completer-extension:manager';
const defaultProvider = {
    id: '@jupyterlab/completer-extension:base-service',
    requires: [ICompletionProviderManager],
    autoStart: true,
    activate: (app, completionManager) => {
        completionManager.registerProvider(new ContextCompleterProvider());
        completionManager.registerProvider(new KernelCompleterProvider());
    }
};
const manager = {
    id: COMPLETION_MANAGER_PLUGIN,
    requires: [ISettingRegistry],
    optional: [IFormRendererRegistry],
    provides: ICompletionProviderManager,
    autoStart: true,
    activate: (app, settings, editorRegistry) => {
        const AVAILABLE_PROVIDERS = 'availableProviders';
        const PROVIDER_TIMEOUT = 'providerTimeout';
        const SHOW_DOCUMENT_PANEL = 'showDocumentationPanel';
        const CONTINUOUS_HINTING = 'autoCompletion';
        const manager = new CompletionProviderManager();
        const updateSetting = (settingValues, availableProviders) => {
            var _a;
            const providersData = settingValues.get(AVAILABLE_PROVIDERS);
            const timeout = settingValues.get(PROVIDER_TIMEOUT);
            const showDoc = settingValues.get(SHOW_DOCUMENT_PANEL);
            const continuousHinting = settingValues.get(CONTINUOUS_HINTING);
            manager.setTimeout(timeout.composite);
            manager.setShowDocumentationPanel(showDoc.composite);
            manager.setContinuousHinting(continuousHinting.composite);
            const selectedProviders = (_a = providersData.user) !== null && _a !== void 0 ? _a : providersData.composite;
            const sortedProviders = Object.entries(selectedProviders !== null && selectedProviders !== void 0 ? selectedProviders : {})
                .filter(val => val[1] >= 0 && availableProviders.includes(val[0]))
                .sort(([, rank1], [, rank2]) => rank2 - rank1)
                .map(item => item[0]);
            manager.activateProvider(sortedProviders);
        };
        app.restored
            .then(() => {
            const availableProviders = [...manager.getProviders().keys()];
            settings.transform(COMPLETION_MANAGER_PLUGIN, {
                fetch: plugin => {
                    const schema = plugin.schema.properties;
                    const defaultValue = {};
                    availableProviders.forEach((item, index) => {
                        defaultValue[item] = (index + 1) * 100;
                    });
                    schema[AVAILABLE_PROVIDERS]['default'] = defaultValue;
                    return plugin;
                }
            });
            const settingsPromise = settings.load(COMPLETION_MANAGER_PLUGIN);
            settingsPromise
                .then(settingValues => {
                updateSetting(settingValues, availableProviders);
                settingValues.changed.connect(newSettings => {
                    updateSetting(newSettings, availableProviders);
                });
            })
                .catch(console.error);
        })
            .catch(console.error);
        if (editorRegistry) {
            const renderer = {
                fieldRenderer: (props) => {
                    return renderAvailableProviders(props);
                }
            };
            editorRegistry.addRenderer(`${COMPLETION_MANAGER_PLUGIN}.availableProviders`, renderer);
        }
        return manager;
    }
};
/**
 * Export the plugins as default.
 */
const plugins = [manager, defaultProvider];
export default plugins;
//# sourceMappingURL=index.js.map