import type * as nbformat from '@jupyterlab/nbformat';
import { PartialJSONValue } from '@lumino/coreutils';
import { ISignal, Signal } from '@lumino/signaling';
import * as Y from 'yjs';
import type { IMapChange, ISharedNotebook, NotebookChange, SharedCell } from './api.js';
import { YDocument } from './ydocument.js';
import { YBaseCell, YCellType } from './ycell.js';
/**
 * Shared implementation of the Shared Document types.
 *
 * Shared cells can be inserted into a SharedNotebook.
 * Shared cells only start emitting events when they are connected to a SharedNotebook.
 *
 * "Standalone" cells must not be inserted into a (Shared)Notebook.
 * Standalone cells emit events immediately after they have been created, but they must not
 * be included into a (Shared)Notebook.
 */
export declare class YNotebook extends YDocument<NotebookChange> implements ISharedNotebook {
    /**
     * Create a new notebook
     *
     * #### Notes
     * The document is empty and must be populated
     *
     * @param options
     */
    constructor(options?: Omit<ISharedNotebook.IOptions, 'data'>);
    /**
     * Creates a standalone YNotebook
     *
     * Note: This method is useful when we need to initialize
     * the YNotebook from the JavaScript side.
     */
    static create(options?: ISharedNotebook.IOptions): YNotebook;
    /**
     * YJS map for the notebook metadata
     */
    readonly ymeta: Y.Map<any>;
    /**
     * Cells list
     */
    readonly cells: YCellType[];
    /**
     * Wether the undo/redo logic should be
     * considered on the full document across all cells.
     *
     * Default: false
     */
    get disableDocumentWideUndoRedo(): boolean;
    /**
     * Notebook metadata
     */
    get metadata(): nbformat.INotebookMetadata;
    set metadata(v: nbformat.INotebookMetadata);
    /**
     * Signal triggered when a metadata changes.
     */
    get metadataChanged(): ISignal<this, IMapChange>;
    /**
     * nbformat major version
     */
    get nbformat(): number;
    set nbformat(value: number);
    /**
     * nbformat minor version
     */
    get nbformat_minor(): number;
    set nbformat_minor(value: number);
    /**
     * Dispose of the resources.
     */
    dispose(): void;
    /**
     * Get a shared cell by index.
     *
     * @param index: Cell's position.
     *
     * @returns The requested shared cell.
     */
    getCell(index: number): YCellType;
    /**
     * Add a shared cell at the notebook bottom.
     *
     * @param cell Cell to add.
     *
     * @returns The added cell.
     */
    addCell(cell: SharedCell.Cell): YBaseCell<nbformat.IBaseCellMetadata>;
    /**
     * Insert a shared cell into a specific position.
     *
     * @param index: Cell's position.
     * @param cell: Cell to insert.
     *
     * @returns The inserted cell.
     */
    insertCell(index: number, cell: SharedCell.Cell): YBaseCell<nbformat.IBaseCellMetadata>;
    /**
     * Insert a list of shared cells into a specific position.
     *
     * @param index: Position to insert the cells.
     * @param cells: Array of shared cells to insert.
     *
     * @returns The inserted cells.
     */
    insertCells(index: number, cells: SharedCell.Cell[]): YBaseCell<nbformat.IBaseCellMetadata>[];
    /**
     * Move a cell.
     *
     * @param fromIndex: Index of the cell to move.
     * @param toIndex: New position of the cell.
     */
    moveCell(fromIndex: number, toIndex: number): void;
    /**
     * Move cells.
     *
     * @param fromIndex: Index of the first cells to move.
     * @param toIndex: New position of the first cell (in the current array).
     * @param n: Number of cells to move (default 1)
     */
    moveCells(fromIndex: number, toIndex: number, n?: number): void;
    /**
     * Remove a cell.
     *
     * @param index: Index of the cell to remove.
     */
    deleteCell(index: number): void;
    /**
     * Remove a range of cells.
     *
     * @param from: The start index of the range to remove (inclusive).
     * @param to: The end index of the range to remove (exclusive).
     */
    deleteCellRange(from: number, to: number): void;
    /**
     * Delete a metadata notebook.
     *
     * @param key The key to delete
     */
    deleteMetadata(key: string): void;
    /**
     * Returns some metadata associated with the notebook.
     *
     * If no `key` is provided, it will return all metadata.
     * Else it will return the value for that key.
     *
     * @param key Key to get from the metadata
     * @returns Notebook's metadata.
     */
    getMetadata(): nbformat.INotebookMetadata;
    getMetadata(key: string): PartialJSONValue | undefined;
    /**
     * Sets some metadata associated with the notebook.
     *
     * If only one argument is provided, it will override all notebook metadata.
     * Otherwise a single key will be set to a new value.
     *
     * @param metadata All Notebook's metadata or the key to set.
     * @param value New metadata value
     */
    setMetadata(metadata: nbformat.INotebookMetadata): void;
    setMetadata(metadata: string, value: PartialJSONValue): void;
    /**
     * Updates the metadata associated with the notebook.
     *
     * @param value: Metadata's attribute to update.
     */
    updateMetadata(value: Partial<nbformat.INotebookMetadata>): void;
    /**
     * Override the notebook with a JSON-serialized document.
     *
     * @param value The notebook
     */
    fromJSON(value: nbformat.INotebookContent): void;
    /**
     * Serialize the model to JSON.
     */
    toJSON(): nbformat.INotebookContent;
    /**
     * Handle a change to the ystate.
     */
    private _onMetaChanged;
    /**
     * Handle a change to the list of cells.
     */
    private _onYCellsChanged;
    protected _metadataChanged: Signal<this, IMapChange<any>>;
    /**
     * Internal Yjs cells list
     */
    protected readonly _ycells: Y.Array<Y.Map<any>>;
    private _disableDocumentWideUndoRedo;
    private _ycellMapping;
}
