/*
 * Copyright (c) Jupyter Development Team.
 * Distributed under the terms of the Modified BSD License.
 */
import { IEditorServices } from '@jupyterlab/codeeditor';
import { CodeMirrorEditorFactory, CodeMirrorMimeTypeService, EditorExtensionRegistry, EditorLanguageRegistry, EditorThemeRegistry, IEditorExtensionRegistry, IEditorLanguageRegistry, IEditorThemeRegistry, ybinding } from '@jupyterlab/codemirror';
import { ISettingRegistry } from '@jupyterlab/settingregistry';
import { ITranslator, nullTranslator } from '@jupyterlab/translation';
import { FormComponent, IFormRendererRegistry } from '@jupyterlab/ui-components';
import { JSONExt } from '@lumino/coreutils';
import validatorAjv8 from '@rjsf/validator-ajv8';
import React from 'react';
/**
 * CodeMirror settings plugin ID
 */
const SETTINGS_ID = '@jupyterlab/codemirror-extension:plugin';
/**
 * CodeMirror language registry provider.
 */
export const languagePlugin = {
    id: '@jupyterlab/codemirror-extension:languages',
    provides: IEditorLanguageRegistry,
    optional: [ITranslator],
    activate: (app, translator) => {
        const languages = new EditorLanguageRegistry();
        // Register default languages
        for (const language of EditorLanguageRegistry.getDefaultLanguages(translator)) {
            languages.addLanguage(language);
        }
        // Add Jupyter Markdown flavor here to support
        // code block highlighting.
        languages.addLanguage({
            name: 'ipythongfm',
            mime: 'text/x-ipythongfm',
            load: async () => {
                // TODO: add support for LaTeX
                const m = await import('@codemirror/lang-markdown');
                return m.markdown({
                    codeLanguages: (info) => languages.findBest(info)
                });
            }
        });
        return languages;
    }
};
/**
 * CodeMirror theme registry provider.
 */
export const themePlugin = {
    id: '@jupyterlab/codemirror-extension:themes',
    provides: IEditorThemeRegistry,
    optional: [ITranslator],
    activate: (app, translator) => {
        const themes = new EditorThemeRegistry();
        // Register default themes
        for (const theme of EditorThemeRegistry.getDefaultThemes(translator)) {
            themes.addTheme(theme);
        }
        return themes;
    }
};
/**
 * CodeMirror editor extensions registry provider.
 */
export const extensionPlugin = {
    id: '@jupyterlab/codemirror-extension:extensions',
    provides: IEditorExtensionRegistry,
    requires: [IEditorThemeRegistry],
    optional: [ITranslator, ISettingRegistry, IFormRendererRegistry],
    activate: (app, themes, translator, settingRegistry, formRegistry) => {
        const registry = new EditorExtensionRegistry();
        // Register default extensions
        for (const extensionFactory of EditorExtensionRegistry.getDefaultExtensions({
            themes,
            translator
        })) {
            registry.addExtension(extensionFactory);
        }
        if (settingRegistry) {
            const updateSettings = (settings) => {
                var _a;
                registry.baseConfiguration =
                    (_a = settings.get('defaultConfig').composite) !== null && _a !== void 0 ? _a : {};
            };
            void Promise.all([
                settingRegistry.load(SETTINGS_ID),
                app.restored
            ]).then(([settings]) => {
                updateSettings(settings);
                settings.changed.connect(updateSettings);
            });
            formRegistry === null || formRegistry === void 0 ? void 0 : formRegistry.addRenderer(`${SETTINGS_ID}.defaultConfig`, {
                fieldRenderer: (props) => {
                    const properties = React.useMemo(() => registry.settingsSchema, []);
                    const editorConfiguration = props.formContext.settings.id ===
                        SETTINGS_ID
                        ? registry.baseConfiguration
                        : registry.defaultConfiguration;
                    const defaultFormData = {};
                    // Only provide customizable options
                    for (const [key, value] of Object.entries(editorConfiguration)) {
                        if (typeof properties[key] !== 'undefined') {
                            defaultFormData[key] = value;
                        }
                    }
                    return (React.createElement("div", { className: "jp-FormGroup-contentNormal" },
                        React.createElement("h3", { className: "jp-FormGroup-fieldLabel jp-FormGroup-contentItem" }, props.schema.title),
                        props.schema.description && (React.createElement("div", { className: "jp-FormGroup-description" }, props.schema.description)),
                        React.createElement(FormComponent, { schema: {
                                title: props.schema.title,
                                description: props.schema.description,
                                type: 'object',
                                properties,
                                additionalProperties: false
                            }, validator: validatorAjv8, formData: { ...defaultFormData, ...props.formData }, formContext: { defaultFormData }, liveValidate: true, onChange: e => {
                                var _a;
                                // Only save non-default values
                                const nonDefault = {};
                                for (const [property, value] of Object.entries((_a = e.formData) !== null && _a !== void 0 ? _a : {})) {
                                    const default_ = defaultFormData[property];
                                    if (default_ === undefined ||
                                        !JSONExt.deepEqual(value, default_)) {
                                        nonDefault[property] = value;
                                    }
                                }
                                props.onChange(nonDefault);
                            }, tagName: "div", translator: translator !== null && translator !== void 0 ? translator : nullTranslator })));
                }
            });
        }
        return registry;
    }
};
/**
 * CodeMirror real-time collaboration binding provider.
 */
export const bindingPlugin = {
    id: '@jupyterlab/codemirror-extension:binding',
    autoStart: true,
    requires: [IEditorExtensionRegistry],
    activate: (app, extensions) => {
        extensions.addExtension({
            name: 'yjs-binding',
            factory: options => {
                var _a;
                const sharedModel = options.model.sharedModel;
                return EditorExtensionRegistry.createImmutableExtension(ybinding({
                    ytext: sharedModel.ysource,
                    undoManager: (_a = sharedModel.undoManager) !== null && _a !== void 0 ? _a : undefined
                }));
            }
        });
    }
};
/**
 * The editor services.
 */
export const servicesPlugin = {
    id: '@jupyterlab/codemirror-extension:services',
    provides: IEditorServices,
    requires: [
        IEditorLanguageRegistry,
        IEditorExtensionRegistry,
        IEditorThemeRegistry
    ],
    optional: [ITranslator],
    activate: (app, languages, extensions, translator) => {
        const factory = new CodeMirrorEditorFactory({
            extensions,
            languages,
            translator: translator !== null && translator !== void 0 ? translator : nullTranslator
        });
        return {
            factoryService: factory,
            mimeTypeService: new CodeMirrorMimeTypeService(languages)
        };
    }
};
//# sourceMappingURL=services.js.map