// Copyright (c) Jupyter Development Team.
// Distributed under the terms of the Modified BSD License.
/**
 * @packageDocumentation
 * @module codemirror-extension
 */
import { ILabShell } from '@jupyterlab/application';
import { IPositionModel, LineCol } from '@jupyterlab/codeeditor';
import { IStatusBar } from '@jupyterlab/statusbar';
import { ITranslator } from '@jupyterlab/translation';
import { bindingPlugin, extensionPlugin, languagePlugin, servicesPlugin, themePlugin } from './services';
/**
 * A plugin providing a line/column status item to the application.
 */
export const lineColItem = {
    id: '@jupyterlab/codemirror-extension:line-col-status',
    autoStart: true,
    requires: [ITranslator],
    optional: [ILabShell, IStatusBar],
    provides: IPositionModel,
    activate: (app, translator, labShell, statusBar) => {
        const item = new LineCol(translator);
        const providers = new Set();
        if (statusBar) {
            // Add the status item to the status bar.
            statusBar.registerStatusItem(lineColItem.id, {
                item,
                align: 'right',
                rank: 2,
                isActive: () => !!item.model.editor
            });
        }
        const addEditorProvider = (provider) => {
            providers.add(provider);
            if (app.shell.currentWidget) {
                updateEditor(app.shell, {
                    newValue: app.shell.currentWidget,
                    oldValue: null
                });
            }
        };
        const update = () => {
            updateEditor(app.shell, {
                oldValue: app.shell.currentWidget,
                newValue: app.shell.currentWidget
            });
        };
        function updateEditor(shell, changes) {
            Promise.all([...providers].map(provider => provider(changes.newValue)))
                .then(editors => {
                var _a;
                item.model.editor =
                    (_a = editors.filter(editor => editor !== null)[0]) !== null && _a !== void 0 ? _a : null;
            })
                .catch(reason => {
                console.error('Get editors', reason);
            });
        }
        if (labShell) {
            labShell.currentChanged.connect(updateEditor);
        }
        return { addEditorProvider, update };
    }
};
/**
 * Export the plugins as default.
 */
const plugins = [
    languagePlugin,
    themePlugin,
    bindingPlugin,
    extensionPlugin,
    servicesPlugin,
    lineColItem
];
export default plugins;
//# sourceMappingURL=index.js.map