// Copyright (c) Jupyter Development Team.
// Distributed under the terms of the Modified BSD License.
import { nullTranslator } from '@jupyterlab/translation';
import { PromiseDelegate } from '@lumino/coreutils';
import { Platform } from '@lumino/domutils';
import { MessageLoop } from '@lumino/messaging';
import { Widget } from '@lumino/widgets';
import { ITerminal } from '.';
/**
 * The class name added to a terminal widget.
 */
const TERMINAL_CLASS = 'jp-Terminal';
/**
 * The class name added to a terminal body.
 */
const TERMINAL_BODY_CLASS = 'jp-Terminal-body';
/**
 * A widget which manages a terminal session.
 */
export class Terminal extends Widget {
    /**
     * Construct a new terminal widget.
     *
     * @param session - The terminal session object.
     *
     * @param options - The terminal configuration options.
     *
     * @param translator - The language translator.
     */
    constructor(session, options = {}, translator) {
        super();
        this._needsResize = true;
        this._offsetWidth = -1;
        this._offsetHeight = -1;
        this._isReady = false;
        this._ready = new PromiseDelegate();
        this._termOpened = false;
        translator = translator || nullTranslator;
        this._trans = translator.load('jupyterlab');
        this.session = session;
        // Initialize settings.
        this._options = { ...ITerminal.defaultOptions, ...options };
        const { theme, ...other } = this._options;
        const xtermOptions = {
            theme: Private.getXTermTheme(theme),
            ...other
        };
        this.addClass(TERMINAL_CLASS);
        this._setThemeAttribute(theme);
        // Buffer session message while waiting for the terminal
        let buffer = '';
        const bufferMessage = (sender, msg) => {
            switch (msg.type) {
                case 'stdout':
                    if (msg.content) {
                        buffer += msg.content[0];
                    }
                    break;
                default:
                    break;
            }
        };
        session.messageReceived.connect(bufferMessage);
        session.disposed.connect(() => {
            if (this.getOption('closeOnExit')) {
                this.dispose();
            }
        }, this);
        // Create the xterm.
        Private.createTerminal(xtermOptions)
            .then(([term, fitAddon]) => {
            this._term = term;
            this._fitAddon = fitAddon;
            this._initializeTerm();
            this.id = `jp-Terminal-${Private.id++}`;
            this.title.label = this._trans.__('Terminal');
            this._isReady = true;
            this._ready.resolve();
            if (buffer) {
                this._term.write(buffer);
            }
            session.messageReceived.disconnect(bufferMessage);
            session.messageReceived.connect(this._onMessage, this);
            if (session.connectionStatus === 'connected') {
                this._initialConnection();
            }
            else {
                session.connectionStatusChanged.connect(this._initialConnection, this);
            }
            this.update();
        })
            .catch(reason => {
            console.error('Failed to create a terminal.\n', reason);
            this._ready.reject(reason);
        });
    }
    /**
     * A promise that is fulfilled when the terminal is ready.
     */
    get ready() {
        return this._ready.promise;
    }
    /**
     * Get a config option for the terminal.
     */
    getOption(option) {
        return this._options[option];
    }
    /**
     * Set a config option for the terminal.
     */
    setOption(option, value) {
        if (option !== 'theme' &&
            (this._options[option] === value || option === 'initialCommand')) {
            return;
        }
        this._options[option] = value;
        switch (option) {
            case 'fontFamily':
                this._term.options.fontFamily = value;
                break;
            case 'fontSize':
                this._term.options.fontSize = value;
                break;
            case 'lineHeight':
                this._term.options.lineHeight = value;
                break;
            case 'screenReaderMode':
                this._term.options.screenReaderMode = value;
                break;
            case 'scrollback':
                this._term.options.scrollback = value;
                break;
            case 'theme':
                this._term.options.theme = {
                    ...Private.getXTermTheme(value)
                };
                this._setThemeAttribute(value);
                break;
            case 'macOptionIsMeta':
                this._term.options.macOptionIsMeta = value;
                break;
            default:
                // Do not transmit options not listed above to XTerm
                break;
        }
        this._needsResize = true;
        this.update();
    }
    /**
     * Dispose of the resources held by the terminal widget.
     */
    dispose() {
        if (!this.session.isDisposed) {
            if (this.getOption('shutdownOnClose')) {
                this.session.shutdown().catch(reason => {
                    console.error(`Terminal not shut down: ${reason}`);
                });
            }
        }
        void this.ready.then(() => {
            this._term.dispose();
        });
        super.dispose();
    }
    /**
     * Refresh the terminal session.
     *
     * #### Notes
     * Failure to reconnect to the session should be caught appropriately
     */
    async refresh() {
        if (!this.isDisposed && this._isReady) {
            await this.session.reconnect();
            this._term.clear();
        }
    }
    /**
     * Check if terminal has any text selected.
     */
    hasSelection() {
        if (!this.isDisposed && this._isReady) {
            return this._term.hasSelection();
        }
        return false;
    }
    /**
     * Paste text into terminal.
     */
    paste(data) {
        if (!this.isDisposed && this._isReady) {
            return this._term.paste(data);
        }
    }
    /**
     * Get selected text from terminal.
     */
    getSelection() {
        if (!this.isDisposed && this._isReady) {
            return this._term.getSelection();
        }
        return null;
    }
    /**
     * Process a message sent to the widget.
     *
     * @param msg - The message sent to the widget.
     *
     * #### Notes
     * Subclasses may reimplement this method as needed.
     */
    processMessage(msg) {
        super.processMessage(msg);
        switch (msg.type) {
            case 'fit-request':
                this.onFitRequest(msg);
                break;
            default:
                break;
        }
    }
    /**
     * Set the size of the terminal when attached if dirty.
     */
    onAfterAttach(msg) {
        this.update();
    }
    /**
     * Set the size of the terminal when shown if dirty.
     */
    onAfterShow(msg) {
        this.update();
    }
    /**
     * On resize, use the computed row and column sizes to resize the terminal.
     */
    onResize(msg) {
        this._offsetWidth = msg.width;
        this._offsetHeight = msg.height;
        this._needsResize = true;
        this.update();
    }
    /**
     * A message handler invoked on an `'update-request'` message.
     */
    onUpdateRequest(msg) {
        var _a;
        if (!this.isVisible || !this.isAttached || !this._isReady) {
            return;
        }
        // Open the terminal if necessary.
        if (!this._termOpened) {
            this._term.open(this.node);
            (_a = this._term.element) === null || _a === void 0 ? void 0 : _a.classList.add(TERMINAL_BODY_CLASS);
            this._termOpened = true;
        }
        if (this._needsResize) {
            this._resizeTerminal();
        }
    }
    /**
     * A message handler invoked on an `'fit-request'` message.
     */
    onFitRequest(msg) {
        const resize = Widget.ResizeMessage.UnknownSize;
        MessageLoop.sendMessage(this, resize);
    }
    /**
     * Handle `'activate-request'` messages.
     */
    onActivateRequest(msg) {
        var _a;
        (_a = this._term) === null || _a === void 0 ? void 0 : _a.focus();
    }
    _initialConnection() {
        if (this.isDisposed) {
            return;
        }
        if (this.session.connectionStatus !== 'connected') {
            return;
        }
        this.title.label = this._trans.__('Terminal %1', this.session.name);
        this._setSessionSize();
        if (this._options.initialCommand) {
            this.session.send({
                type: 'stdin',
                content: [this._options.initialCommand + '\r']
            });
        }
        // Only run this initial connection logic once.
        this.session.connectionStatusChanged.disconnect(this._initialConnection, this);
    }
    /**
     * Initialize the terminal object.
     */
    _initializeTerm() {
        const term = this._term;
        term.onData((data) => {
            if (this.isDisposed) {
                return;
            }
            this.session.send({
                type: 'stdin',
                content: [data]
            });
        });
        term.onTitleChange((title) => {
            this.title.label = title;
        });
        // Do not add any Ctrl+C/Ctrl+V handling on macOS,
        // where Cmd+C/Cmd+V works as intended.
        if (Platform.IS_MAC) {
            return;
        }
        term.attachCustomKeyEventHandler(event => {
            if (event.ctrlKey && event.key === 'c' && term.hasSelection()) {
                // Return so that the usual OS copy happens
                // instead of interrupt signal.
                return false;
            }
            if (event.ctrlKey && event.key === 'v' && this._options.pasteWithCtrlV) {
                // Return so that the usual paste happens.
                return false;
            }
            return true;
        });
    }
    /**
     * Handle a message from the terminal session.
     */
    _onMessage(sender, msg) {
        switch (msg.type) {
            case 'stdout':
                if (msg.content) {
                    this._term.write(msg.content[0]);
                }
                break;
            case 'disconnect':
                this._term.write('\r\n\r\n[Finished… Term Session]\r\n');
                break;
            default:
                break;
        }
    }
    /**
     * Resize the terminal based on computed geometry.
     */
    _resizeTerminal() {
        if (this._options.autoFit) {
            this._fitAddon.fit();
        }
        if (this._offsetWidth === -1) {
            this._offsetWidth = this.node.offsetWidth;
        }
        if (this._offsetHeight === -1) {
            this._offsetHeight = this.node.offsetHeight;
        }
        this._setSessionSize();
        this._needsResize = false;
    }
    /**
     * Set the size of the terminal in the session.
     */
    _setSessionSize() {
        const content = [
            this._term.rows,
            this._term.cols,
            this._offsetHeight,
            this._offsetWidth
        ];
        if (!this.isDisposed) {
            this.session.send({ type: 'set_size', content });
        }
    }
    _setThemeAttribute(theme) {
        if (this.isDisposed) {
            return;
        }
        this.node.setAttribute('data-term-theme', theme ? theme.toLowerCase() : 'inherit');
    }
}
/**
 * A namespace for private data.
 */
var Private;
(function (Private) {
    /**
     * An incrementing counter for ids.
     */
    Private.id = 0;
    /**
     * The light terminal theme.
     */
    Private.lightTheme = {
        foreground: '#000',
        background: '#fff',
        cursor: '#616161',
        cursorAccent: '#F5F5F5',
        selectionBackground: 'rgba(97, 97, 97, 0.3)',
        selectionInactiveBackground: 'rgba(189, 189, 189, 0.3)' // md-grey-400
    };
    /**
     * The dark terminal theme.
     */
    Private.darkTheme = {
        foreground: '#fff',
        background: '#000',
        cursor: '#fff',
        cursorAccent: '#000',
        selectionBackground: 'rgba(255, 255, 255, 0.3)',
        selectionInactiveBackground: 'rgba(238, 238, 238, 0.3)' // md-grey-200
    };
    /**
     * The current theme.
     */
    Private.inheritTheme = () => ({
        foreground: getComputedStyle(document.body)
            .getPropertyValue('--jp-ui-font-color0')
            .trim(),
        background: getComputedStyle(document.body)
            .getPropertyValue('--jp-layout-color0')
            .trim(),
        cursor: getComputedStyle(document.body)
            .getPropertyValue('--jp-ui-font-color1')
            .trim(),
        cursorAccent: getComputedStyle(document.body)
            .getPropertyValue('--jp-ui-inverse-font-color0')
            .trim(),
        selectionBackground: getComputedStyle(document.body)
            .getPropertyValue('--jp-layout-color3')
            .trim(),
        selectionInactiveBackground: getComputedStyle(document.body)
            .getPropertyValue('--jp-layout-color2')
            .trim()
    });
    function getXTermTheme(theme) {
        switch (theme) {
            case 'light':
                return Private.lightTheme;
            case 'dark':
                return Private.darkTheme;
            case 'inherit':
            default:
                return Private.inheritTheme();
        }
    }
    Private.getXTermTheme = getXTermTheme;
})(Private || (Private = {}));
/**
 * Utility functions for creating a Terminal widget
 */
(function (Private) {
    let supportWebGL = false;
    let Xterm_;
    let FitAddon_;
    let WeblinksAddon_;
    let Renderer_;
    /**
     * Detect if the browser supports WebGL or not.
     *
     * Reference: https://developer.mozilla.org/en-US/docs/Web/API/WebGL_API/By_example/Detect_WebGL
     */
    function hasWebGLContext() {
        // Create canvas element. The canvas is not added to the
        // document itself, so it is never displayed in the
        // browser window.
        const canvas = document.createElement('canvas');
        // Get WebGLRenderingContext from canvas element.
        const gl = canvas.getContext('webgl') || canvas.getContext('experimental-webgl');
        // Report the result.
        try {
            return gl instanceof WebGLRenderingContext;
        }
        catch (error) {
            return false;
        }
    }
    function addRenderer(term) {
        let renderer = new Renderer_();
        term.loadAddon(renderer);
        if (supportWebGL) {
            renderer.onContextLoss(event => {
                console.debug('WebGL context lost - reinitialize Xtermjs renderer.');
                renderer.dispose();
                // If the Webgl context is lost, reinitialize the addon
                addRenderer(term);
            });
        }
    }
    /**
     * Create a xterm.js terminal asynchronously.
     */
    async function createTerminal(options) {
        var _a;
        if (!Xterm_) {
            supportWebGL = hasWebGLContext();
            const [xterm_, fitAddon_, renderer_, weblinksAddon_] = await Promise.all([
                import('xterm'),
                import('xterm-addon-fit'),
                supportWebGL
                    ? import('xterm-addon-webgl')
                    : import('xterm-addon-canvas'),
                import('xterm-addon-web-links')
            ]);
            Xterm_ = xterm_.Terminal;
            FitAddon_ = fitAddon_.FitAddon;
            Renderer_ =
                (_a = renderer_.WebglAddon) !== null && _a !== void 0 ? _a : renderer_.CanvasAddon;
            WeblinksAddon_ = weblinksAddon_.WebLinksAddon;
        }
        const term = new Xterm_(options);
        addRenderer(term);
        const fitAddon = new FitAddon_();
        term.loadAddon(fitAddon);
        term.loadAddon(new WeblinksAddon_());
        return [term, fitAddon];
    }
    Private.createTerminal = createTerminal;
})(Private || (Private = {}));
//# sourceMappingURL=widget.js.map