"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.combineBaseUrl = combineBaseUrl;
exports.getPublicUrl = getPublicUrl;
exports.getWebProtocol = getWebProtocol;
exports.isHost = isHost;
exports.isURLhasValidProtocol = isURLhasValidProtocol;
exports.validateURL = validateURL;
exports.wrapPrefix = wrapPrefix;
var _debug = _interopRequireDefault(require("debug"));
var _url = require("url");
var _isURL = _interopRequireDefault(require("validator/lib/isURL"));
var _core = require("@verdaccio/core");
function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }
const debug = (0, _debug.default)('verdaccio:core:url');
const validProtocols = ['https', 'http'];

/**
 * Check if URI is starting with "http://", "https://" or "//"
 * @param {string} uri
 */
function isURLhasValidProtocol(uri) {
  return /^(https?:)?\/\//.test(uri);
}
function isHost(url = '', options = {}) {
  return (0, _isURL.default)(url, {
    require_host: true,
    allow_trailing_dot: false,
    require_valid_protocol: false,
    // @ts-ignore
    require_port: false,
    require_tld: false,
    ...options
  });
}

/**
 * Detect running protocol (http or https)
 */
function getWebProtocol(headerProtocol, protocol) {
  let returnProtocol;
  const [, defaultProtocol] = validProtocols;
  // HAProxy variant might return http,http with X-Forwarded-Proto
  if (typeof headerProtocol === 'string' && headerProtocol !== '') {
    debug('header protocol: %o', protocol);
    const commaIndex = headerProtocol.indexOf(',');
    returnProtocol = commaIndex > 0 ? headerProtocol.slice(0, commaIndex) : headerProtocol;
  } else {
    debug('req protocol: %o', headerProtocol);
    returnProtocol = protocol;
  }
  return validProtocols.includes(returnProtocol) ? returnProtocol : defaultProtocol;
}
function wrapPrefix(prefix) {
  if (prefix === '' || typeof prefix === 'undefined' || prefix === null) {
    return '';
  } else if (!prefix.startsWith('/') && prefix.endsWith('/')) {
    return `/${prefix}`;
  } else if (!prefix.startsWith('/') && !prefix.endsWith('/')) {
    return `/${prefix}/`;
  } else if (prefix.startsWith('/') && !prefix.endsWith('/')) {
    return `${prefix}/`;
  } else {
    return prefix;
  }
}

/**
 * Create base url for registry.
 * @return {String} base registry url
 */
function combineBaseUrl(protocol, host, prefix = '') {
  debug('combined protocol %o', protocol);
  debug('combined host %o', host);
  const newPrefix = wrapPrefix(prefix);
  debug('combined prefix %o', newPrefix);
  const groupedURI = new _url.URL(wrapPrefix(prefix), `${protocol}://${host}`);
  const result = groupedURI.href;
  debug('combined url %o', result);
  return result;
}
function validateURL(publicUrl) {
  try {
    const parsed = new _url.URL(publicUrl);
    if (!validProtocols.includes(parsed.protocol.replace(':', ''))) {
      throw Error('invalid protocol');
    }
    return true;
  } catch (err) {
    // TODO: add error logger here
    return false;
  }
}
function getPublicUrl(url_prefix = '', requestOptions) {
  if (validateURL(process.env.VERDACCIO_PUBLIC_URL)) {
    const envURL = new _url.URL(wrapPrefix(url_prefix), process.env.VERDACCIO_PUBLIC_URL).href;
    debug('public url by env %o', envURL);
    return envURL;
  } else if (requestOptions.headers['host']) {
    var _process$env$VERDACCI, _process$env$VERDACCI2;
    const host = requestOptions.headers['host'];
    if (!isHost(host)) {
      throw new Error('invalid host');
    }
    const protoHeader = (_process$env$VERDACCI = (_process$env$VERDACCI2 = process.env.VERDACCIO_FORWARDED_PROTO) === null || _process$env$VERDACCI2 === void 0 ? void 0 : _process$env$VERDACCI2.toLocaleLowerCase()) !== null && _process$env$VERDACCI !== void 0 ? _process$env$VERDACCI : _core.HEADERS.FORWARDED_PROTO.toLowerCase();
    const protocol = getWebProtocol(requestOptions.headers[protoHeader], requestOptions.protocol);
    const combinedUrl = combineBaseUrl(protocol, host, url_prefix);
    debug('public url by request %o', combinedUrl);
    return combinedUrl;
  } else {
    return '/';
  }
}
//# sourceMappingURL=index.js.map